/* @(#)fitsmdb.c	19.1.1.2 (ESO-DMD) 04/03/03 17:08:02 */
/*===========================================================================
  Copyright (C) 1995 European Southern Observatory (ESO)
 
  This program is free software; you can redistribute it and/or 
  modify it under the terms of the GNU General Public License as 
  published by the Free Software Foundation; either version 2 of 
  the License, or (at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public 
  License along with this program; if not, write to the Free 
  Software Foundation, Inc., 675 Massachusetts Ave, Cambridge, 
  MA 02139, USA.
 
  Correspondence concerning ESO-MIDAS should be addressed as follows:
	Internet e-mail: midas@eso.org
	Postal address: European Southern Observatory
			Data Management Division 
			Karl-Schwarzschild-Strasse 2
			D 85748 Garching bei Muenchen 
			GERMANY
===========================================================================*/

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
.COPYRIGHT  (c)  1996  European Southern Observatory
.IDENT      fitsmdb.c
.LAUGUAGE   C
.AUTHOR     P.Grosbol   ESO/IPG
.KEYWORDS   MIDAS descriptor buffer
.COMMENT    Define buffer for MIDAS descriptors
.VERSION    1.0  1991-Mar-19 : Creation,   PJG
.VERSION    1.1  1993-Feb-23 : Return pointer to structure,   PJG
.VERSION    1.2  1993-Jul-05 : Save keyword commends,   PJG
.VERSION    1.3  1993-Sep-27 : strlen() used for SCDWRC() instead of 72. CG.
.VERSION    1.4  1993-Oct-26 : Update to new SC + prototypes, PJG
.VERSION    1.5  1993-Dec-13 : Disable SC-error handling for descriptors, PJG
.VERSION    1.6  1994-Jan-11 : Check zero length desc. history, PJG
.VERSION    1.7  1996-Oct-22 : Correct length of descriptor names, PJG

 030320 	last modif

---------------------------------------------------------------------*/

#include   <stdlib.h>
#include   <fitsfmt.h>
#include   <fitsdef.h>
#include   <fileexts.h>

static     int first = 0;
static     int   nmdb;		/* no. of entries in keyword buffer */
static     int   max_nmdb;	/* current limit of mdb */
static     MDBUF  *mdbptr;	/* pointer to MIDAS descriptor buffer */
static     MDBUF  *myptr;	/* working pointer */
/*

*/

MDBUF *mdb_init()
/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
.PURPOSE    initiate the MIDAS descriptor buffer
.RETURN     pointer to internal MIDAS descriptor buffer 
---------------------------------------------------------------------*/

{
int  i;


nmdb = 0;
if (KEYALL.ORIGIN == -1)
   max_nmdb = 1024;			/* outside Midas */
else if (FCT.PARM[0] > 500)		/* we count on many descriptors... */
   max_nmdb = 1024;			
else
   max_nmdb = 512;

max_nmdb = 30;

i = (max_nmdb + 1) * (sizeof(MDBUF));

if (first == 0)
   {					/* keep to single malloc in prepa.exe */
   first = 1;
   mdbptr = (MDBUF *) malloc((size_t) (i));
   if (mdbptr == (MDBUF *) NULL)
      {
      (void)printf
      ("mdb_init: could not allocate %d entries for MDBUF \n\r",max_nmdb);
      ospexit(0);
      }
   }

return mdbptr;
}

/*

*/

int mdb_size()
/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
.PURPOSE    return current size of MIDAS descriptor buffer
.RETURN     the no. of entries
---------------------------------------------------------------------*/

{

return nmdb;		/* return next (free) index - since 1st index = 0 */
			/* this is the no. of used entries */
}


MDBUF *mdb_info(nsize)
/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
.PURPOSE    provide current size of MIDAS descriptor buffer
.RETURN     pointer to internal MIDAS descriptor buffer 
	    this pointer may be different from the one returned before
	    by mdb_init(), if the internal buffer size has been increased!!
---------------------------------------------------------------------*/
int  *nsize;

{
*nsize = nmdb;		/* the next (free) index - since 1st index = 0
			   so that's the no. of entries in mdbuf ...  */
return mdbptr;
}

/*

*/

mdb_put(kw,kwd)
/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
.PURPOSE    put keyword values into MIDAS descriptor buffer
.RETURN     Status 0:OK, 1:error, keyword not stored
---------------------------------------------------------------------*/
KWORD   *kw;
KWDEF   *kwd;

{
int  i;

char *pc, *ps;



/* if needed, we allocate more space */

if (nmdb >= max_nmdb) 
   {
   int  nsize;
   void  *doof;

   nsize = (2*max_nmdb + 1) * (sizeof(MDBUF));
   doof = realloc((void *)mdbptr,(size_t)nsize);
   if (doof == (void *)NULL)
      {
      (void)printf
      ("mdb_put: could not reallocate %d bytes for MDBUF \n\r",nsize);
      return 1;
      }
   mdbptr = doof;
   max_nmdb *= 2;
   }


/* save keyword in buffer */

myptr = mdbptr + nmdb;			/* point to free entry */

myptr->pcom = (char *) 0;		/* init what's not always updated */
myptr->buf[0] = '\0';

ps = myptr->desc;
pc = kwd->desc;
for (i=0; i<MXMDN; i++) *ps++ = *pc++;

myptr->type = kwd->type;
myptr->idx = kwd->idx;

ps = myptr->buf;
switch (kwd->type) 
   {
   case 'S' : myptr->val.pc = myptr->buf;
              pc = kw->val.pc;
              while (*ps++ = *pc++);
              break;
   case 'L' :
   case 'I' : myptr->val.i = kw->val.i; 
              break;
   case 'R' :
   case 'D' : myptr->val.d[0] = kw->val.d[0];
              myptr->val.d[1] = kw->val.d[1];
              break;
   default  : return 1;
   }

if (kw->pcom)
   {                       /* copy keyword comment    */
   pc = kw->pcom; myptr->pcom = ps;
   while (*ps++ = *pc++);
   }

nmdb++;			/* increase counter */
return 0;
}

/*

*/

mdb_get(mfd)
/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
.PURPOSE    get values from MIDAS descriptor buffer and store them
            in file
.RETURN     status  SC error code
---------------------------------------------------------------------*/
int    mfd;               /* IN: MIDAS file descriptor               */

{
int   unit[4];
int   i, n, err, ival;
int   ec, ed, el, ln0, ln1;

float fval;

char  *dscptr, *cptr;



(void)SCECNT("GET",&ec,&el,&ed);		/* disable SC-error handling */
ln0 = 0;  ln1 = 1;
(void)SCECNT("PUT",&ln1,&ln0,&ln0);

err = 0;
myptr = mdbptr;

for (i=0; i<nmdb; i++) 
   {						/* read MD-buffer      */
   dscptr = myptr->desc;

   switch (myptr->type)
      {
      case 'S':
         cptr = myptr->val.pc;
         err = SCDWRC(mfd,dscptr,1,cptr,myptr->idx,strlen(cptr),unit);
         break;
      case 'L':
         ival = myptr->val.i;
         err = SCDWRL(mfd,dscptr,&ival,myptr->idx,1,unit);
         break;
      case 'I':
         ival = myptr->val.i;
         err = SCDWRI(mfd,dscptr,&ival,myptr->idx,1,unit);
         break;
      case 'R': 
         fval = myptr->val.d[0];
         err = SCDWRR(mfd,dscptr,&fval,myptr->idx,1,unit);
         break;
      case 'D':
         err = SCDWRD(mfd,dscptr,myptr->val.d,myptr->idx,1,unit);
         break;
      }

   if (myptr->pcom && (0 < myptr->idx))
      {
      n = (int) strlen(myptr->pcom);
      if (0<n) SCDWRH(mfd,dscptr,myptr->pcom,-1,n);
      }

   myptr ++;
   }
 
nmdb = 0;					/* reset MDBUF */

(void)SCECNT("PUT",&ec,&el,&ed);		/* reset SC-error variables */
return err;
}



