/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
! COPYRIGHT    (c)  2002 AIP, Potsdam, Germany
! IDENT        Euro3D.c
! LANGUAGE     C
! AUTHOR       S.F.Sanchez
! KEYWORDS
! PURPOSE      Library for the Visualization Tool
! COMMENT      Teel process
! VERSION      0.1  2003-Jan-29 : Creation, SF. Sanchez
------------------------------------------------------------------------------*/

#include <stdio.h> //For Standard I/O
#include <stdlib.h> //For Functions like ATOI
#include <ctype.h> //For Type check
#include <string.h> //For String manipulation
#include <math.h> //For Math functions
#include <IFU_io.h> //For Lyon I/O Library
#include <data_io.h> //For Lyon I/O data
#include <tk.h>
#include <time.h>

/******************************
 * For Shared Memory
 ******************************/

#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/shm.h>



/******************************
 *To use PGPLOT
 *******************************/

#include <cpgplot.h> // PGPLOT
#include "tkpgplot.h"

/************************************
 * NCARG: Never used anymore
 ************************************/
//#include <ncarg/ngmath.h>
//#include <ncarg/ncargC.h>
//#include <ncarg/gks.h>

/*******************************
 * MY GRID: Interpolation routines
 *******************************/
#include "csa.h"
#include "nn.h"
#include "nan.h"
#include "my_grid.h"



#include <Euro3D.h>
#include <my_convolve.h>
#include <my_math.h>
#include <fft2d.h>
#include <gl_lsq.h>

/*********************************************
 * Different procedures to read
 * the basic information on the SHM
 *********************************************/

/*
 * Limitation in shared memory due to the computer
 *
 */
#define MAX_SHM 33554432



static void non_grid_int (PLFLT *x, PLFLT *y, PLFLT *z, int npts,
	      PLFLT *xg, PLFLT *yg,  PLFLT *zg, int nptsg, int type);

/* 
 * Bivariate Cubic Spline Approximation using Pavel Sakov's csa package
 *
 * NaNs are returned where no interpolation can be done.
 *
 * Using a non_grided output
 *
 */

static void
non_grid_int (PLFLT *x, PLFLT *y, PLFLT *z, int npts,
	  PLFLT *xg, PLFLT *yg,  PLFLT *zg, int nptsg, int type)
{
  PLFLT *xt, *yt, *zt;
  point *pin, *pgrid, *pt;
  csa* a = NULL;
  int i, j;

  /*
   * Limits of the interpolation
   */
  float x_min,x_max,y_min,y_max;
  x_min=10000;
  y_min=10000;
  x_max=-10000;
  y_max=-10000;



  pin = (point *) malloc(npts * sizeof(point));

  xt = x; yt = y; zt = z; pt = pin;
  for(i=0; i<npts; i++) {
    pt->x = (double) *xt++;
    pt->y = (double) *yt++;
    pt->z = (double) *zt++;
    if (xt[i]<x_min) x_min=xt[i];
    if (yt[i]<y_min) y_min=yt[i];
    if (xt[i]>x_max) x_max=xt[i];
    if (yt[i]>y_max) y_max=yt[i];
    pt++;
  }

//  nptsg = nptsx * nptsy;
  pgrid =  (point *) malloc(nptsg * sizeof(point));

  xt = xg; yt = yg; pt = pgrid;
  for(i=0; i<npts; i++) {
    pt->x = (double) *xt++;
    pt->y = (double) *yt++;
    pt++;
  }


//  printf("npts=%d\n",npts);
  switch (type) {    
      case (GRID_CSA):  
	  a = csa_create();
	  csa_addpoints(a, npts, pin);
	  csa_calculatespline(a);
//	  printf("Paso\n");
//	  fflush(stdout);
	  csa_approximate_points(a, nptsg, pgrid);
	  csa_destroy(a);
	  break;
      case (GRID_DTLI):	  
	  lpi_interpolate_points(npts, pin, nptsg, pgrid);
	  break;
      case (GRID_NNI):	  
	  nnpi_interpolate_points(npts, pin, 1e-16, nptsg, pgrid);
	  break;
      default:
	  lpi_interpolate_points(npts, pin, nptsg, pgrid);
	  break;
  }

  for(i=0; i<nptsg; i++) {
      pt = &pgrid[i];      
      zg[i] = (PLFLT) pt->z;
      if (xg[i]<x_min) zg[i]=0.0;
      if (yg[i]<y_min) zg[i]=0.0;
      if (xg[i]>x_max) zg[i]=0.0;
      if (yg[i]>y_max) zg[i]=0.0;
      if (zg[i]!=zg[i]) zg[i]=0.0;
  }


  free(pin);
  free(pgrid); 
}


/*
 * Correct one monocromatic image for DAR
 */

/*
 * Create an interpolated Map
 * saving its WCS
 */
int correct_mono_dar(int nb_spec, int *n_spaxels, SPAXEL **spaxels, float *slice, struct DAR *dar_t, int i_slice) {
  int i,j,k,l,m;
  float *temp_data;
  int ier;
  float *x,*y,*z;
  int n_tot;
  float *xo,*yo,rpnt;

  float *xg,*yg,*zg;
  float x_min,x_max,y_min,y_max;
  float dx,dy;

  float *slice_tmp;
  

  slice_tmp= (float *)malloc((nb_spec)*sizeof(float));
  /*
   * We count the number of points included in the
   * SPAXELS array
   *
   */
  for (k=0;k<nb_spec;k++) {
    slice_tmp[k]=slice[k];
  }


  n_tot=0;
  for (k=0;k<nb_spec;k++) {
    for (l=0;l<n_spaxels[k];l++) {
      n_tot++;
    }
  }
  x = (float *)malloc(n_tot*sizeof(float));    
  y = (float *)malloc(n_tot*sizeof(float));    
  xo = (float *)malloc(n_tot*sizeof(float));    
  yo = (float *)malloc(n_tot*sizeof(float));    
  z = (float *)malloc(n_tot*sizeof(float));    
  /*
   * We feed with data
   */
  n_tot=0;
  for (k=0;k<nb_spec;k++) {
    for (l=0;l<n_spaxels[k];l++) {
      x[n_tot]=spaxels[k][l].xpos-dar_t[spaxels[k][l].group-1].dx[i_slice];
      y[n_tot]=spaxels[k][l].ypos-dar_t[spaxels[k][l].group-1].dy[i_slice];
      xo[n_tot]=spaxels[k][l].xpos;
      yo[n_tot]=spaxels[k][l].ypos;
      z[n_tot]=slice_tmp[k];
      n_tot++;
    }
  }
  zg = (float *)malloc(n_tot*sizeof(float));    
//  zg = (PLFLT *) calloc(n_tot ,sizeof(PLFLT));
/*
 * WE NEED TO INTERPOLATE!!!!
 */
//  c_nnpntinits(n_tot, x, y, z);
  

//  printf("i_slice %d ",i_slice);

//  printf("n_tot=%d\n",n_tot);
  non_grid_int(x,y,z,n_tot,xo,yo,zg,n_tot,2);
//  printf("Done\n");
  n_tot=0;
  for (k=0;k<nb_spec;k++) {
    for (l=0;l<n_spaxels[k];l++) {
	slice[k]=zg[n_tot];
	n_tot++;
    }
  }
/*
  m=0;
  for (k=0;k<nb_spec;k++) {
      for (l=0;l<n_spaxels[k];l++) {
	  for (i=0;i<3;i++) {
	      xg[i]=xo[m]+0.2*i-0.2;
	      yg[i]=yo[m]+0.2*i-0.2;
	  }
	  m++;
	  my_griddata(x, y, z, n_tot, xg, 3, yg, 3, zg, GRID_CSA, 0.0);
	  slice[k] = zg[1][1];	
      }
  }
*/


  free(slice_tmp);
  free(z);
  free(zg);
  free(x);
  free(y);
  free(xo);
  free(yo);

  return(0);
}


/*
 * Load the DAR correction
 */
int load_dar(char* filename, struct DAR *dar, int npix, int n_groups) {
    FILE *fichero;
    int i,j,tmp_i;
    float tmp_float;
    
    for (j=0;j<n_groups;j++) {
	dar[j].n=npix;
	dar[j].dx= (float *)calloc(npix,sizeof(float));  
	dar[j].dy= (float *)calloc(npix,sizeof(float));  
	for (i=0;i<npix;i++) {
	    dar[j].dx[i]=0.0;
	    dar[j].dy[i]=0.0;
	}
    }
    
    if ((fichero=fopen(filename,"r"))==NULL) {
	printf("Error opening file: %s\n",filename);
	return(-1);
    } else {
	
	for (j=0;j<n_groups;j++) {
	    for (i=0;i<npix;i++) {
		fscanf(fichero, "%d", &tmp_i);
		fscanf(fichero, "%f", &tmp_float);
		dar[j].dx[i]=tmp_float;
		fscanf(fichero, "%f", &tmp_float);
		dar[j].dy[i]=tmp_float;
	    }      
	}
    }
    fclose(fichero);
    return(0);
}

/*
 * Save the DAR correction
 */
int save_dar(char* filename, struct DAR *dar, int npix, int n_groups) {
  FILE *fichero;
  int i,j;
  if ((fichero=fopen(filename,"w"))==NULL) {
    printf("Error opening file: %s\n",filename);
  } else {
      for (j=0;j<n_groups;j++) {
	  for (i=0;i<npix;i++) {
	      fprintf(fichero,"%d %g %g\n",i+1,dar[j].dx[i],dar[j].dy[i]);
	  }
      }
  }
  fclose(fichero);
  return(0);
}




/*
 * Determine the DAR of a certain GROUP using Theoretical Values
 */
int determine_dar(struct DAR dar, GROUP group, float start, float step, int npix) {
  int i,j;
  float l;
  float dr;
  float z;
  float dx0,dy0;
  z=acos(1/group.airmass);
  for (i=0;i<npix;i++) {
    l=start+step*i;    
    if ((group.pressure==group.pressure)&&(group.temperature==group.temperature)&&(group.rel_humidity==group.rel_humidity)) {
	dar_l(&dr,l,group.pressure,group.temperature,group.rel_humidity,z);
	dar.dx[i]=dr*cos(group.parang*3.141592654/180.);
	dar.dy[i]=dr*sin(group.parang*3.141592654/180.);
    } else {
	dar.dx[i]=0.0;
	dar.dy[i]=0.0;
    }

  }
  dx0=dar.dx[(int)(npix/2)];
  dy0=dar.dy[(int)(npix/2)];
  for (i=0;i<npix;i++) {
    dar.dx[i]=(-1)*(dar.dx[i]-dx0);
    dar.dy[i]=(-1)*(dar.dy[i]-dy0);

  }
  return(0);
}


/*
 * Differential Atmospheric Refraction calculator
 */
int dar_l(float *dr_out, float l, float p, float t, float f, float z) {
  float l0=5000;
  float a,b,c;
  float a0,b0,c0;
  float dr;
  float out;
  float il,il0;

  f=p*f;
  il=pow(1/(l/10000.0),2);
  il0=pow(1/l0,2);

  a=64.328+29498.1/(146-il)+255.4/(41.0-il);
  b=pow(10.0,-6)*a*((p*(1+(1.049-0.0157*t)*pow(10.0,-6)*p))/(720.883*(1+0.003661*t)));
  c=b/(((0.0624-0.000680*il)/(1+0.003661*t))*f);

  a0=64.328+29498.1/(146-il0)+255.4/(41.0-il0);
  b0=pow(10.0,-6)*a0*((p*(1+(1.049-0.0157*t)*pow(10.0,-6)*p))/(720.883*(1+0.003661*t)));
  c0=b0/(((0.0624-0.000680*il0)/(1+0.003661*t))*f);
  
  dr=c-c0;
  out=206265.0*dr*tan(z);
  *dr_out=out;
  return(0);
}




/*
 * Save Slice
 */ 

int save_slice(float *spaxId, float *x, float *y, float* slice_data, int n, char* filename, char* units) {
  TABLE table;
  int colID_x,colID_y,colID_spaxId,colID_data;
  int i,j;
  create_table(&table,filename,n,(int)4,'I',"Slice Table");
  colID_spaxId=create_col(&table,"spaxId",INT,'N',"E12.6","N/A");
  colID_x=create_col(&table,"x",FLOAT,'N',"E12.6","arcsec");
  colID_y=create_col(&table,"y",FLOAT,'N',"E12.6","arcsec");
  colID_data=create_col(&table,"DATA",FLOAT,'N',"E12.6",units);
  for (i=0;i<n;i++) {
    WR_tbl(&table,i,colID_spaxId,&spaxId[i]);
    WR_tbl(&table,i,colID_x,&x[i]);
    WR_tbl(&table,i,colID_y,&y[i]);
    WR_tbl(&table,i,colID_data,&slice_data[i]);
  }
  close_table(&table);
}


/*
 * Save Spectra
 */ 

int save_spec_table(float *w, float* spec_data, int n, char* filename, char* units) {
  TABLE table;
  int colID_w,colID_data;
  int i,j;
  create_table(&table,filename,n,(int)4,'I',"Spectra Table");
  colID_w=create_col(&table,"w",FLOAT,'N',"E12.6","Angstroms");
  colID_data=create_col(&table,"DATA",FLOAT,'N',"E12.6",units);
  for (i=0;i<n;i++) {
    WR_tbl(&table,i,colID_w,&w[i]);
    WR_tbl(&table,i,colID_data,&spec_data[i]);
  }
  close_table(&table);
}

int save_spec_ascii(float *w, float* spec_data, int n, char* filename, char* units) {
  FILE *fichero;
  int i,j;
  if ((fichero=fopen(filename,"w"))==NULL) {
    printf("Error opening file: %s\n",filename);
  } else {
    for (i=0;i<n;i++) {
      fprintf(fichero,"%d %g %g\n",i+1,w[i],spec_data[i]);
    }
  }
  fclose(fichero);
}

/*
 * Save Slice ASCII
 */ 

int save_slice_ascii(float *spaxId, float *x, float *y, float* slice_data, int n, char* filename, char* units) {
  FILE *fichero;
  int colID_x,colID_y,colID_spaxId,colID_data;
  int i,j;
  if ((fichero=fopen(filename,"w"))==NULL) {
    printf("Error opening file: %s\n",filename);
  } else {
    for (i=0;i<n;i++) {
      fprintf(fichero,"%d %g %g %g\n",i+1,x[i],y[i],slice_data[i]);
      // printf("%d %f %f %f\n",i,x[i],y[i],slice_data[i]);
    }
  }
  fclose(fichero);
}

/*
 * Save position table ascii
 */ 

int save_PT_ascii(float *x, float *y, int *int_group, int n, GROUP *grupo, int n_groups, char* filename) {
  FILE *fichero;
  int colID_x,colID_y,colID_spaxId,colID_data;
  int i,j;
  if ((fichero=fopen(filename,"w"))==NULL) {
    printf("Error opening file: %s\n",filename);
  } else {
    for (i=0;i<n_groups;i++) {
      fprintf(fichero,"%c %g %g %g\n",grupo[i].shape,grupo[i].size1,grupo[i].size2,grupo[i].angle);
    }
    for (i=0;i<n;i++) {
      fprintf(fichero,"%d %g %g %d\n",i+1,x[i],y[i],int_group[i]);
    }
  }
  fclose(fichero);
}

/*
 * Save a Map (fitsfile)
 */ 
int save_map(float* map_data, int nx, int ny, char* filename) {
  IMAGE2D image;
  int npix[2];
  double start[2];
  double step[2];
  int i,j;
  npix[0]=nx;
  npix[1]=ny;
  start[0]=0.0;
  start[1]=0.0;
  step[0]=1.0;
  step[1]=1.0;
  create_frame(&image,filename,npix,start,step,FLOAT,"MAP",NULL);
  for (i=0;i<image.nx;i++) {
    for (j=0;j<image.ny;j++) {
      WR_frame(&image,i,j,map_data[i+j*nx]);
      WR_qframe(&image,i,j,(long) 0);
    }
  }
  /*
   * We include some useful information!
   */
  close_frame(&image);  
  return(0);
}


/*
 * Load Row Staked Spectra  (Fitsfile)
 */ 
int load_rss(char *data_filename, char *pt_filename, int *out_nb_spec, int *out_npix, int **out_n_spaxels, SPAXEL ***out_spaxels, int **out_index_start, int **out_npts, float **out_raw_data, float **out_raw_noise, int *out_n_groups, GROUP **out_groups, double *out_start, double *out_step) {

  FILE *fichero;
  IMAGE2D image;
  int npix_p[2];
  double start;
  double step;
  int i,j;

  
  int *index_start, *npts, *n_spaxels;
  float *raw_data,*raw_noise;

  short s_temp[10];
  int i_temp[10];
  float f_temp[10];
  double d_temp[10];
  char str_temp[80];
  float temp_float;
  int temp_int;


  int n_groups;
  GROUP *groups;
  SPAXEL **spaxels,*cur_spax;
  int n_lines,count_f;

  int existe=0;

  float size_x=1.0;
  float size_y=1.0;	
  char shape[20];
  float g_angle;

  int bitpix;



  open_frame(&image,data_filename,"I");

  if(!RD_desc(&image,"BITPIX",SHORT,1,s_temp)) {
    bitpix=-32;
  } else {
    bitpix=s_temp[0];
  }

  if(!RD_desc(&image,"CRVAL1",DOUBLE,1,d_temp)) {
    start=0.0;
  } else {
    start= (float)d_temp[0];
  }
  
  if(!RD_desc(&image,"CD1_1",DOUBLE,1,d_temp)) {
    step=1.0;
  } else {
    step= (float) d_temp[0];
  }

  if(RD_desc(&image,"CDELT1",DOUBLE,1,d_temp)) {
    step= (float) d_temp[0];
  }
  
  if(!RD_desc(&image,"NAXIS1",INT,1,i_temp)) {
    existe=1;
  } else {
    npix_p[0]=i_temp[0];
  }

  if(!RD_desc(&image,"NAXIS2",INT,1,i_temp)) {
    existe=2;
  } else {
    npix_p[1]=i_temp[0];
  }

  printf("Header Readed...\n");
  fflush(stdout);

  /*
   * We read the data if they exist.
   */
  if (existe==0) {
    raw_data = (float *)malloc((npix_p[0]*npix_p[1])*sizeof(float));  
    raw_noise = (float *)malloc((npix_p[0]*npix_p[1])*sizeof(float));  
    for (j=0;j<image.ny;j++) {
      for (i=0;i<image.nx;i++) {
	raw_data[i+j*image.nx] = (float) RD_frame(&image,i,j);
	raw_noise[i+j*image.nx] = sqrt(sqrt(raw_data[i+j*image.nx]*raw_data[i+j*image.nx]));
      }

    }    
    *out_nb_spec=npix_p[1];
    *out_npix=npix_p[0];
    index_start = (int *)malloc((npix_p[1])*sizeof(int));  
    npts = (int *)malloc((npix_p[1])*sizeof(int));  
    n_spaxels = (int *)malloc((npix_p[1])*sizeof(int));  
    //   printf("Paso\n"); 
    //fflush(stdout);
    spaxels = (SPAXEL *) calloc(npix_p[1], sizeof(SPAXEL));
    for (j=0;j<npix_p[1];j++) {
      index_start[j]=0;
      npts[j]=npix_p[0];
      n_spaxels[j]=1;
      spaxels[j] = (SPAXEL *) calloc(1, sizeof(SPAXEL));
    }
  }
  close_frame(&image);  

  printf("RSS Fits readed...\n");
  fflush(stdout);
  /*
   * We read the position table.
   */
  if (existe==0) {
    /*
     * We determine the number of lines
     */
    count_f=0;
    n_lines=0;
    if ((fichero=fopen(pt_filename,"r"))==NULL) {
	printf("Error opening file: %s\n",pt_filename);
	return(-1);
    } else {
	while( fscanf(fichero, "%s", str_temp) != EOF) {
	    count_f++;
	    if (count_f>3) {
		count_f=0;
		n_lines++;
	    }
	}
    }
    rewind(fichero);
    n_groups=n_lines-npix_p[1];
    groups = (GROUP *)malloc((n_groups)*sizeof(GROUP));  
    /*
     * We read the groups information
     */
    for (j=0;j<n_groups;j++) {
      fscanf(fichero, "%s", &str_temp);
      strcpy(shape,str_temp);
      fscanf(fichero, "%f", &size_x);
      fscanf(fichero, "%f", &size_y);
      fscanf(fichero, "%f", &g_angle);      
      printf("GROUP=%d, %s, %f %f %f\n",j+1,str_temp,size_x,size_y,g_angle);      
      groups[j].groupId=j+1;
      groups[j].size1=1.0*size_x; 
      groups[j].size2=1.0*size_y;
      groups[j].shape=shape[0];
      groups[j].angle=g_angle;
      groups[j].poswav=(float)start;
      groups[j].airmass=0.0;
      groups[j].parang=0.0;
      groups[j].pressure=0.0;
      groups[j].temperature=0.0;
    }
    /*
     * Read the Spaxels information
     */
    for (j=0;j<npix_p[1];j++) {
      fscanf(fichero, "%d", &temp_int);
      spaxels[j][0].specId=temp_int;
      fscanf(fichero, "%f", &temp_float);
      spaxels[j][0].xpos=temp_float;
      fscanf(fichero, "%f", &temp_float);
      spaxels[j][0].ypos=temp_float;
      fscanf(fichero, "%d", &temp_int);
      spaxels[j][0].group=temp_int;
      //      printf("SPAXEL= %d,%d, %f, %f, %d\n",j,spaxels[j][0].specId,spaxels[j][0].xpos,spaxels[j][0].ypos,spaxels[j][0].group);


    }
    fclose(fichero);

    /*
     * We present the out of the data
     */
    *out_spaxels=spaxels;
    *out_n_spaxels=n_spaxels;
    *out_index_start=index_start;
    *out_npts=npts;
    *out_raw_data=raw_data;
    *out_raw_noise=raw_noise;
    *out_n_groups=n_groups;
    *out_groups=groups;
    *out_start=start;
    *out_step=step;
  }
  return(existe);
}


/*
 * Load a Slit Spectra  (Fitsfile)
 */ 
int load_spectra(char *data_filename, int *out_nb_spec, int *out_npix, int **out_n_spaxels, SPAXEL ***out_spaxels, int **out_index_start, int **out_npts, float **out_raw_data, float **out_raw_noise, int *out_n_groups, GROUP **out_groups, double *out_start, double *out_step) {

  FILE *fichero;
  IMAGE2D image;
  int npix_p[2];
  double start;
  double step;
  int i,j;

  
  int *index_start, *npts, *n_spaxels;
  float *raw_data,*raw_noise;

  short s_temp[10];
  int i_temp[10];
  float f_temp[10];
  double d_temp[10];
  char str_temp[80];
  float temp_float;
  int temp_int;


  int n_groups;
  GROUP *groups;
  SPAXEL **spaxels,*cur_spax;
  int n_lines,count_f;

  int existe=0;

  float size_x=1.0;
  float size_y=1.0;	
  char shape[20];
  float g_angle;

  int bitpix;
  float x_min,dx;
  float y_min,dy;
  int nx,ny;


  open_frame(&image,data_filename,"I");

  if(!RD_desc(&image,"BITPIX",SHORT,1,s_temp)) {
    bitpix=-32;
  } else {
    bitpix=s_temp[0];
  }

  if(!RD_desc(&image,"CRVAL1",DOUBLE,1,d_temp)) {
    start=0.0;
  } else {
    start= (float)d_temp[0];
  }
  
  if(!RD_desc(&image,"CD1_1",DOUBLE,1,d_temp)) {
    step=1.0;
  } else {
    step= (float) d_temp[0];
  }

  if(RD_desc(&image,"CDELT1",DOUBLE,1,d_temp)) {
    step= (float) d_temp[0];
  }
  
  if(!RD_desc(&image,"NAXIS1",INT,1,i_temp)) {
    existe=1;
  } else {
    npix_p[0]=i_temp[0];
  }

  if(!RD_desc(&image,"NAXIS2",INT,1,i_temp)) {
    existe=2;
  } else {
    npix_p[1]=i_temp[0];
  }


    x_min=0.0;
    dx=1.0;
    y_min=0.0;
    dy=1.0;




  /*
   * We read the data if they exist.
   */
  if (existe==0) {
    raw_data = (float *)malloc((npix_p[0]*npix_p[1])*sizeof(float));  
    raw_noise = (float *)malloc((npix_p[0]*npix_p[1])*sizeof(float));  
    for (j=0;j<image.ny;j++) {
      for (i=0;i<image.nx;i++) {
	raw_data[i+j*image.nx] = (float) RD_frame(&image,i,j);
	raw_noise[i+j*image.nx] = sqrt(sqrt(raw_data[i+j*image.nx]*raw_data[i+j*image.nx]));
      }

    }    
    *out_nb_spec=npix_p[1];
    *out_npix=npix_p[0];
    index_start = (int *)malloc((npix_p[1])*sizeof(int));  
    npts = (int *)malloc((npix_p[1])*sizeof(int));  
    n_spaxels = (int *)malloc((npix_p[1])*sizeof(int));  
    //   printf("Paso\n"); 
    //fflush(stdout);
    spaxels = (SPAXEL *) calloc(npix_p[1], sizeof(SPAXEL));
    for (j=0;j<npix_p[1];j++) {
      index_start[j]=0;
      npts[j]=npix_p[0];
      n_spaxels[j]=1;
      spaxels[j] = (SPAXEL *) calloc(1, sizeof(SPAXEL));
    }
  }
  close_frame(&image);  

  printf("Slit Spectra Fits readed...\n");
  fflush(stdout);



  /*
   * Creating GROUP
   */
  groups = (GROUP *)malloc((1)*sizeof(GROUP));  
  strcpy(shape,"R");
  groups[0].groupId=0+1;
  groups[0].size1=1.0*dx; 
  groups[0].size2=1.0*dy;
  groups[0].shape=shape[0];
  groups[0].angle=0.0;
  groups[0].poswav=(float)start;
  groups[0].airmass=0.0;
  groups[0].parang=0.0;
  groups[0].pressure=0.0;
  groups[0].temperature=0.0;

  temp_int=1;
  for (j=0;j<image.ny;j++) {    
    //    for (i=0;i<image.nx;i++) {
      spaxels[j][0].specId=temp_int;
      spaxels[j][0].xpos=x_min;//+dx*i;
      spaxels[j][0].ypos=y_min+dy*j;
      spaxels[j][0].group=1;
      temp_int++;
      //    }
  }


    /*
     * We present the out of the data
     */
    *out_spaxels=spaxels;
    *out_n_spaxels=n_spaxels;
    *out_index_start=index_start;
    *out_npts=npts;
    *out_raw_data=raw_data;
    *out_raw_noise=raw_noise;
    *out_n_groups=n_groups;
    *out_groups=groups;
    *out_start=start;
    *out_step=step;

  return(existe);
}



/*
 * Load a 3D Cube  (Fitsfile)
 */ 
int load_cube(char *cube_filename, int *out_nb_spec, int *out_npix, int **out_n_spaxels, SPAXEL ***out_spaxels, int **out_index_start, int **out_npts, float **out_raw_data, float **out_raw_noise, int *out_n_groups, GROUP **out_groups, double *out_start, double *out_step) {

  FILE *fichero;
  IMAGE3D cube;
  int npix_p[2];
  double start;
  double step;
  int i,j,k;

  
  int *index_start, *npts, *n_spaxels;
  float *raw_data,*raw_noise;

  short s_temp[10];
  int i_temp[10];
  float f_temp[10];
  double d_temp[10];
  char str_temp[80];
  float temp_float;
  int temp_int;


  int n_groups;
  GROUP *groups;
  SPAXEL **spaxels,*cur_spax;
  int n_lines,count_f;

  int existe=0;

  float size_x=1.0;
  float size_y=1.0;	
  char shape[20];
  float g_angle;

  int bitpix;

  float x_min,dx;
  float y_min,dy;
  int nx,ny;

  open_cube(&cube,cube_filename,"I");
  printf("Opening cube %s\n",cube_filename);
  fflush(stdout);

  if(!RD_desc(&cube,"CRVAL3",DOUBLE,1,d_temp)) {
    start=0.0;
  } else {
    start= (float)d_temp[0];
  }
  
  if(!RD_desc(&cube,"CDELT3",DOUBLE,1,d_temp)) {
    step=1.0;
  } else {
    step= (float) d_temp[0];
  }

  if(!RD_desc(&cube,"CRVAL1",DOUBLE,1,d_temp)) {
    y_min=0.0;
  } else {
    y_min= (float)d_temp[0];
  }
  
  if(!RD_desc(&cube,"CDELT1",DOUBLE,1,d_temp)) {
    dy=1.0;
  } else {
    dy= (float) d_temp[0];
  }

  if(!RD_desc(&cube,"CRVAL2",DOUBLE,1,d_temp)) {
    x_min=0.0;
  } else {
    x_min= (float)d_temp[0];
  }
  
  if(!RD_desc(&cube,"CDELT2",DOUBLE,1,d_temp)) {
    dx=1.0;
  } else {
    dx= (float) d_temp[0];
  }



  nx=cube.nx;
  ny=cube.ny;
  npix_p[0]=cube.nz;
  npix_p[1]=nx*ny;

  printf("Header Readed...\n");
  fflush(stdout);
  /*
   * Creating GROUP
   */
  groups = (GROUP *)malloc((1)*sizeof(GROUP));  
  strcpy(shape,"R");
  groups[0].groupId=0+1;
  groups[0].size1=1.0*dx; 
  groups[0].size2=1.0*dy;
  groups[0].shape=shape[0];
  groups[0].angle=0.0;
  groups[0].poswav=(float)start;
  groups[0].airmass=0.0;
  groups[0].parang=0.0;
  groups[0].pressure=0.0;
  groups[0].temperature=0.0;

  *out_nb_spec=npix_p[1];
  *out_npix=npix_p[0];
  printf("NB_SPEC=%d,NPIX=%d\n",npix_p[1],npix_p[0]);

  index_start = (int *)malloc((npix_p[1])*sizeof(int));  
  npts = (int *)malloc((npix_p[1])*sizeof(int));  
  n_spaxels = (int *)malloc((npix_p[1])*sizeof(int));  
  spaxels = (SPAXEL *) calloc(npix_p[1], sizeof(SPAXEL));
  for (j=0;j<npix_p[1];j++) {
    index_start[j]=0;
    npts[j]=npix_p[0];
    n_spaxels[j]=1;
    spaxels[j] = (SPAXEL *) calloc(1, sizeof(SPAXEL));
  }

  /*
   * We read the data if they exist.
   */
  raw_data = (float *)malloc((npix_p[0]*npix_p[1])*sizeof(float));  
  raw_noise = (float *)malloc((npix_p[0]*npix_p[1])*sizeof(float));  
  temp_int=1;
  printf("%d,%d,%d\n",nx,ny,cube.nz);
  for (j=0;j<nx;j++) {    
    for (i=0;i<ny;i++) {
      spaxels[j*ny+i][0].specId=temp_int;
//      spaxels[j*ny+i][0].xpos=y_min+dy*i;
//      spaxels[j*ny+i][0].ypos=x_min+dx*j;
//      spaxels[j*ny+i][0].xpos=x_min+dx*j;
      spaxels[j*ny+i][0].xpos=x_min-dx*j;
      spaxels[j*ny+i][0].ypos=y_min+dy*i;
      spaxels[j*ny+i][0].group=1;
      //      printf("(%f,%f) %d %d %d\n",spaxels[j*nx+i][0].xpos,spaxels[j*nx+i][0].ypos,i,j,i+j*nx);
      temp_int++;
    }
  }

  for (j=0;j<ny;j++) {    
    for (i=0;i<nx;i++) {
      for (k=0;k<cube.nz;k++) {
	temp_float=(float) RD_cube(&cube,i,j,k);
	raw_data[k+(j+(nx-i-1)*ny)*cube.nz] = temp_float;
	raw_noise[k+(j+(nx-i-1)*ny)*cube.nz] = sqrt(sqrt(temp_float*temp_float));
//	raw_data[k+((ny-j-1)+(nx-i-1)*ny)*cube.nz] = temp_float;
//	raw_noise[k+((ny-j-1)+(nx-i-1)*ny)*cube.nz] = sqrt(sqrt(temp_float*temp_float));
//	raw_noise[k+(j+i*ny)*cube.nz] = sqrt(sqrt(temp_float*temp_float));
      }
    }
  }
    

    
      
  close_cube(&cube);
  /*
   * We present the out of the data
   */
  *out_spaxels=spaxels;
  *out_n_spaxels=n_spaxels;
  *out_index_start=index_start;
  *out_npts=npts;
  *out_raw_data=raw_data;
  *out_raw_noise=raw_noise;
  *out_n_groups=1;
  *out_groups=groups;
  *out_start=start;
  *out_step=step;
  return(0);
}


/*
 * Load a Map  (Fitsfile)
 */ 
int load_map(char *map_filename, int *out_nb_spec, int *out_npix, int **out_n_spaxels, SPAXEL ***out_spaxels, int **out_index_start, int **out_npts, float **out_raw_data, float **out_raw_noise, int *out_n_groups, GROUP **out_groups, double *out_start, double *out_step) {

  FILE *fichero;
  IMAGE2D map;
  int npix_p[2];
  double start;
  double step;
  int i,j,k;

  
  int *index_start, *npts, *n_spaxels;
  float *raw_data,*raw_noise;

  short s_temp[10];
  int i_temp[10];
  float f_temp[10];
  double d_temp[10];
  char str_temp[80];
  float temp_float;
  int temp_int;


  int n_groups;
  GROUP *groups;
  SPAXEL **spaxels,*cur_spax;
  int n_lines,count_f;

  int existe=0;

  float size_x=1.0;
  float size_y=1.0;	
  char shape[20];
  float g_angle;

  int bitpix;

  float x_min,dx;
  float y_min,dy;
  int nx,ny;

  open_frame(&map,map_filename,"I");
  printf("Opening map %s\n",map_filename);
  fflush(stdout);

  /*
    if(!RD_desc(&map,"CRVAL3",DOUBLE,1,d_temp)) {
    start=0.0;
  } else {
    start= (float)d_temp[0];
  }
  
  if(!RD_desc(&map,"CDELT3",DOUBLE,1,d_temp)) {
    step=1.0;
  } else {
    step= (float) d_temp[0];
  }
  */
  start=0.0;
  step=1.0;

  if(!RD_desc(&map,"CRVAL1",DOUBLE,1,d_temp)) {
    y_min=0.0;
  } else {
    y_min= (float)d_temp[0];
  }
  
  if(!RD_desc(&map,"CDELT1",DOUBLE,1,d_temp)) {
    dy=1.0;
  } else {
    dy= (float) d_temp[0];
  }

  if(!RD_desc(&map,"CRVAL2",DOUBLE,1,d_temp)) {
    x_min=0.0;
  } else {
    x_min= (float)d_temp[0];
  }
  
  if(!RD_desc(&map,"CDELT2",DOUBLE,1,d_temp)) {
    dx=1.0;
  } else {
    dx= (float) d_temp[0];
  }


  nx=map.nx;
  ny=map.ny;
  npix_p[0]=1;
  npix_p[1]=nx*ny;

  //  printf("Header Readed...\n");
  //fflush(stdout);
  /*
   * Creating GROUP
   */
  groups = (GROUP *)malloc((1)*sizeof(GROUP));  
  strcpy(shape,"R");
  groups[0].groupId=0+1;
  groups[0].size1=1.0*dx; 
  groups[0].size2=1.0*dy;
  groups[0].shape=shape[0];
  groups[0].angle=0.0;
  groups[0].poswav=(float)start;
  groups[0].airmass=0.0;
  groups[0].parang=0.0;
  groups[0].pressure=0.0;
  groups[0].temperature=0.0;

  *out_nb_spec=npix_p[1];
  *out_npix=npix_p[0];
  //  printf("NB_SPEC=%d,NPIX=%d\n",npix_p[1],npix_p[0]);

  index_start = (int *)malloc((npix_p[1])*sizeof(int));  
  npts = (int *)malloc((npix_p[1])*sizeof(int));  
  n_spaxels = (int *)malloc((npix_p[1])*sizeof(int));  
  spaxels = (SPAXEL *) calloc(npix_p[1], sizeof(SPAXEL));
  for (j=0;j<npix_p[1];j++) {
    index_start[j]=0;
    npts[j]=npix_p[0];
    n_spaxels[j]=1;
    spaxels[j] = (SPAXEL *) calloc(1, sizeof(SPAXEL));
  }

  /*
   * We read the data if they exist.
   */
  raw_data = (float *)malloc((npix_p[0]*npix_p[1])*sizeof(float));  
  raw_noise = (float *)malloc((npix_p[0]*npix_p[1])*sizeof(float));  
  temp_int=1;
  //  printf("%d,%d,%d\n",nx,ny,1);
  for (j=0;j<nx;j++) {    
    for (i=0;i<ny;i++) {
      spaxels[j*ny+i][0].specId=temp_int;
      spaxels[j*ny+i][0].xpos=y_min+dy*i;
      spaxels[j*ny+i][0].ypos=x_min+dx*j;
      spaxels[j*ny+i][0].group=1;
      //      printf("(%f,%f) %d %d %d\n",spaxels[j*nx+i][0].xpos,spaxels[j*nx+i][0].ypos,i,j,i+j*nx);
      temp_int++;
    }
  }

  for (j=0;j<ny;j++) {    
    for (i=0;i<nx;i++) {
      //   for (k=0;k<map.nz;k++) {
	temp_float=(float) RD_frame(&map,i,j);
	raw_data[j+i*ny] = temp_float;
	raw_noise[j+i*ny] = sqrt(sqrt(temp_float*temp_float));
	//      }
    }
  }
    

    
      
  close_frame(&map);
  /*
   * We present the out of the data
   */
  *out_spaxels=spaxels;
  *out_n_spaxels=n_spaxels;
  *out_index_start=index_start;
  *out_npts=npts;
  *out_raw_data=raw_data;
  *out_raw_noise=raw_noise;
  *out_n_groups=1;
  *out_groups=groups;
  *out_start=start;
  *out_step=step;
  return(0);
}


int save_raw(float* map_data, int nx, int ny, char* filename, double start_w, double step_w) {
  IMAGE2D image;
  int npix[2];
  double start[2];
  double step[2];
  int i,j;
  npix[0]=nx;
  npix[1]=ny;
  start[0]=0.0;
  start[1]=0.0;
  step[0]=1.0;
  step[1]=1.0;
  create_frame(&image,filename,npix,start,step,FLOAT,"MAP",NULL);
  for (i=0;i<image.nx;i++) {
    for (j=0;j<image.ny;j++) {
      WR_frame(&image,i,j,map_data[i+j*nx]);
      WR_qframe(&image,i,j,(long) 0);
    }
  }
  /*
   * We include some useful information!
   */
  start[0]=start_w;
  step[0]=step_w;
  WR_desc(&image,"CRVAL1",DOUBLE,1,start);
  WR_desc(&image,"CD1_1",DOUBLE,1,step);
  WR_desc(&image,"CDELT1",DOUBLE,1,step);
  close_frame(&image);  
  return(0);
}


/*
 * Create an interpolated Map
 * saving its WCS
 */

int create_map(int nb_spec, int *n_spaxels, SPAXEL **spaxels, int *npts, float* slice, float dpix, float** map_data, float x_min, float x_max, float y_min, float y_max, float *ttr[8], int *nnx, int *nny, int grid_func, float grid_opt) {
  int i,j,k,l,m;
  int nx,ny;
  float *temp_data;
  float x_c,y_c,xx[4],yy[4];
  float tr[]={0.0,1.0,0.0,0.0,0.0,1.0};

  int ier;
  float *x,*y,*z;
  int n_tot;

  float *xo,*yo,**zo,rpnt;


  nx = floor((x_max-x_min)/dpix);
  ny = floor((y_max-y_min)/dpix);


  /*
   * We define the points in the grid
   */
  xo = (float *) calloc(nx,sizeof(float));    
  yo = (float *) calloc(ny,sizeof(float));    
  zo = (PLFLT **) calloc(nx ,sizeof(PLFLT*));
  for (i = 0; i < nx; i++) {
      zo[i] = (PLFLT *) calloc(ny ,sizeof(PLFLT));
  }


 
  for (i=0;i<nx;i++) {
      xo[i]=x_min+dpix*i;
  }
  for (j=0;j<ny;j++) {
      yo[j]=y_min+dpix*j;
  }
  


  /*
   * We count the number of points included in the
   * SPAXELS array
   *
   */
  n_tot=0;
  for (k=0;k<nb_spec;k++) {
    for (l=0;l<n_spaxels[k];l++) {
      n_tot++;
    }
  }
  x = (float *)malloc(n_tot*sizeof(float));    
  y = (float *)malloc(n_tot*sizeof(float));    
  z = (float *)malloc(n_tot*sizeof(float));    
  /*
   * We feed with data
   */
  n_tot=0;
  for (k=0;k<nb_spec;k++) {
    for (l=0;l<n_spaxels[k];l++) {
      x[n_tot]=spaxels[k][l].xpos;
      y[n_tot]=spaxels[k][l].ypos;
      z[n_tot]=slice[k];
      n_tot++;
    }
  }


  /*
   * We interpolate!
   */
//  my_griddata(x, y, z, n_tot, xo, nx, yo, ny, zo, GRID_CSA, 0.0);
  my_griddata(x, y, z, n_tot, xo, nx, yo, ny, zo, grid_func, grid_opt);
  
  /*
   *  Enter single point mode.
   */

  temp_data = (float *) calloc(nx*ny,sizeof(float));    

  for (i=0;i<nx;i++) {
      for (j=0;j<ny;j++) {
	  temp_data[i+j*nx] = zo[i][j];	  
      }
  }
  

  /*
  c_nnseti("ext",0);
  c_nnsetr("nul",0);  
  c_nnpntinits(n_tot, x, y, z);
  k=0;
  for (i=0;i<nx;i++) {
    for (j=0;j<ny;j++) {
      c_nnpnts(xo[k], yo[k], &rpnt);
      temp_data[i+j*nx] = rpnt;
      k++;
    }
  }
  
  */




  *nnx=nx;
  *nny=ny;
  *ttr=tr;
  *map_data=temp_data;
  return(0);
}



/*
 * Create a 3D CUBE from a Euro3D pseudo-format (E3D stored data).
 */

int save_cube(int nb_spec, int npix_p, int *n_spaxels, SPAXEL **spaxels, float* raw_data, float dpix, char *cube_filename, float x_min, float x_max, float y_min, float y_max, int grid_func, float grid_opt, double start_w, double step_w) {
  int i,j,k,l,m;
  int nx,ny;
  float *temp_data;
  float x_c,y_c,xx[4],yy[4];


  int ier;
  float *x,*y,*z;
  int n_tot;

  float *xo,*yo,**zo,rpnt;


  IMAGE3D cube;
  double start[3],step[3];
  int npix[3];




  nx = floor((x_max-x_min)/dpix);
  ny = floor((y_max-y_min)/dpix);

  start[0]=x_min; start[1]=y_min; start[2]=start_w;
  step[0]=dpix; step[1]=dpix; step[2]=step_w;
  /*
  start[0]=1; start[1]=1; start[2]=1;
  step[0]=1; step[1]=1; step[2]=1;
  */
  npix[0]=nx; npix[1]=ny; npix[2]=npix_p;
  
  create_cube(&cube,cube_filename,npix,start,step,FLOAT,"E3D Cube",NULL);
  printf("Creating cube %s\n",cube_filename);
  fflush(stdout);
  //  printf("name=%s, [%d,%d,%d] [%d,%d,%d], [%f,%f,%f]\n",cube_filename,npix[0],npix[1],npix[2],cube.nx,cube.ny,cube.nz,start[0],start[1],start[2]);

  /*
   * We define the points in the grid
   */
  xo = (float *) calloc(nx,sizeof(float));    
  yo = (float *) calloc(ny,sizeof(float));    
  zo = (PLFLT **) calloc(nx ,sizeof(PLFLT*));
  for (i = 0; i < nx; i++) {
      zo[i] = (PLFLT *) calloc(ny ,sizeof(PLFLT));
  }


 
  for (i=0;i<nx;i++) {
      xo[i]=x_min+dpix*i;
  }
  for (j=0;j<ny;j++) {
      yo[j]=y_min+dpix*j;
  }
  


  /*
   * We count the number of points included in the
   * SPAXELS array
   *
   */
  n_tot=0;
  for (k=0;k<nb_spec;k++) {
    for (l=0;l<n_spaxels[k];l++) {
      n_tot++;
    }
  }
  x = (float *)malloc(n_tot*sizeof(float));    
  y = (float *)malloc(n_tot*sizeof(float));    
  z = (float *)malloc(n_tot*sizeof(float));    
  //  temp_data = (float *) calloc(nx*ny,sizeof(float));    

  for (m=0;m<npix_p;m++) {

    n_tot=0;
    for (k=0;k<nb_spec;k++) {
      //      x[n_tot]=0;
      //y[n_tot]=0;
      for (l=0;l<n_spaxels[k];l++) {
	/*
	x[n_tot]+=spaxels[k][l].xpos/n_spaxels[k];
	y[n_tot]+=spaxels[k][l].ypos/n_spaxels[k];
	*/
	//	z[n_tot]=slice[k];
	x[n_tot]=spaxels[k][l].xpos;
	y[n_tot]=spaxels[k][l].ypos;
	z[n_tot]=raw_data[m+k*npix_p];
	n_tot++;
      }
    }
    my_griddata(x, y, z, n_tot, xo, nx, yo, ny, zo, grid_func, grid_opt);
    for (i=0;i<nx;i++) {
      for (j=0;j<ny;j++) {
	WR_cube(&cube,i,j,m,zo[i][j]);
	//	printf("(%d,%d,%d)=%f\n",i,j,m,zo[i][j]);
      }
    }
    

  }
  close_cube(&cube);
  printf("Closing cube %s\n",cube_filename);
  fflush(stdout);
  return(0);
}

/*
 * Create an interpolated Map
 * saving its WCS
 */

int old_create_map(int nb_spec, int *n_spaxels, SPAXEL **spaxels, int *index_start, int *npts, float* slice, float dpix, float** map_data, float x_min, float x_max, float y_min, float y_max, float *ttr[8], int *nnx, int *nny) {
  int i,j,k,l,m;
  int nx,ny;
  float *temp_data;
  float x,y,x_c,y_c,xx[4],yy[4];
  //  float tr[8];
  float tr[]={0.0,1.0,0.0,0.0,0.0,1.0};
  float d[4],dist_temp,d_par[4][4],max_par[4];
  int index[4];
  float prod=1.0;

  nx = floor((x_max-x_min)/dpix);
  ny = floor((y_max-y_min)/dpix);
  temp_data = (float *)malloc(nx*ny*sizeof(float));    


  //  printf("%d,%d,%f,%f,%f,%f\n",nx,ny,x_min,x_max,y_min,y_max);

  //  printf("Paso= CREATE MAP\n");
  for (j=0;j<ny;j++) {
    for (i=0;i<nx;i++) {
      for (k=0;k<4;k++) {
	index[k]=0;
	d[k]=32000.0;
      }
      temp_data[i+j*nx]=0.0;
      x=((float)i)*dpix+x_min;
      y=((float)j)*dpix+y_min;
      /*
       * we move along all the SPAXELs
       * to determine the distances
       */
      for (k=0;k<nb_spec;k++) {
	for (l=0;l<n_spaxels[k];l++) {
	  x_c=spaxels[k][l].xpos;
	  y_c=spaxels[k][l].ypos;
	  dist_temp=sqrt(pow(x_c-x,2)+pow(y_c-y,2));
	  //	  printf("%f,%f,%f,%f,%f\n",dist_temp,d[0],d[1],d[2],d[3]);
	  if (dist_temp<=d[0]) {
	    d[3]=d[2];
	    d[2]=d[1];
	    d[1]=d[0];
	    d[0]=dist_temp;
	    index[3]=index[2];
	    index[2]=index[1];
	    index[1]=index[0];
	    index[0]=k;
	    xx[3]=xx[2];
	    xx[2]=xx[1];
	    xx[1]=xx[0];
	    xx[0]=x_c;
	    yy[3]=yy[2];
	    yy[2]=yy[1];
	    yy[1]=yy[0];
	    yy[0]=y_c;
	  } else if (dist_temp<=d[1]) {
	    d[3]=d[2];
	    d[2]=d[1];
	    d[1]=dist_temp;
	    index[3]=index[2];
	    index[2]=index[1];
	    index[1]=k;
	    xx[3]=xx[2];
	    xx[2]=xx[1];
	    xx[1]=x_c;
	    yy[3]=yy[2];
	    yy[2]=yy[1];
	    yy[1]=y_c;
	  }  else if (dist_temp<=d[2]) {
	    d[3]=d[2];
	    d[2]=dist_temp;
	    index[3]=index[2];
	    index[2]=k;
	    xx[3]=xx[2];
	    xx[2]=x_c;
	    yy[3]=yy[2];
	    yy[2]=y_c;
	  }   else if (dist_temp<=d[3]) {
	    d[3]=dist_temp;
	    index[3]=k;
	    xx[3]=x_c;
	    yy[3]=y_c;
	  }	  
	}	
      }

      for (l=0;l<4;l++) {
	max_par[l]=-1000;
	for (k=0;k<4;k++) {
	  d_par[l][k]=sqrt(pow(xx[l]-xx[k],2)+pow(yy[l]-yy[k],2));
	  if (d_par[l][k]>max_par[l]) max_par[l]=d_par[l][k];
	}
      }
      
      /*
       * We test if it is among the points!
       */
      if ((d[0]+d[1]+d[2])<(d_par[0][1]+d_par[0][2]+d_par[1][2])) {
	
	prod=0.0;
	for (l=0;l<3;l++) {
	  prod+=1/(1.0+d[l]);
	  temp_data[i+j*nx]+=slice[index[l]]/(1.0+d[l]);	
	  
	}
	temp_data[i+j*nx]/=prod;
      } else {
	temp_data[i+j*nx]=-1000;
      }
    }
  }

  //  printf("Paso= CREATE MAP\n");

  *nnx=nx;
  //  printf("1\n");
  *nny=ny;
  //printf("2\n");
  *ttr=tr;
  //printf("3\n");
  *map_data=temp_data;
  // printf("4\n");
  //printf("Paso\n");
  return(0);
}



/*
 * Mark a range with an arrow and a text 
 */
int mark_range(float x1, float y1, float x2, float y2, char* text) {
  float x_m,y_m;
  x_m=(x2+x1)/2;
  y_m=(y2+y1)/2;
  cpgarro(x_m,y_m,x1,y1);
  cpgarro(x_m,y_m,x2,y2);
  cpgptxt(x_m,y_m+1.0,0.0,1.5,text);
  return(0);
}

/*
 * Plot a certain spectra
 */
void spec_plot(float* data, float datamin, float datamax, int npix, double start_w, double delta_w) {
  float *x,*y;
  int i,j;

  if (start_w==0.0)
    start_w=1.0;
  if (delta_w==0.0)
    delta_w=1.0;

  x = (float *)malloc(npix*sizeof(float));  
  y = (float *)malloc(npix*sizeof(float));  
  for (i=0;i<npix;i++) {
    x[i]=(float)start_w+(float)delta_w*i;
    y[i]=data[i];
  }
  cpgsch(1.5);  
  cpgenv(x[0],x[npix-1],datamin,datamax,0,0);
  cpgsch(1.2);  
  cpglab("\\gl (Angstroms)", "Flux", "");	  
  cpgsls(1);  
  cpgline(npix,x,y);
  
}

/*
 * Plot a certain section of a spectrum
 *
 */

void spec_plot_range(float* data, float datamin, float datamax, int npix_w1, int npix_w2, double start_w, double delta_w, int clean) {
  float *x,*y;
  int i,j;

  if (start_w==0.0)
    start_w=1.0;
  if (delta_w==0.0)
    delta_w=1.0;

  x = (float *)malloc((npix_w2-npix_w1)*sizeof(float));  
  y = (float *)malloc((npix_w2-npix_w1)*sizeof(float));  
  for (i=npix_w1;i<npix_w2;i++) {
    x[i-npix_w1]=(float)start_w+(float)delta_w*i;
    y[i-npix_w1]=data[i];
  }
  //  printf("clean=%d\n",clean);
  if (clean==0) {
      cpgsch(1.2);    
      cpgenv(x[0],x[npix_w2-npix_w1-1],datamin,datamax,0,0);
      cpgsch(1.5);  
      cpglab("\\gl (Angstroms)", "Flux", "");	
  }
  cpgsls(1);  
  cpgsci(clean+1);
  cpgline((npix_w2-npix_w1),x,y);
  cpgsci(1);
}

/*
 * Fit a single line to a section of an spectra.
 *
 */

float fit_single_line(float *data, float *noise, float datamin, float datamax, int npix_w1, int npix_w2, double start_w, double delta_w, int clean, float a[4], float d_a[4], int ia[4], float chisq_cont) {
  float *x,*y,*s_y,*y_mod,*y_res,y_tmp;
  int i,j,ndat;

  float s_a[4], chisq;
  char text[80];

  //  int ia[]={1,1,1,1};

  if (start_w==0.0)
    start_w=1.0;
  if (delta_w==0.0)
    delta_w=1.0;

  /*
   * Creating the data
   */
  ndat=npix_w2-npix_w1;
  x = (float *)malloc((npix_w2-npix_w1)*sizeof(float));  
  y = (float *)malloc((npix_w2-npix_w1)*sizeof(float));  
  y_mod = (float *)malloc((npix_w2-npix_w1)*sizeof(float));  
  y_res = (float *)malloc((npix_w2-npix_w1)*sizeof(float));  
  s_y = (float *)malloc((npix_w2-npix_w1)*sizeof(float));  
  for (i=npix_w1;i<npix_w2;i++) {
    x[i-npix_w1]=(float)start_w+(float)delta_w*i;
    y[i-npix_w1]=data[i];
    s_y[i-npix_w1]=noise[i];
  }
  /*
   * Fitting
   */
  gl_fit(x,y,s_y,0,ndat,a,s_a,d_a,ia,4,&chisq,chisq_cont,&gl_fit_gauss);

  for (i=0;i<ndat;i++) {
    gl_fit_gauss(x[i],a,&y_tmp);
    y_mod[i]=y_tmp;
    y_res[i]=y[i]-y_mod[i];
  }
  /*
   * Plotting the results
   */
  if (clean>-1) {
    if (clean==0) {
      cpgsch(1.2);    
      cpgenv(x[0],x[npix_w2-npix_w1-1],datamin,datamax,0,0);
      cpgsch(1.5);  
      cpglab("\\gl (Angstroms)", "Flux", "");	
    }
    cpgsls(1);  
    cpgsci(clean+1);
    cpgline((npix_w2-npix_w1),x,y);
    cpgsci(clean+2);
    cpgline((npix_w2-npix_w1),x,y_mod);
    cpgsci(clean+3);
    cpgpt((npix_w2-npix_w1),x,y_res,2);
    cpgsci(1);

    sprintf(text,"Results=%f %f %f %f %f",a[0],a[1],a[2],a[3],chisq);  
    cpgsci(3);
    cpgptxt(x[0],0.9*datamax,0.0,1.5,text); 
    cpgsci(1);
  }
  return(chisq);


}




/**************************************
 * Ask to Open a device
 **************************************/
int ask_open_device(char *commands) {
  int *flag;
  int tmp_address;
  int mem_key;
  int n_bytes;
  //  printf("%s\n",commands);

  flag = (int *)malloc(15*sizeof(int));
  load_mem_char(commands,4,80,&tmp_address);
  mem_key=3;
  flag[0]=101;
  flag[1]=0;
  flag[2]=0;
  n_bytes=15;       
  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
  while(flag[1]==0) { 
    //    printf("Aqui me quedo?\n");
    read_mem_int(&flag,mem_key,n_bytes);		
  }
  return(flag[3]);
}


/**************************************
 * Ask to Close a device
 **************************************/
int ask_close_device(char *commands) {
  int *flag;
  int tmp_address;
  int mem_key;
  int n_bytes;

  flag = (int *)malloc(15*sizeof(int));
  load_mem_char(commands,4,80,&tmp_address);
  mem_key=3;
  flag[0]=102;
  flag[1]=0;
  flag[2]=0;
  n_bytes=15;       
  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
  while(flag[1]==0) { 
    read_mem_int(&flag,mem_key,n_bytes);		
  }
  return(0);
}


/**************************************
 * Ask to draw the raw data
 **************************************/
int ask_draw_raw(char *commands) {
  int *flag;
  int tmp_address;
  int mem_key;
  int n_bytes;

  flag = (int *)malloc(15*sizeof(int));
  load_mem_char(commands,4,80,&tmp_address);
  mem_key=3;
  flag[0]=103;
  flag[1]=0;
  flag[2]=0;
  n_bytes=15;       
  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
  while(flag[1]==0) { 
    read_mem_int(&flag,mem_key,n_bytes);		
  }
  return(0);
}

/**********************************************
 * Ask to change the Palette
 * Where Commands='Pal Bright Contra Sign DEV_ID'
 **********************************************/
int ask_change_palette(char *commands) {
  int *flag;
  int tmp_address;
  int mem_key;
  int n_bytes;

  flag = (int *)malloc(15*sizeof(int));
  load_mem_char(commands,4,80,&tmp_address);
  mem_key=3;
  flag[0]=100;
  flag[1]=0;
  flag[2]=0;
  n_bytes=15;       
  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
  while(flag[1]==0) { 
    read_mem_int(&flag,mem_key,n_bytes);		
  }
  return(0);
}


/**********************************************
 * Ask for a single intensity in a point
 * defined by its index
 **********************************************/
int ask_intensity(float *out_raw_data, int nb_spec, int index_w) {
  int *flag;
  int tmp_address;
  int mem_key;
  int n_bytes;
  int i,j,jj;
  float *raw_spec;
  flag = (int *)malloc(15*sizeof(int));  
  mem_key=3;
  n_bytes=15;
  flag[0]=7;
  flag[1]=0;
  flag[2]=index_w;
  flag[3]=nb_spec;
  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
  while(flag[1]==0) { 
    read_mem_int(&flag,mem_key,n_bytes);		
  }
  read_mem_float(&raw_spec,2,1);	    
  *out_raw_data=raw_spec[0];  
}

/**********************************************
 * Ask for an SLIDE of DATA at a certain Wavelength
 * defined by its index
 **********************************************/
int ask_alive() {
  int *flag;
  int tmp_address;
  int mem_key;
  int n_bytes;
  time_t t1,t2;
  int overtime=0;

  flag = (int *)malloc(15*sizeof(int));  
  mem_key=3;
  n_bytes=15;
  /*
   * We send a signal to the server
   */
  flag[0]=666;
  flag[1]=0;
  flag[2]=666;
  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
  (void) time(&t1);
  /*
   * Waint until an answer
   */
  while((flag[1]==0)&&(overtime==0)) { 
    read_mem_int(&flag,mem_key,n_bytes);		
    (void) time(&t2);
    //    printf("TIME=%d\n",(int)(t2-t1));
    if (((int)(t2-t1))>1) overtime=1;
  }
  /*
   * Read each spectra plot by the 
   */
  //  printf("Overtime=%d\n",overtime);
  return(overtime);
}


/**********************************************
 * Ask for an SLIDE of DATA at a certain Wavelength
 * defined by its index
 **********************************************/
int ask_slice(float **out_raw_data, int nb_spec, int index_w) {
  int *flag;
  int tmp_address;
  int mem_key;
  int n_bytes;
  int i,j,jj;
  float *raw_spec,*raw_data;
  flag = (int *)malloc(15*sizeof(int));  
  raw_data = (float *)malloc((nb_spec)*sizeof(float));
  mem_key=3;
  n_bytes=15;
  /*
   * We send a signal to the server
   */
  flag[0]=5;
  flag[1]=0;
  flag[2]=index_w;
  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
  /*
   * Waint until an answer
   */
  while(flag[1]==0) { 
    read_mem_int(&flag,mem_key,n_bytes);		
  }
  /*
   * Read each spectra plot by the 
   */
  read_mem_float(&raw_spec,2,nb_spec);	    
  for (i=0;i<nb_spec;i++) {
    raw_data[i]=raw_spec[i];
  }
  *out_raw_data=raw_data;  
  return(0);
}


/**********************************************
 * Ask for Only one Spectra, of specId
 **********************************************/
int ask_spectrum(float **out_raw_data, int specId, int npix, int index_start) {
  int *flag;
  int tmp_address;
  int mem_key;
  int n_bytes;
  int i,j,jj;
  float *raw_spec,*raw_data;
  flag = (int *)malloc(15*sizeof(int));  
  raw_data = (float *)malloc((npix)*sizeof(float));
  mem_key=3;
  n_bytes=15;
  j=specId-1;
  jj=j+1;
  /*
   * We send a signal to the server
   */
  flag[0]=2;
  flag[1]=0;
  flag[2]=j;
  flag[3]=jj;
  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
  /*
   * Waint until an answer
   */
  while(flag[1]==0) { 
    read_mem_int(&flag,mem_key,n_bytes);		
  }
  flag[0]=0;
  flag[1]=0;
  flag[2]=0;
  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
  /*
   * Read each spectra plot by the 
   */
  read_mem_float(&raw_spec,2,npix);	    
  for (i=0;i<npix;i++) {
    raw_data[i]=raw_spec[i+index_start];
  }
  *out_raw_data=raw_data;  
}


/**********************************************
 * Ask for the Staked Spectra as an array of floats!
 **********************************************/
int ask_raw_data(float **out_raw_data, int nb_spec, int npix, int delta_n) {
  int *flag;
  int tmp_address;
  int mem_key;
  int n_bytes;
  int i,j,jj;
  float *raw_spec,*raw_data;
  flag = (int *)malloc(15*sizeof(int));  
  raw_data = (float *)malloc((nb_spec*npix)*sizeof(float));
  mem_key=3;
  n_bytes=15;
  for (j=0;j<nb_spec;j=j+delta_n) {
    jj=j+delta_n;
    if (jj>=nb_spec) {
      jj=nb_spec;
      delta_n=jj-j;
    }
     /*
      * We send a signal to the server
      */
    flag[0]=2;
    flag[1]=0;
    flag[2]=j;
    flag[3]=jj;
    load_mem_int(flag,mem_key,n_bytes,&tmp_address);
    /*
     * Waint until an answer
     */
    while(flag[1]==0) { 
      read_mem_int(&flag,mem_key,n_bytes);		
    }
    flag[0]=0;
    flag[1]=0;
    flag[2]=0;
    load_mem_int(flag,mem_key,n_bytes,&tmp_address);
    /*
     * Read each spectra plot by the 
     */
    read_mem_float(&raw_spec,2,npix);	    
    for (i=0;i<npix*delta_n;i++) {
      raw_data[i+j*npix]=raw_spec[i];
    }
  }
  *out_raw_data=raw_data;
}

/**********************************************
 * Ask for the E3D image INFO
 **********************************************/

int ask_e3d_info(E3D_file *e3d_frame, int **n_spaxels, SPAXEL ***spaxels, int **index_start, int **npts, GROUP **groups) {

  E3D_file e3d_temp;
  int *n_spaxels_temp;
  SPAXEL **spaxels_temp;
  int *index_start_temp;
  int *npts_temp;
  GROUP *groups_temp;
  int i,j,k;
  int nb_spec, npix, specId;
  int *nol_temp, max_npts;
  float delta_w;
  int mem_key;
  int n_groups;
  double start_w,end_w;

  mem_key=1;
  //printf("We read the E3D_file from memory\n");
  if (read_mem_e3d(&e3d_temp,mem_key)!=0) {
    //printf("No E3D descriptor on SHM");
    //printf("\n");
    exit(0);
  }
  start_w=e3d_temp.common_parameters[0];
  end_w=e3d_temp.common_parameters[1];
  npix=e3d_temp.common_parameters[2];
  delta_w=(end_w-start_w)/npix;
  nb_spec=e3d_temp.nbspec;
  n_groups=e3d_temp.ngroups;
  //printf("Ok\n");

  //printf("We read the N_SPAXELS from the memory\n");
  mem_key=5;
  read_mem_int(&n_spaxels_temp,mem_key,nb_spec);
  //printf("Ok\n");
  //printf("We read the SPAXELS from the memory\n");
  mem_key=6;
  read_mem_spaxels(&spaxels_temp,n_spaxels_temp,mem_key,nb_spec);
  //printf("Ok\n");
  //printf("We read the INDEX_START from the memory\n");
  mem_key=7;	
  read_mem_int(&index_start_temp,mem_key,nb_spec);
  //printf("Ok\n");
  //printf("We read the NPTS from the memory\n");
  mem_key=8;
  read_mem_int(&npts_temp,mem_key,nb_spec);
  //printf("Ok\n");
  //printf("We read the GROUPS from the memory\n");
  mem_key=9;
  read_mem_group(&groups_temp,mem_key,nb_spec);
  //printf("Ok\n");

  *e3d_frame=e3d_temp;  
  *n_spaxels=n_spaxels_temp;
  *spaxels=spaxels_temp;
  *index_start=index_start_temp;
  *npts=npts_temp;
  *groups=groups_temp;

  return(0);
}


/**********************************************
 * ask to load a new FILE and ask for the 
 * descriptors information.
 **********************************************/
int ask_load_e3d(char *input_filename) {
  int *flag;
  int tmp_address;
  int mem_key;
  int n_bytes;

  flag = (int *)malloc(15*sizeof(int));

  // printf("Asking to load '");
  // printf(input_filename);
  // printf("'\n");
  fflush(stdout);
  /* Load the Filename in the SHM space */
  load_mem_char(input_filename,4,80,&tmp_address);
  // printf("Paso");
  mem_key=3;
  flag[0]=3;
  flag[1]=0;
  flag[2]=0;
  n_bytes=15;       
  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
  while(flag[1]==0) { 
    read_mem_int(&flag,mem_key,n_bytes);		
  }
  return(0);
}

/**********************************************
 * ask to delete all the SHM
 * Send a command to the server to delete the SHM
 **********************************************/
int ask_delete_shm() {
  int *flag;
  int tmp_address;
  int mem_key;
  int n_bytes;

  flag = (int *)malloc(15*sizeof(int));

  mem_key=3;
  flag[0]=4;
  flag[1]=0;
  flag[2]=0;
  n_bytes=15;
  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
  while(flag[1]==0) { 
    read_mem_int(&flag,mem_key,n_bytes);		
  }
  flag[0]=0;
  flag[1]=0;
  flag[2]=0;
  n_bytes=15;
  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
  return(0);
}


/*********************************************
 * Load in Memory, not in the SHM, all the information available 
 * for a certain E3D_file
 *********************************************/
int load_mem_all (E3D_file e3d_image, int *n_spaxels,SPAXEL **spaxels,int *index_start, int *npts, int **address) {
  int tmp_address;
  int npix,nb_spec,n_groups;
  int mem_key,n_bytes;

  int j;

  int caracter;

  nb_spec=e3d_image.nbspec;  
  n_groups=e3d_image.ngroups;
  npix=e3d_image.common_parameters[2];

  /*
   * We load in memory the Euro3D image
   */
  
  //  address = (int *)malloc(15*sizeof(int));  	
  
  mem_key=1;
  if (load_mem_e3d(e3d_image,mem_key,&tmp_address)!=0) {
    //printf("Error loading in SHM\n");
    return(1);
  }
  address[mem_key]=tmp_address;
  //  printf("add=%d\n",address[mem_key]);
  //printf("Load Ok\n");

  //caracter=getc(stdin);

  /*
   * We load the N_SPAXELS in Memory
   */
  //printf("\nWe store in the shared memory the N_SPAXELS\n");
  mem_key=5;
  n_bytes=nb_spec;
  if (n_bytes<MAX_SHM) {
    load_mem_int(n_spaxels,mem_key,n_bytes,&tmp_address);
    //printf("Ok\n");
  } else {
    //printf("*******************************\n");
    //printf("** TOO MUCH LARGE FOR THE SHM**\n");
    //printf("*******************************\n");
  }
  address[mem_key]=tmp_address;
  //caracter=getc(stdin);
  /*
   * We load the SPAXELS in Memory
   */
  //printf("\nWe store in the shared memory the SPAXELS\n");
  mem_key=6;
  n_bytes=nb_spec;
  if (n_bytes<MAX_SHM) {
    load_mem_spaxels(spaxels,n_spaxels,mem_key,n_bytes,&tmp_address);
    //printf("Ok\n");
  } else {
    //printf("*******************************\n");
    //printf("** TOO MUCH LARGE FOR THE SHM**\n");
    //printf("*******************************\n");
  }
  address[mem_key]=tmp_address;
  //caracter=getc(stdin);
  /*
   * We load the INDEX_START in Memory
   */
  //printf("\nWe store in the shared memory the INDEX_START\n");
  mem_key=7;
  n_bytes=nb_spec;
  if (n_bytes<MAX_SHM) {
    load_mem_int(index_start,mem_key,n_bytes,&tmp_address);
    //printf("Ok\n");
  } else {
    //printf("*******************************\n");
    //printf("** TOO MUCH LARGE FOR THE SHM**\n");
    //printf("*******************************\n");
  }
  address[mem_key]=tmp_address;
  //caracter=getc(stdin);
  /*
   * We load the NPTS in Memory
   */
  //printf("\nWe store in the shared memory the NPTS\n");
  mem_key=8;
  n_bytes=nb_spec;
  if (n_bytes<MAX_SHM) {
    load_mem_int(npts,mem_key,n_bytes,&tmp_address);
    //printf("Ok\n");
  } else {
    //printf("*******************************\n");
    //printf("** TOO MUCH LARGE FOR THE SHM**\n");
    //printf("*******************************\n");
  }
  address[mem_key]=tmp_address;
  //caracter=getc(stdin);
  //printf("\nWe store in the shared memory the GROUPS info\n");
  mem_key=9;
  n_bytes=nb_spec;
  if (n_bytes<MAX_SHM) {
    load_mem_group(e3d_image.groups,mem_key,n_bytes,&tmp_address);
    //printf("Ok\n");
  } else {
    //printf("*******************************\n");
    //printf("** TOO MUCH LARGE FOR THE SHM**\n");
    //printf("*******************************\n");
  }
  address[mem_key]=tmp_address;
  //caracter=getc(stdin);
  /*
  for (j=1;j<10;j++) {
    printf("address[%d]=%d\n",j,address[j]);
  }
  */
}



/*********************************************
 * Function that uses SHM to work as a server
 * to serve the requested information
 *
 * You have to use a FLAG inserved in the SHM
 * to allow the input of Commands in the 
 * SERVER
 * The FLAG is an array 15 elements
 * the 1st elements is the command.
 *
 * SHM key of the FLAG = 3
 * 
 * FLAG[0]=1 ask for a certain spectra, specId in FLAG[2]
 * FLAG[0]=2 ask for a secuence of spectra, specId from FLAG[2] to FLAG[3]
 *
 * FLAG[0]=3 ask for loading a new E3D_FILE
 * 
 *********************************************/
int server() {

  /*************************************************
   * Painting variables
   *
   *
   *************************************************/
  int p=2;
  float contra=0.8;
  float bright=0.4;
  float sign=1.0;
  
  float tr_now[]={0.0,1.0,0.0,0.0,0.0,1.0};




  int caracter;

    int *flag,temp_flag;

    int *address,tmp_address;
    int n_bytes;

    int window_size;
    float window_aspect;

    int n_colors,nc=200,color_index;
    float x_c,y_c;
    float size1=0.5,size2;
    



    GROUP group, *groups, fake_group;
    GROUP grupo;
    int n_groups;
    
    SPAXEL *spax, **spaxels, *order_spaxels;  
    int *n_spaxels,*index_start,*npts,max_npts;
    
    float x_min,y_min,x_max,y_max;
    
    
    E3D_file e3d_image,e3d_mem,shm;
    SPECTRUM signal,noise;
    SPECTRUM *spectra;
    SLICE s_signal;
    IMAGE2D image;
    char **argval, **arglabel;
    int nbal;
    
    int mem_key;
    int mem_size;

    short s_temp[10];
    int i_temp[10];
    float f_temp[10];
    double d_temp[10];
    char str_temp[80];
    char c_temp[10];
    
    float *raw_data, *raw_noise;
    float *raw_spec;
    float *false_data;

    int shmid;

    /*
      All the images has an initial BITPIX.
      It will be stored in this initial BITPIX,
      but the data will be internally managed as DOUBLE
      not to reduce the precision on none of them.
    */
    short bitpix; // Variable that stores the initial BITPIX value.
    char euro3d;
    
    /* Euro3D variables */
    int specId, i,j,k, npix, nb_spec, status, *nol, nfib;
    double start, step;
    short sval;
    long lval;
    float fval, *x,*y;
    double dval;
    char *format;
    /* General variables */
    char *input_filename, output_filename[80], author[80], origin[80];
    char *commands,**tokens;
    int n_tokens;
    char ref[80],date[80],date_obs[80],object[80],observer[80],telescope[80];
    float equinox;
    int error=0;
    float min,max,mean,sigma,min_i,max_i;
    int naxes[2],daxes[2];
    int npix_int;
    double start_w, end_w, delta_w;
    
    /* PGPLOT needed varibles */
    char answer_now;
    char answer[10];    
    int answer_len = sizeof(answer);
    int real_x,real_y,old_x,old_y;
    
    
    //    FILE *fichero;



    address = (int *)malloc(15*sizeof(int));  	
    /*
     * We load the N_SPAXELS in Memory
     */
    //



    //printf("\nWe store in the shared memory the FLAGs\n");

      
    mem_key=3;
    n_bytes=15;
    flag = (int *)malloc(n_bytes*sizeof(int));  
    for (j=0;j<15;j++) {
      flag[j]=0;
    }
    
    
    if (n_bytes<MAX_SHM) {
      load_mem_int(flag,mem_key,n_bytes,&tmp_address);
	//printf("Ok\n");
    } else {
      //printf("*******************************\n");
      //printf("** TOO MUCH LARGE FOR THE SHM**\n");
      //printf("*******************************\n");

    }
    address[mem_key]=tmp_address;
    
    //    printf("Waiting for the Flag to give a signal\n");
    temp_flag=-666;

    /*
     * We set some values not to crash
     */
    while (flag[0]!=666) {
      read_mem_int(&flag,mem_key,n_bytes);

      /*
       * 
       *
       */


      /*
       * Ask the server if it is alive
       * FLAG[0]=666 -> COMMAND
       * FLAG[1]=1 temp flag: indicates the client that can read
       *           the client wait until it is set to 1
       */
      if ((flag[0]==666)&&(flag[2]!=-666)) {
	/*
	 * We send the signal to the client
	 */
	flag[1]=999;
	flag[2]=-999;
	load_mem_int(flag,mem_key,n_bytes,&tmp_address);				address[mem_key]=tmp_address;
      }



      /*
       * To I/O only one spectra.
       * FLAG[0]=1 -> COMMAND
       * FLAG[1]=1 temp flag: indicates the client that can read
       *           the client wait until it is set to 1
       * FLAG[2]= Spectra to load in SHM
       */
      if ((flag[0]==1)&&(flag[2]!=-1)) {
	// printf("FLAG[0]=1\n");
	/*
	 * We read the proper information to  be
	 * loaded
	 */
 	raw_spec = (float *)malloc(npix*sizeof(float));
	for (j=0;j<npix;j++) {
	  raw_spec[j]=raw_data[j+flag[2]*npix];
	}
	load_mem_float(raw_spec,2,npix,&tmp_address);
	address[2]=tmp_address;
	// printf("load Spectra %d in SHM \n",flag[2]);
	free(raw_spec);
	/*
	 * We send the signal to the client
	 */
	flag[1]=1;
	flag[2]=-1;
	load_mem_int(flag,mem_key,n_bytes,&tmp_address);				// printf("FLAG[1]=0\n");
	address[mem_key]=tmp_address;
	//	printf("No paso\n");
      }
      /*
       * We load in SHM a series of Spectra
       * it is faster
       * FLAG[0]=2 -> COMMAND
       * FLAG[1]=1 temp flag: indicates the client that can read
       *           the client wait until it is set to 1
       * FLAG[2]= initial spectra to load in SHM
       * FLAG[3]= Last spectra to load in SHM
       */
      
      if ((flag[0]==2)&&(flag[2]!=-2)) {
	// printf("FLAG[0]=1\n");
	/*
	 * We read the proper information to  be
	 * loaded
	 */	


	shmctl(address[2], IPC_RMID, 0);
	raw_spec = (float *)malloc(npix*(flag[3]-flag[2])*sizeof(float));
	for (j=0;j<npix*(flag[3]-flag[2]);j++) {
	  raw_spec[j]=raw_data[j+flag[2]*npix];
	}
	load_mem_float(raw_spec,2,npix*(flag[3]-flag[2]),&tmp_address);
	free(raw_spec);
	address[2]=tmp_address;


	/*
	 * We send the signal to the client
	 */
	flag[1]=1;
	flag[2]=-2;
	load_mem_int(flag,mem_key,n_bytes,&tmp_address);
	address[mem_key]=tmp_address;

      }

      /*
       * This FLAG indicates the server to load a 
       * new E3D_image in the memory
       * and upload all the descriptors to the SHM
       * to be readed by the client.
       *
       * it is faster
       * FLAG[0]=3 -> COMMAND
       * FLAG[1]=1 temp flag: indicates the client that can read
       *           the client wait until it is set to 1
       */      
      if ((flag[0]==3)&&(flag[2]!=-3)) {
	/*
	 * We read the FILE_NAME from the SHM
	 * KEY=4
	 * We assume a maximun length for a file of 80 characters.
	 */
	// printf("Paso Server\n");
	if (read_mem_char(&input_filename,4,80)!=0) {
	  //printf("No file in the SHM to be upload\n");
	  flag[1]=-1;
	  flag[2]=-3;
	} else {
	  // printf("LOADING FILE=");
	  // printf(input_filename);
	  // printf("...\n");
	  load_e3d(input_filename,&e3d_image,&n_spaxels, &spaxels, &index_start, &npts, &raw_data, &raw_noise);
//	  load_e3d(input_filename,&e3d_image,&n_spaxels, &spaxels, &index_start, &npts, raw_data, raw_noise);
	  // printf("He leido el fichero\n");
	  nb_spec=e3d_image.nbspec;  
	  n_groups=e3d_image.ngroups;
	  npix=e3d_image.common_parameters[2];	 	  
	  load_mem_all(e3d_image,n_spaxels,spaxels,index_start,npts,&address);
	  // printf("He Cargado las cosas en memoria\n");
	  min=3000000;
	  max=-3000000;
	  for (j=0;j<npix*nb_spec;j++) {
	    if (min>raw_data[j]) 
	      min=raw_data[j];
	    if (max<raw_data[j]) 
	      max=raw_data[j];
	  }	      	 

	  flag[1]=1;
	  flag[2]=-3;
	}
	  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
	  address[mem_key]=tmp_address;
	  temp_flag=flag[2];

      }
      
      /*
       * FLAG[0]=4 -> CLEAN THE SHM
       * FLAG[1]=1 temp flag: indicates the client that can read
       *           the client wait until it is set to 1       
       *
       */      
      if ((flag[0]==4)&&(flag[2]!=-4)) {
	for (j=1;j<10;j++) {
	  if (!((shmid = shmget(j, 1, 0666)) < 0)) { 
	    shmctl(shmid, IPC_RMID, NULL);
	  }
	}
	flag[1]=1;
	flag[2]=-4;
	//	printf("flag={%d,%d,%d}\n",flag[0],flag[1],flag[3]);
	load_mem_int(flag,mem_key,n_bytes,&tmp_address);    
	address[mem_key]=tmp_address;
      }
    

      /*
       * We load in SHM an SLICE
       * it is faster
       * FLAG[0]=5 -> Get a SLICE at a certain wavelength
       * FLAG[1]=1 temp flag: indicates the client that can read
       *           the client wait until it is set to 1
       * FLAG[2]= The Wavelength index of the CUT
       */
      
      if ((flag[0]==5)&&(flag[2]!=-5)) {
	// printf("FLAG[0]=1\n");
	/*
	 * We read the proper information to  be
	 * loaded
	 */	
	shmctl(address[2], IPC_RMID, 0);
	raw_spec = (float *)malloc(nb_spec*sizeof(float));
	for (j=0;j<nb_spec;j++) {
	  raw_spec[j]=raw_data[flag[2]+j*npix];
	}
	load_mem_float(raw_spec,2,nb_spec,&tmp_address);
	free(raw_spec);
	address[2]=tmp_address;
	/*
	 * We send the signal to the client
	 */
	flag[1]=1;
	flag[2]=-5;
	load_mem_int(flag,mem_key,n_bytes,&tmp_address);
	address[mem_key]=tmp_address;
      }

      /*
       * We load in SHM a series of SLICES
       * it is faster
       * FLAG[0]=6 -> Get a SLICE at a certain wavelength
       * FLAG[1]=1 temp flag: indicates the client that can read
       *           the client wait until it is set to 1
       * FLAG[2]= The Wavelength index of the CUT
       */
      
      if ((flag[0]==6)&&(flag[2]!=-6)) {
	/*
	 * We read the proper information to  be
	 * loaded
	 */	
	shmctl(address[2], IPC_RMID, 0);
	raw_spec = (float *)malloc((flag[3]-flag[2])*nb_spec*sizeof(float));
	for (j=0;j<(flag[3]-flag[2])*nb_spec;j++) {
	  raw_spec[j]=raw_data[flag[2]+j*npix];
	}
	load_mem_float(raw_spec,2,(flag[3]-flag[2])*nb_spec,&tmp_address);
	free(raw_spec);
	address[2]=tmp_address;
	/*
	 * We send the signal to the client
	 */
	flag[1]=1;
	flag[2]=-6;
	load_mem_int(flag,mem_key,n_bytes,&tmp_address);
	address[mem_key]=tmp_address;
      }


      /*
       * We load in SHM a series of SLICES
       * it is faster
       * FLAG[0]=7 -> Get a certain point intensity
       * FLAG[1]=1 temp flag: indicates the client that can read
       *           the client wait until it is set to 1
       * FLAG[2]= The Wavelength index of the CUT.
       * FLAG[3]= The SpectraID.
       */
      
      if ((flag[0]==7)&&(flag[2]!=-7)) {
	/*
	 * We read the proper information to  be
	 */	
	shmctl(address[2], IPC_RMID, 0);
	raw_spec = (float *)malloc(1*sizeof(float));
	raw_spec[0]=raw_data[flag[2]+flag[3]*npix];
	load_mem_float(raw_spec,2,1,&tmp_address);
	free(raw_spec);
	address[2]=tmp_address;
	/*
	 * We send the signal to the client
	 */
	flag[1]=1;
	flag[2]=-7;
	load_mem_int(flag,mem_key,n_bytes,&tmp_address);
	address[mem_key]=tmp_address;
      }

    

      /*************************
       * Painting Flags
       *************************/

      /*
       * Change the Palette
       *
       * it is faster
       * FLAG[0]=100 -> COMMAND
       * FLAG[1]=1 temp flag: indicates the client that can read
       *           the client wait until it is set to 1
       */      
      if ((flag[0]==100)&&(flag[2]!=-100)) {
	/*
	 * We read the FILE_NAME from the SHM
	 * KEY=4
	 * We assume a maximun length for a file of 80 characters.
	 */
	if (read_mem_char(&commands,4,80)!=0) {
	  //printf("No Commands found\n");
	  flag[1]=-1;
	  flag[2]=-100;
	} else {
	  split_text(commands," ",&n_tokens,&tokens);
	  p=atoi(tokens[0]);
	  bright=atof(tokens[1]);
	  contra=atof(tokens[2]);
	  sign=atof(tokens[3]);
	  cpgslct(atoi(tokens[4]));
	  palette(p,sign*contra,bright);
	  flag[1]=1;
	  flag[2]=-100;
	}
	  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
	  address[mem_key]=tmp_address;
	  temp_flag=flag[2];

      }
      


      /*
       * Open a Device
       *
       * it is faster
       * FLAG[0]=101 -> COMMAND
       * FLAG[1]=1 temp flag: indicates the client that can read
       *           the client wait until it is set to 1
       */      
      if ((flag[0]==101)&&(flag[2]!=-101)) {
	//printf("Paso por aqui?\n");
	/*
	 * We read the FILE_NAME from the SHM
	 * KEY=4
	 * We assume a maximun length for a file of 80 characters.
	 */
	if (read_mem_char(&commands,4,80)!=0) {
	  //printf("No Commands found\n");
	  flag[1]=-1;
	  flag[2]=-101;
	} else {
	  //printf("commands='%s'\n",commands);
	  flag[3] = cpgopen(commands);
	  flag[1]=1;
	  flag[2]=-101;
	}
	  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
	  address[mem_key]=tmp_address;
	  temp_flag=flag[2];

      }
      

      /*
       * Plot Staked Spectra
       *
       * it is faster
       * FLAG[0]=102 -> COMMAND
       * FLAG[1]=1 temp flag: indicates the client that can read
       *           the client wait until it is set to 1
       */      
      if ((flag[0]==102)&&(flag[2]!=-102)) {
	/*
	 * We read the FILE_NAME from the SHM
	 * KEY=4
	 * We assume a maximun length for a file of 80 characters.
	 */
	if (read_mem_char(&commands,4,80)!=0) {
	  //printf("No Commands found\n");
	  flag[1]=-1;
	  flag[2]=-102;
	} else {
	  //	  split_text(commands," ",&n_tokens,&tokens);
	  //	  image_plot(raw_data,min,max,npix,nb_spec,atoi(commands),tr_now);
	  flag[1]=1;
	  flag[2]=-102;
	}
	  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
	  address[mem_key]=tmp_address;
	  temp_flag=flag[2];

      }
      





      /*
       * Plot Staked Spectra
       *
       * it is faster
       * FLAG[0]=103 -> COMMAND
       * FLAG[1]=1 temp flag: indicates the client that can read
       *           the client wait until it is set to 1
       */      
      if ((flag[0]==103)&&(flag[2]!=-103)) {
	/*
	 * We read the FILE_NAME from the SHM
	 * KEY=4
	 * We assume a maximun length for a file of 80 characters.
	 */
	if (read_mem_char(&commands,4,80)!=0) {
	  //printf("No Commands found\n");
	  flag[1]=-1;
	  flag[2]=-103;
	} else {
	  //printf("COMMANDS='%s'\n",commands);
	  split_text(commands," ",&n_tokens,&tokens);
	  image_plot(raw_data,min,max,npix,nb_spec,atoi(tokens[0]),tr_now);
	  flag[1]=1;
	  flag[2]=-103;
	}
	  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
	  address[mem_key]=tmp_address;
	  temp_flag=flag[2];

      }
      



      /*********************************************
       * ALL THE COMMANDS SHOULD BE BEFORE THIS!!!
       *********************************************
       * We delete the FLAG SHARED MEMORY
       * and if I try to read then I will have an error!
       */
      mem_size=15*sizeof(float);            
      shmid = shmget(mem_key, mem_size, 0666);
      shmctl(shmid, IPC_RMID, NULL);
      while (!((shmid = shmget(mem_key, mem_size, IPC_EXCL | IPC_CREAT | 0666))<0)) {
	shmctl(shmid, IPC_RMID, NULL);
	sleep(1);
      }


    }

    
    free(raw_data);
    
    /*
     * We clean the SHARED memory
     */
    shmdt(address[1]);
    shmdt(address[2]);
    shmdt(address[3]);
    shmdt(address[4]);
    shmdt(address[5]);
    shmdt(address[6]);
    shmdt(address[7]);
    shmdt(address[8]);
    shmdt(address[9]);
    shmctl(address[1], IPC_RMID, NULL);
    shmctl(address[2], IPC_RMID, 0);
    shmctl(address[3], IPC_RMID, 0);
    shmctl(address[4], IPC_RMID, 0);
    shmctl(address[5], IPC_RMID, 0);
    shmctl(address[6], IPC_RMID, 0);
    shmctl(address[7], IPC_RMID, 0);
    shmctl(address[8], IPC_RMID, 0);
    shmctl(address[9], IPC_RMID, 0);
    exit_session(0);
    return(0);	
    
}




/*
int save_e3d(E3D_file *e3d_frame, char *filename, int npix, double start_w, double delta_w) {  
  create_E3D_frame(&e3d_frame,text,npix,(double)start_w,(double)delta_w,FLOAT,"save",NULL);
  return(0);
}

*/

/*********************************************
 * A basic function that load an E3D file and the
 * basic parameters needed to handle the image
 *********************************************/

int old_load_e3d(char *filename, E3D_file *e3d_frame, int **n_spaxels, SPAXEL ***spaxels, int **index_start, int **npts, float *raw_data_temp, float *raw_noise_temp) {

  E3D_file e3d_temp;
  int *n_spaxels_temp;
  SPAXEL **spaxels_temp;
  int *index_start_temp;
  int *npts_temp;

  int i,j,k;
  int nb_spec, npix, specId;
  int *nol_temp, max_npts;
  float start_w,end_w,delta_w;
  SPECTRUM signal,noise;
  SPECTRUM *spectra_signal_temp,*spectra_noise_temp;

  open_E3D_frame(&e3d_temp,filename,"I");
  nb_spec=e3d_temp.nbspec;
  spaxels_temp = (SPAXEL *) calloc(nb_spec, sizeof(SPAXEL));
  n_spaxels_temp = (int *) calloc(nb_spec, sizeof(int));
  index_start_temp = (int *) calloc(nb_spec, sizeof(int));
  npts_temp = (int *) calloc(nb_spec, sizeof(int));
  nol_temp = (int *) calloc(nb_spec, sizeof(int));

  /*
   * Reading common parameters
   */
  start_w= (float) e3d_temp.common_parameters[0];
  end_w= (float) e3d_temp.common_parameters[1];
  npix= floor(e3d_temp.common_parameters[2]);

  //  get_common_param(&e3d_temp,&npix,&start_w,&end_w);	
  delta_w=(end_w-start_w)/npix;

  // printf("%d,%f,%f\n Ok\n",npix,start_w,end_w);


  raw_data_temp = (float *) calloc(nb_spec*npix, sizeof(float));
  raw_noise_temp = (float *) calloc(nb_spec*npix, sizeof(float));


  /*
  raw_data_temp = (float *) malloc(nb_spec*npix*sizeof(float));
  raw_noise_temp = (float *) malloc(nb_spec*npix*sizeof(float));
  */


  get_spectra_ID(&e3d_temp,nol_temp);

  max_npts=0;

  /*
   * We read the spectra one by one
   */
  for (specId=1;specId<=nb_spec;specId++) {	 
    /*
      We should read the 'index_start' when available
      in the new format.
      By now it is set to 0.
    */
    index_start_temp[specId-1]=0;
    /*
      We get the spaxels information!!!
    */
    n_spaxels_temp[specId-1]=get_E3D_spaxels(&e3d_temp,nol_temp[specId-1],&spaxels_temp[specId-1]);    
    /*
      We get the spectra information.
    */
    get_E3D_spec(&e3d_temp,&signal,NULL,nol_temp[specId-1]); 
    npts_temp[specId-1]=signal.npts;

    if (signal.npts>max_npts) {
      max_npts=signal.npts;
    }
  }



    printf("Ok***\n");
  /*
   * We build the RAW_DATA image
   */

  for (specId=1;specId<=nb_spec;specId++) {	 
      get_E3D_spec(&e3d_temp,&signal,NULL,nol_temp[specId-1]); 
    for (j=0;j<max_npts;j++) {
      if (j<index_start_temp[specId-1]) {
	raw_data_temp[j+(specId-1)*npix]=0.0;
	raw_noise_temp[j+(specId-1)*npix]=0.0;
      } else {
	raw_data_temp[j+(specId-1)*npix]=RD_spec(&signal,j-index_start_temp[specId]-1);
	raw_noise_temp[j+(specId-1)*npix]=sqrt(fabs(raw_data_temp[j+(specId-1)*npix]));

      }
    }		
  }



  *e3d_frame=e3d_temp;  
  *n_spaxels=n_spaxels_temp;
  *spaxels=spaxels_temp;
  *index_start=index_start_temp;
  *npts=npts_temp;

  close_E3D_frame(&e3d_temp);
  return(0);
}


int load_e3d(char *filename, E3D_file *e3d_frame, int **n_spaxels, SPAXEL ***spaxels, int **index_start, int **npts, float **raw_data, float **raw_noise) {

  E3D_file e3d_temp;
  int *n_spaxels_temp;
  SPAXEL **spaxels_temp;
  int *index_start_temp;
  int *npts_temp;
  float *raw_data_temp;
  float *raw_noise_temp;


  int i,j,k;
  int nb_spec, npix, specId;
  int *nol_temp, max_npts;
  float start_w,end_w,delta_w;
  SPECTRUM signal,noise;
  SPECTRUM *spectra_signal_temp,*spectra_noise_temp;

  open_E3D_frame(&e3d_temp,filename,"I");
  nb_spec=e3d_temp.nbspec;
  spaxels_temp = (SPAXEL *) calloc(nb_spec, sizeof(SPAXEL));
  n_spaxels_temp = (int *) calloc(nb_spec, sizeof(int));
  index_start_temp = (int *) calloc(nb_spec, sizeof(int));
  npts_temp = (int *) calloc(nb_spec, sizeof(int));
  nol_temp = (int *) calloc(nb_spec, sizeof(int));

  /*
   * Reading common parameters
   */
  start_w= (float) e3d_temp.common_parameters[0];
  end_w= (float) e3d_temp.common_parameters[1];
  npix= floor(e3d_temp.common_parameters[2]);

  //  get_common_param(&*e3d_frame,&npix,&start_w,&end_w);	
  delta_w=(end_w-start_w)/npix;


  // printf("%d,%f,%f\n Ok\n",npix,start_w,end_w);

  raw_data_temp = (float *) calloc(nb_spec*npix, sizeof(float));
  raw_noise_temp = (float *) calloc(nb_spec*npix, sizeof(float));
  get_spectra_ID(&e3d_temp,nol_temp);

  max_npts=0;

  /*
   * We read the spectra one by one
   */
  for (specId=1;specId<=nb_spec;specId++) {	 
    /*
      We should read the 'index_start' when available
      in the new format.
      By now it is set to 0.
    */
    index_start_temp[specId-1]=0;
    /*
      We get the spaxels information!!!
    */
    n_spaxels_temp[specId-1]=get_E3D_spaxels(&e3d_temp,nol_temp[specId-1],&spaxels_temp[specId-1]);    
    /*
      We get the spectra information.
    */
    get_E3D_spec(&e3d_temp,&signal,NULL,nol_temp[specId-1]); 
    npts_temp[specId-1]=signal.npts;

    if (signal.npts>max_npts) {
      max_npts=signal.npts;
    }
  }



  // printf("Ok***\n");
  /*
   * We build the RAW_DATA image
   */

  for (specId=1;specId<=nb_spec;specId++) {	 
      get_E3D_spec(&e3d_temp,&signal,NULL,nol_temp[specId-1]); 
    for (j=0;j<max_npts;j++) {
      //printf("specId=%d,j=%d\n",specId,j);
      if (j<index_start_temp[specId-1]) {

	raw_data_temp[j+(specId-1)*npix]=0.0;
	raw_noise_temp[j+(specId-1)*npix]=0.0;
      } else {
//	raw_data_temp[j+(specId-1)*npix]=RD_spec(&signal,j-index_start_temp[specId]-1);
// **** SHIFT by 1 pixel 18.04.04 ****
	raw_data_temp[j+(specId-1)*npix]=RD_spec(&signal,j-index_start_temp[specId]);
	raw_noise_temp[j+(specId-1)*npix]=sqrt(fabs(raw_data_temp[j+(specId-1)*npix]));

      }
    }		
  }
  

  //  free(signal);

  *e3d_frame=e3d_temp;  
  *n_spaxels=n_spaxels_temp;
  *spaxels=spaxels_temp;
  *index_start=index_start_temp;
  *npts=npts_temp;
  *raw_data=raw_data_temp;
  *raw_noise=raw_data_temp;
  close_E3D_frame(&e3d_temp);
  return(0);
}

/************************
 * We read the Euro3D file, without saving the 
 * complete file.
 ************************/

int load_e3d_groups(char *filename, int *n_groups, GROUP **groups, float *common_parameters, int *nb_spec_out, int **n_spaxels, SPAXEL ***spaxels, int **index_start, int **npts, float **raw_data, float **raw_noise) {

  E3D_file e3d_temp;
  int *n_spaxels_temp;
  SPAXEL **spaxels_temp;
  int *index_start_temp;
  int *npts_temp;
  float *raw_data_temp;
  float *raw_noise_temp;
  int n_groups_temp;

  int i,j,k;
  int nb_spec, npix, specId;
  int *nol_temp, max_npts;
  float start_w,end_w,delta_w;
  SPECTRUM signal,noise;
  SPECTRUM *spectra_signal_temp,*spectra_noise_temp;

  open_E3D_frame(&e3d_temp,filename,"I");
  nb_spec=e3d_temp.nbspec;
  spaxels_temp = (SPAXEL *) calloc(nb_spec, sizeof(SPAXEL));
  n_spaxels_temp = (int *) calloc(nb_spec, sizeof(int));
  index_start_temp = (int *) calloc(nb_spec, sizeof(int));
  npts_temp = (int *) calloc(nb_spec, sizeof(int));
  nol_temp = (int *) calloc(nb_spec, sizeof(int));
  n_groups_temp=e3d_temp.ngroups;
  *groups=e3d_temp.groups;

  /*
   * Reading common parameters
   */
  for (i=0;i<3;i++) {
      common_parameters[i]= (float) e3d_temp.common_parameters[i];
  }
  start_w= (float) e3d_temp.common_parameters[0];
  end_w= (float) e3d_temp.common_parameters[1];
  npix= floor(e3d_temp.common_parameters[2]);

  //  get_common_param(&e3d_temp,&npix,&start_w,&end_w);	
  delta_w=(end_w-start_w)/npix;

  // printf("%d,%f,%f\n Ok\n",npix,start_w,end_w);

  raw_data_temp = (float *) calloc(nb_spec*npix,sizeof(float));
  raw_noise_temp = (float *) calloc(nb_spec*npix, sizeof(float));




  get_spectra_ID(&e3d_temp,nol_temp);

  max_npts=0;

  /*
   * We read the spectra one by one
   */
  for (specId=1;specId<=nb_spec;specId++) {	 
    /*
      We should read the 'index_start' when available
      in the new format.
      By now it is set to 0.
    */
    index_start_temp[specId-1]=0;
    /*
      We get the spaxels information!!!
    */
    n_spaxels_temp[specId-1]=get_E3D_spaxels(&e3d_temp,nol_temp[specId-1],&spaxels_temp[specId-1]);    
    /*
      We get the spectra information.
    */
    get_E3D_spec(&e3d_temp,&signal,NULL,nol_temp[specId-1]); 
    npts_temp[specId-1]=signal.npts;

    if (signal.npts>max_npts) {
      max_npts=signal.npts;
    }
  }



  //  printf("Ok***\n");
  /*
   * We build the RAW_DATA image
   */

  for (specId=1;specId<=nb_spec;specId++) {	 
      get_E3D_spec(&e3d_temp,&signal,NULL,nol_temp[specId-1]); 
    for (j=0;j<max_npts;j++) {
      if (j<index_start_temp[specId-1]) {
	raw_data_temp[j+(specId-1)*npix]=0.0;
	raw_noise_temp[j+(specId-1)*npix]=0.0;
      } else {
// **** SHIFT by 1 pixel 18.04.04 ****
//	raw_data_temp[j+(specId-1)*npix]=RD_spec(&signal,j-index_start_temp[specId]-1);
	raw_data_temp[j+(specId-1)*npix]=RD_spec(&signal,j-index_start_temp[specId]);
	raw_noise_temp[j+(specId-1)*npix]=sqrt(fabs(raw_data_temp[j+(specId-1)*npix]));
		
      }
    }		
  }


  *nb_spec_out=nb_spec;
  *n_groups=n_groups_temp;
  *n_spaxels=n_spaxels_temp;
  *spaxels=spaxels_temp;
  *index_start=index_start_temp;
  *npts=npts_temp;
  *raw_data=raw_data_temp;
  *raw_noise=raw_data_temp;
  close_E3D_frame(&e3d_temp);
  return(0);
}


/*****************************
 * We read only the common information
 *****************************/

int load_e3d_only_groups(char *filename, int *n_groups, GROUP **groups, float *common_parameters, int *nb_spec_out) {
  E3D_file e3d_temp;
  int n_groups_temp;
  int i,j,k;
  int nb_spec, npix, specId;
  int *nol_temp;
  float start_w,end_w,delta_w;
  open_E3D_frame(&e3d_temp,filename,"I");
  nb_spec=e3d_temp.nbspec;
  *n_groups=e3d_temp.ngroups;
  *groups=e3d_temp.groups;
  if (has_common_bounds(&e3d_temp)) {
  	for (i=0;i<3;i++) {
      		common_parameters[i]= (float) e3d_temp.common_parameters[i];
  	}
  } else {
  	npix=0;
	start_w = +10e8;
	end_w = -10e8;
  	for (i=0;i<nb_spec;i++) {	 
    		if (e3d_temp.signal[i].npix > npix)
      			npix=e3d_temp.signal[i].npix;
    		if (e3d_temp.signal[i].start < start_w)
      			start_w=e3d_temp.signal[i].start;
    		if (e3d_temp.signal[i].end > end_w)
      			end_w=e3d_temp.signal[i].end;
	}
  	common_parameters[0] = (double)start_w;
  	common_parameters[1] = (double)end_w;
  	common_parameters[2] = (double)npix;
  }
  delta_w=e3d_temp.step;
  // printf("PARAM=%f %f %f %d %d %d\n",start_w,end_w,delta_w,nb_spec,e3d_temp.nbspec,npix);
  *nb_spec_out=nb_spec;
  //*n_groups=n_groups_temp;
  close_E3D_frame(&e3d_temp);
  return(0);
}


/*
 * We only read the data...
 */

int load_e3d_only_data(char *filename, int nb_spec, int npix, int *n_spaxels_temp, SPAXEL ***spaxels, int *index_start_temp, int *npts_temp, float *raw_data_temp, float *raw_noise_temp) {
  //, SPECTRUM signal_t, SPECTRUM noise_t) {

  E3D_file e3d_temp;
  SPAXEL **spaxels_temp;
//  int *index_start_temp;
  SPECTRUM signal,noise;

  int i,j,k;
  int specId;
  int *nol_temp, max_npts;
  float start_w;

  open_E3D_frame(&e3d_temp,filename,"I");
  nol_temp = (int *) calloc(nb_spec, sizeof(int));
  get_spectra_ID(&e3d_temp,nol_temp);
  max_npts=0;

  if (has_common_bounds(&e3d_temp)) {
	max_npts = e3d_temp.signal[0].npix;
	start_w = e3d_temp.signal[0].start;
  } else {
	start_w = +10e8;
  	for (i=0;i<nb_spec;i++) {	 
    		if (e3d_temp.signal[i].start < start_w)
      			start_w=e3d_temp.signal[i].start;
    		if (e3d_temp.signal[i].npix > max_npts)
      			max_npts=e3d_temp.signal[i].npix;
	}
  }

  spaxels_temp=*spaxels;
  /*
   * We read the spectra one by one
   */
  for (specId=1;specId<=nb_spec;specId++) {	 

    /*
      We get the spaxels information!!!
    */
    n_spaxels_temp[specId-1]=get_E3D_spaxels(&e3d_temp,nol_temp[specId-1],&spaxels_temp[specId-1]);    
  }

  /*
   * We build the RAW_DATA image
   */

  for (specId=1;specId<=nb_spec;specId++) {	 
    get_E3D_spec(&e3d_temp,&signal,NULL,nol_temp[specId-1]); 
    index_start_temp[specId-1] = (signal.start - start_w)/signal.step;
    for (j=0;j<signal.npts;j++) {
      if (j<index_start_temp[specId-1]) {
	raw_data_temp[j+(specId-1)*npix]=0.0;
	raw_noise_temp[j+(specId-1)*npix]=0.0;
      } else {

//	raw_data_temp[j+(specId-1)*npix]=RD_spec(&signal,j-index_start_temp[specId]-1);
// **** SHIFT by 1 pixel 18.04.04 ****
	raw_data_temp[j+(specId-1)*npix] = RD_spec(&signal,j-index_start_temp[specId-1]);
 	raw_noise_temp[j+(specId-1)*npix]=sqrt(fabs(raw_data_temp[j+(specId-1)*npix]));
      }

    }	
    for (;j<max_npts;j++) {
	raw_data_temp[j+(specId-1)*npix]=0.0;
	raw_noise_temp[j+(specId-1)*npix]=0.0;
    }

    /*
     * Trying to clean signal!!!
     */
    //
    free_spec_mem(&signal);
  }

  free(nol_temp);
  close_E3D_frame(&e3d_temp);
  return(0);
}



/********************************
 * Program that reads an Slice of data
 * as a float array.
 * The array contain from 0, NB_SPEC-1 data
 * containing a Wavelength cut.
 ********************************/
float* read_spax_int(int i_spec, int nb_spec, int npix) {    
  int key=2; // We read from the RAW_data
  int mem_size;
  int shmid;
  int j;
  float *shm,*spax_int;
  int size;
  size=nb_spec*npix;
  mem_size=size*sizeof(float);

    /*
     * Locate the segment.
     */
    if ((shmid = shmget(key, mem_size, 0666)) < 0) {
        perror("shmget");
        return(1);
    }
    //printf("MEM=%d\n",shmid);
    /*
     * Now we attach the segment to our data space.
     */
    
    if ((shm = shmat(shmid, NULL, 0)) == (float *) -1) {
        perror("shmat");
        return(1);
    }

    spax_int = (float *)malloc(nb_spec*sizeof(float));    
    if (i_spec<0) i_spec=0;
    if (i_spec>=npix) i_spec=npix-1;

    for (j=0;j<nb_spec;j++) {
      spax_int[j]=shm[i_spec+j*npix];
    }

    return spax_int;
}







/********************************
 * This function reads the groups from
 * the SHM
 ********************************/
//GROUP* read_mem_group(int key, int size) {
int read_mem_group(GROUP **group,int key, int size) {
    
  int mem_size;
  int shmid;
  int j;
  GROUP* shm;
  
  mem_size=size*sizeof(GROUP);

    /*
     * Locate the segment.
     */
    if ((shmid = shmget(key, mem_size, 0666)) < 0) {
        perror("shmget");
        return(1);
    }
    //printf("MEM=%d\n",shmid);
    /*
     * Now we attach the segment to our data space.
     */
    
    if ((shm = shmat(shmid, NULL, 0)) == (GROUP *) -1) {
        perror("shmat");
        return(1);
    }
    //    return shm;
    *group=shm;
    return(0);
}


/********************************
 * Function that load a GROUP array to the memory
 ********************************/

int load_mem_group(GROUP *grupo, int key, int size, int *address) {

  int mem_size;
    int shmid;
    GROUP *shm,*s;
    int j;

    mem_size=size*sizeof(GROUP);
    /*
     * Creating the Segment
     */
    if ((shmid = shmget(key, mem_size, IPC_CREAT | 0666)) < 0) {
        perror("shmget");
        return(1);
    }

    *address=shmid;
    //printf("MEM=%d, key=%d\n",mem_size,key); 
    /*
     * Now we attach the segment to our data space.
     */
    if ((shm = shmat(shmid, NULL, 0)) == (GROUP *) -1) {
        perror("shmat");
        return(1);
    }

    s=shm;

    for (j=0;j<size;j++) {
      s[j]=grupo[j];
    }

    return(0);
}



/********************************
 * Function that read a INT array from the memory.
 * Used to write in memory the 2D images of staked
 * spectra for the QUALITY
 * or the N_SPAXELS.
 ********************************/
int read_mem_int(int **out_int, int key, int size) {
    
  int mem_size;
  int shmid;
  int j;
  int* shm;
  
  mem_size=size*sizeof(int);

    /*
     * Locate the segment.
     */
    if ((shmid = shmget(key, mem_size, 0666)) < 0) {
      perror("shmget");
        return(1);
    }
    //printf("MEM=%d\n",shmid);
    /*
     * Now we attach the segment to our data space.
     */
    
    if ((shm = shmat(shmid, NULL, 0)) == (int *) -1) {
              perror("shmat");
        return(2);
    }
    //    return shm;
    *out_int=shm;
    return(0);
}


/********************************
 * Function that load a int array to the memory
 ********************************/

int load_mem_int(int *raw_data, int key, int size, int* address) {

  int mem_size;
    int shmid;
    int *shm,*s;
    int j;

    mem_size=size*sizeof(int);
    /*
     * Creating the Segment
     */
    if ((shmid = shmget(key, mem_size, IPC_CREAT | 0666)) < 0) {
        perror("shmget");
        return(1);
    }
    *address=shmid;
    //printf("MEM=%d, key=%d\n",mem_size,key); 
    /*
     * Now we attach the segment to our data space.
     */
    if ((shm = shmat(shmid, NULL, 0)) == (int *) -1) {
        perror("shmat");
        return(1);
    }

    /*
     * Now put some things into the memory for the
     * other process to read.
     */
    //    shm=raw_data;

    //    raw_data=shm;
    s=shm;

    for (j=0;j<size;j++) {
      s[j]=raw_data[j];
    }

    return(0);
}



/********************************
 * Function that read a CHAR array from the memory.
 * Used to write in memory the 2D images of staked
 * spectra for the QUALITY
 * or the N_SPAXELS.
 ********************************/
int  read_mem_char(char **out_shm,int key, int size) {
    
  int mem_size;
  int shmid;
  int j;
  char* shm;
  
  mem_size=size*sizeof(char);

    /*
     * Locate the segment.
     */
    if ((shmid = shmget(key, mem_size, 0666)) < 0) {
        perror("shmget");
        return(1);
    }
    //printf("MEM=%d, key=%d\n",mem_size,key); 
    /*
     * Now we attach the segment to our data space.
     */
    
    if ((shm = shmat(shmid, NULL, 0)) == (char *) -1) {
        perror("shmat");
        return(1);
    }
    *out_shm=shm;
    return(0);
}


/********************************
 * Function that load a char array to the memory
 ********************************/

char load_mem_char(char *raw_data, int key, int size, int* address) {

  int mem_size;
  int shmid;
  char *shm,*s;
  int j;
  
  mem_size=size*sizeof(char);
  /*
   * Creating the Segment
   */

  // printf("Paso %s\n",raw_data);
  if ((shmid = shmget(key, mem_size, IPC_CREAT | 0666)) < 0) {
    perror("shmget");
    return(1);
  }
  *address=shmid;
  //printf("MEM=%d, key=%d\n",mem_size,key); 
  /*
   * Now we attach the segment to our data space.
   */
  if ((shm = shmat(shmid, NULL, 0)) == (char *) -1) {
    perror("shmat");
        return(1);
  }
  
  /*
   * Now put some things into the memory for the
   * other process to read.
   */
  //    shm=raw_data;
  
  //    raw_data=shm;
  s=shm;
  
  for (j=0;j<size;j++) {
    s[j]=raw_data[j];
  }
  
  // printf("FILE='%s','%s'\n",raw_data,s);

  return(0);
}




/********************************
 * Function that read an array of Spaxels.
 ********************************/
//SPAXEL** read_mem_spaxels(int* n_spaxels,int key, int size) {
    
int read_mem_spaxels(SPAXEL ***out_spax, int* n_spaxels, int key, int size) {

  int mem_size,real_size;
  int shmid;
  int j,i,k;
  SPAXEL *shm,**out_spaxels;

  
  real_size=1;
  for (j=0;j<size;j++) {
    for (i=0;i<n_spaxels[j-1];i++) {
      real_size++;
    }
  }
  

  
  mem_size=real_size*sizeof(SPAXEL);

    if ((shmid = shmget(key, mem_size, 0666)) < 0) {
        perror("shmget");
        return(1);
    }
    //printf("MEM=%d, key=%d\n",mem_size,key); 
    
    if ((shm = shmat(shmid, NULL, 0)) == (SPAXEL *) -1) {
        perror("shmat");
        return(1);
    }

    //    printf("size=%d,real_size=%d\n",size,real_size);

    out_spaxels = (SPAXEL **)malloc(size*sizeof(SPAXEL*));    
    k=0;
    for (j=0;j<size;j++) {
      out_spaxels[j] = (SPAXEL *)malloc(n_spaxels[j]*sizeof(SPAXEL));
      for (i=0;i<n_spaxels[j];i++) {
	out_spaxels[j][i]=shm[k];
	k++;
      }
    }
    
  
    //    return out_spaxels;
    *out_spax=out_spaxels;
    return(0);
}



/****************************************************
 * This function is used to load in memory
 * the SPAXELS.
 ****************************************************/

int load_mem_spaxels(SPAXEL **spaxels, int* n_spaxels, int key, int size, int* address) {
      
  int mem_size,real_size;
  int shmid;
  int i,j,k;
  SPAXEL *shm,*s;
  
  real_size=1;
  for (j=0;j<size;j++) {
    for (i=0;i<n_spaxels[j];i++) {
      real_size++;
    }
  }


  mem_size=real_size*sizeof(SPAXEL);
  
  /*
   * Creating the Segment
   */
  
  if ((shmid = shmget(key, mem_size, IPC_CREAT | 0666)) < 0) {
    perror("shmget");
    return(1);
  }
  
  *address=shmid;
    //printf("MEM=%d, key=%d\n",mem_size,key); 
  /*
   * Now we attach the segment to our data space.
   */
  
  if ((shm = shmat(shmid, NULL, 0)) == (SPAXEL *) -1) {
    perror("shmat");
    return(1);
  }
  
  
  /*
   * Now put some things into the memory for the
   * other process to read.
   */
  
  s=shm;
  
  k=0;

  //  s = (SPAXEL *)malloc(real_size*sizeof(SPAXEL));

  for (j=0;j<size;j++) {
    for (i=0;i<n_spaxels[j];i++) {
      s[k]=spaxels[j][i];
      k++;
    }
  }
  
  
  return(0);

}

/********************************
 * Function that read a float array from the memory.
 * Used to write in memory the 2D images of staked
 * spectra, both the SIGNAL, QUALITY, and NOISE
 ********************************/
//float* read_mem_float(int key, int size) {
int read_mem_float(float **out_float, int key, int size) {
    
  int mem_size;
  int shmid;
  int j;
  float* shm;
  
  mem_size=size*sizeof(float);

    /*
     * Locate the segment.
     */
    if ((shmid = shmget(key, mem_size, 0666)) < 0) {
        perror("shmget");
        return(1);
    }
    //printf("MEM=%d, key=%d\n",mem_size,key); 
    /*
     * Now we attach the segment to our data space.
     */
    
    if ((shm = shmat(shmid, NULL, 0)) == (float *) -1) {
        perror("shmat");
        return(1);
    }
    *out_float=shm;
    return(0);
}


/********************************
 * Function that load a float array to the memory
 ********************************/

int load_mem_float(float *raw_data, int key, int size, int* address) {

  int mem_size;
    int shmid;
    float *shm,*s;
    int j;

    mem_size=size*sizeof(float);
    /*
     * Creating the Segment
     */
    if ((shmid = shmget(key, mem_size, IPC_CREAT | 0666)) < 0) {
        perror("shmget");
        return(1);
    }

    *address=shmid;
    //printf("MEM=%d, key=%d\n",mem_size,key); 
    /*
     * Now we attach the segment to our data space.
     */
    if ((shm = shmat(shmid, NULL, 0)) == (float *) -1) {
        perror("shmat");
        return(1);
    }

    /*
     * Now put some things into the memory for the
     * other process to read.
     */
    //    shm=raw_data;

    //    raw_data=shm;
    s=shm;

    for (j=0;j<size;j++) {
      s[j]=raw_data[j];
    }

    return(0);
}



/********************************
 * Function that read an 2D image from memory
 ********************************/
float* read_mem_staked_spectra(int key, int size) {
    
  int mem_size;
  int shmid;
  int j;
  float* shm;
  
  mem_size=size*sizeof(float);
    /*
     * Locate the segment.
     */
    if ((shmid = shmget(key, mem_size, 0666)) < 0) {
        perror("shmget");
        return(1);
    }
    //printf("MEM=%d, key=%d\n",mem_size,key); 
    /*
     * Now we attach the segment to our data space.
     */
    
    if ((shm = shmat(shmid, NULL, 0)) == (float *) -1) {
        perror("shmat");
        return(1);
    }
    return shm;
}


/********************************
 * Function that load an E3D_file into memory
 ********************************/

int load_mem_staked_spectra(float *raw_data, int key, int size, int* address) {

  int mem_size;
    int shmid;
    float *shm,*s;
    int j;

    mem_size=size*sizeof(float);


    //  printf("mem_size=%d\n",mem_size);

    /*
     * Creating the Segment
     */
  //    if ((shmid = shmget(key, mem_size, IPC_CREAT | 0666)) < 0) {
  if ((shmid = shmget(key, mem_size, IPC_CREAT | 0666)) < 0) {
    perror("shmget");
    return(1);
  }

    *address=shmid;
    //printf("MEM=%d, key=%d\n",mem_size,key); 
    /*
     * Now we attach the segment to our data space.
     */
    if ((shm = shmat(shmid, NULL, 0)) == (float *) -1) {
        perror("shmat");
        return(1);
    }

    /*
     * Now put some things into the memory for the
     * other process to read.
     */
    //    shm=raw_data;

    //    raw_data=shm;
    s=shm;

    for (j=0;j<size;j++) {
      s[j]=raw_data[j];
    }

    return(0);
}



/**************************************************
 * Function that load a 2D image to the Memory
 **************************************************/

int load_mem_e3d(E3D_file e3d_file, int key, int *address) {

    int mem_size=sizeof(E3D_file);
    int shmid;
    E3D_file *shm,*s;
    int j;

    /*
     * Creating the Segment
     */
    if ((shmid = shmget(key, mem_size, IPC_CREAT | 0666)) < 0) {
        perror("shmget");
        return(1);
    }

    *address=shmid;
    //printf("MEM=%d, key=%d\n",mem_size,key); 
    /*
     * Now we attach the segment to our data space.
     */
    if ((shm = shmat(shmid, NULL, 0)) == (char *)-1) {
        perror("shmat");
        return(1);
    }

    /*
     * Now put some things into the memory for the
     * other process to read.
     */
    s=shm;
    *s=e3d_file;
    return(0);
}

/********************************
 * Function that read an E3D_file from memory
 ********************************/

int read_mem_e3d(E3D_file* e3d_out, int key) {
    
    int mem_size=sizeof(E3D_file);
    int shmid;
    E3D_file *shm;

    int j;

    if ((shmid = shmget(key, mem_size, 0666)) < 0) {
        perror("shmget");
        return(1);
    }
    //printf("MEM=%d, key=%d\n",mem_size,key); 
    
    if ((shm = shmat(shmid, NULL, SHM_RDONLY)) == (char *) -1) {
        perror("shmat");
        return(1);
    }
    
    *e3d_out=*shm;

    return(0);
}

/******************************************
 * Function to Change the palett
 ******************************************/

int palette(int type, float contra, float bright){
  
  //  printf("type=%d, %f,%f\n",type,contra,bright);

  float GL[]={0.0, 1.0};
  float GR[]={0.0, 1.0};
  float GG[]={0.0, 1.0};
  float GB[]={0.0, 1.0};
  
  float RL[]={-0.5, 0.0, 0.17, 0.33, 0.50, 0.67, 0.83, 1.0, 1.7};
  float RR[]={ 0.0, 0.0,  0.0,  0.0,  0.6,  1.0,  1.0, 1.0, 1.0};
  float RG[]={ 0.0, 0.0,  0.0,  1.0,  1.0,  1.0,  0.6, 0.0, 1.0};
  float RB[]={ 0.0, 0.3,  0.8,  1.0,  0.3,  0.0,  0.0, 0.0, 1.0};
  
  float HL[]={0.0, 0.2, 0.4, 0.6, 1.0};
  float HR[]={0.0, 0.5, 1.0, 1.0, 1.0};
  float HG[]={0.0, 0.0, 0.5, 1.0, 1.0};
  float HB[]={0.0, 0.0, 0.0, 0.3, 1.0};
  
  float WL[]={0.0, 0.5, 0.5, 0.7, 0.7, 0.85, 0.85, 0.95, 0.95, 1.0};
  float WR[]={0.0, 1.0, 0.0, 0.0, 0.3,  0.8,  0.3,  1.0,  1.0, 1.0};
  float WG[]={0.0, 0.5, 0.4, 1.0, 0.0,  0.0,  0.2,  0.7,  1.0, 1.0};
  float WB[]={0.0, 0.0, 0.0, 0.0, 0.4,  1.0,  0.0,  0.0, 0.95, 1.0};
  
  float AL[]={0.0, 0.1, 0.1, 0.2, 0.2, 0.3, 0.3, 0.4, 0.4, 0.5,
     0.5, 0.6, 0.6, 0.7, 0.7, 0.8, 0.8, 0.9, 0.9, 1.0};
  float AR[]={0.0, 0.0, 0.3, 0.3, 0.5, 0.5, 0.0, 0.0, 0.0, 0.0,
	      0.0, 0.0, 0.0, 0.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0};
  float AG[]={0.0, 0.0, 0.3, 0.3, 0.0, 0.0, 0.0, 0.0, 0.8, 0.8,
	      0.6, 0.6, 1.0, 1.0, 1.0, 1.0, 0.8, 0.8, 0.0, 0.0};
  float AB[]={0.0, 0.0, 0.3, 0.3, 0.7, 0.7, 0.7, 0.7, 0.9, 0.9,
	      0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};

  switch(type) {
  case (1): {
    /* Grey */
    cpgctab(GL, GR, GG, GB, 2, contra, bright);
    break;
  }
  case (2): {
    /* Rainbow */
    cpgctab(RL, RR, RG, RB, 9, contra, bright);
    break;
  }
  case (3): {
    /* Heat */
    cpgctab(HL, HR, HG, HB, 5, contra, bright);
    break;
  }
  case (4): {
    /* Weird IRAF */
    cpgctab(WL, WR, WG, WB, 10, contra, bright);
    break;
  }
  case (5): {
    /* AIPS */
    cpgctab(AL, AR, AG, AB, 20, contra, bright);
    break;
  }
  default: {
    cpgctab(RL, RR, RG, RB, 9, contra, bright);
    //    cpgctab(GL, GR, GG, GB, 2, contra, bright);
    break;
  }
  }

  return(0);  
}

void setVP() {
  float d, vpx1, vpx2, vpy1, vpy2;
  cpgsvp(0.0,1.0,0.0,1.0);
  cpgqvp(1,&vpx1,&vpx2,&vpy1,&vpy2);
  d=Minimun(vpx2-vpx1,vpy2-vpy1)/40.0;
  vpx1=vpx1+5.0*d;
  vpx2=vpx2-2.0*d;
  vpy1=vpy1+8.0*d;
  vpy2=vpy2-2.0*d;
  cpgvsiz(vpx1,vpx2,vpy1,vpy2);
  return;
}

float Minimun(float a, float b) {
  float c;
  if (a<b) {
    c=a;
  } else {
    c=b;
  }
  return c;
}

float Maximun(float a, float b) {
  float c;
  if (a>b) {
    c=a;
  } else {
    c=b;
  }
  return c;
}


/*****************************************
 * Plot Staked Spectra
 *****************************************/
int draw_staked_spectra(float max, float min, int **nnaxes, float **out) {

  int delta_n=512;
  E3D_file e3d_image;
  SPAXEL **spaxels;
  int *n_spaxels,*index_start,*npts;
  GROUP *groups;
  double start_w, end_w, delta_w;
  int npix;
  int nb_spec,n_groups;
  float *raw_data;
  int naxes[2];
  int j;
  float datamin,datamax;
  float tr[]={0.0,1.0,0.0,0.0,0.0,1.0};
  float minmax[2];

  ask_e3d_info(&e3d_image,&n_spaxels,&spaxels,&index_start,&npts,&groups);
  start_w=e3d_image.common_parameters[0];
  end_w=e3d_image.common_parameters[1];
  npix=e3d_image.common_parameters[2];
  delta_w=(end_w-start_w)/npix;
  nb_spec=e3d_image.nbspec;
  n_groups=e3d_image.ngroups;
  raw_data = (float *)malloc(nb_spec*n_groups*sizeof(float));
  ask_raw_data(&raw_data,nb_spec,npix,delta_n);
  naxes[0]=npix;
  naxes[1]=nb_spec;

  //  *nnaxes = (int *)malloc(2*sizeof(int));
  *nnaxes=naxes;


  
  datamin=3000000.0;
  datamax=-3000000.0;
  for (j=0;j<npix*nb_spec;j++) {
    if (datamin>raw_data[j]) 
      datamin=raw_data[j];
    if (datamax<raw_data[j]) 
      datamax=raw_data[j];
  }	      	 
  datamax=2.5*datamax;
  datamin=0.5*datamin;
  if (min!=0.0) datamin=min;
  if (max!=0.0) datamax=max;
  
  minmax[0]=datamin;
  minmax[1]=datamax;
  //  printf("MINMAX=%f,%f\n",datamin,datamax);

  *out=minmax;

  

  //  cpgslct(device_id);
  cpgpage();
  cpgvstd();
  cpgwnad(0.0,1.0+naxes[0],0.0,1.0+naxes[1]);
  cpgimag(raw_data,naxes[0],naxes[1],1,naxes[0],1,naxes[1],datamin,datamax,tr);
  cpgsch(0.6);
  cpgbox("bcntsi",0.0,0,"bcntsiv",0.0,0);
  cpgmtxt("b",3.0,1.0,1.0,"pixel number");
  cpgwedg("BI",4.0,5.0,datamin,datamax,"pixel value");
  cpgsch(1.0);
  free(raw_data);
  return(0);
}



/******************************************
 * Function to Plot an Image
 ******************************************/

int image_plot(float *data, float datamin, float datamax, int n_x, int n_y, int dev_id, float *tr) {
  //  float tr[]={0.0,1.0,0.0,0.0,0.0,1.0};
  //  cpgslct(dev_id);
  //  cpgpage();
  cpgvstd();
  //  cpgwnad(0.0,1.0+n_x,0.0,1.0+n_y);
  cpgsch(1.1);
  cpgenv(0.0,1.0+n_x,0.0,1.0+n_y,-1,0);
  cpgimag(data,n_x,n_y,1,n_x,1,n_y,datamin,datamax,tr);
  cpgsch(0.6);
  //  cpgbox("bcntsi",0.0,0,"bcntsiv",0.0,0); 
  cpgbox("bc",0.0,0,"bc",0.0,0); 
  cpgmtxt("b",3.0,1.0,1.0,"pixel number"); 
  cpgwedg("BI",4.0,5.0,datamin,datamax,"pixel value");
  cpgsch(1.0);
  return(0);
}



/******************************************
 * Function to plot spaxels
 ******************************************/
//void plot_spaxel(int color_index, float x_c, float y_c, GROUP group, int fill, int text_flag) {
void plot_spaxel(int color_index, float x_c, float y_c, GROUP group, int fill, int text_flag) {

  int j,i;
  float angle;
  float size1,size2;
  char shape;


  float xp[6],yp[6];
  float x[6],y[6];
  

  /*
   * OLD
   *  size1=(float)group.size1/100.0;
   * size2=(float)group.size2/100.0;
   */

  size1=(float)group.size1;
  size2=(float)group.size2;

  size1=1.05*size1;
  size2=1.05*size2;

  if (size1==0.0) size1=0.30;

  angle=((float)group.angle)/180*3.141592654;

  //  printf("%d,%f, %f, %f,%f\n",color_index,x_c,y_c,size1,size2);

  cpgsci(color_index);
  if (fill<1) fill=1;
  if (fill>4) fill=4;

  cpgsfs(fill);
  //  shape=toupper(group.shape);
  //printf("%s %s\n",shape,group.shape);

  /*
  if (shape==1075052544) {
    shape=83;
  }
  
  */

  switch(group.shape) {
  case('S'): {
      if (angle==0.0) {
	cpgrect(x_c-size1/2,x_c+size1/2,y_c-size1/2,y_c+size1/2);
      } else {
	xp[0]=-size1; yp[0]=-size1;
	xp[1]=size1; yp[1]=-size1;
	xp[2]=size1; yp[2]=size1;
    	xp[3]=-size1; yp[3]=size1;
	for (i=0;i<4;i++) {
	  x[i]=x_c+xp[i]*cos(angle)-yp[i]*sin(angle);
	  y[i]=y_c+xp[i]*sin(angle)+yp[i]*cos(angle);
	}
	cpgpoly(4,x,y);
      }
      break;
    }
    case('R'): {      
      if (angle==0.0) {
	cpgrect(x_c-size1/2,x_c+size1/2,y_c-size2/2,y_c+size2/2);
      } else {
	xp[0]=-size1; yp[0]=-size2;
	xp[1]=size1; yp[1]=-size2;
	xp[2]=size1; yp[2]=size2;
	xp[3]=-size1; yp[3]=size2;
	for (i=0;i<4;i++) {
	  x[i]=x_c+xp[i]*cos(angle)-yp[i]*sin(angle);
	  y[i]=y_c+xp[i]*sin(angle)+yp[i]*cos(angle);
	}
	cpgpoly(4,x,y);
      }
      break;
    }

    case('H'): {      
      for (i=0;i<6;i++) {
	xp[i]=-size1*cos(((60.0*i)/180)*3.141592654); 
	yp[i]=size1*sin(((60.0*i)/180)*3.141592654);
	//	printf("%d %f %f\n",i,xp[i],yp[i]);
      }
      for (i=0;i<6;i++) {
	x[i]=x_c+xp[i]*cos(angle)-yp[i]*sin(angle);
	y[i]=y_c+xp[i]*sin(angle)+yp[i]*cos(angle);
      }
      cpgpoly(6,x,y);
      break;
    }

    case('C'): {
      cpgcirc(x_c,y_c,size1);
      break;
    }
    default: {
      //      cpgrect(x_c-size1/2,x_c+size1/2,y_c-size1/2,y_c+size1/2);
      //      cpgcirc(x_c,y_c,0.15);
      cpgcirc(x_c,y_c,size1/2);
      break;
    }
    }

  
  //  return(0);
  

}


/*********************************************
 * Utility to split a text in different tokens
 *********************************************/

int split_text(char *text, char *splitter, int *n_words, char ***words) {
  int j;
  int n_w;
  //  char *w[1000000];
  char **w;
  n_w=0;
  for (j=0;j<strlen(text);j++) {
    if (text[j]==splitter[0]) {
      n_w++;
    }
  }
  //  n_w++;
  w = (char **)malloc((n_w+1)*sizeof(char*));

  w[0]=strtok(text,splitter);
  for (j=0;j<n_w;j++) {
    w[j+1]=strtok(NULL,splitter);
  }
  *words=w;
  *n_words=(n_w+1);
  //  free(w);
  return(0);
}

/*
 * Basic Stats
 */

int basic_stats(float *data, int n, float *mean, float *sigma, int sigma_clip) {

  float mean_temp=0.0;
  float sigma_temp=0.0;
  float mean1=0.0;
  float sigma1=0.0;
  int i,sum;
  int *marca;

  marca = (int *)malloc(n*sizeof(int));  

  for (i=0;i<n;i++) {
    mean1+=data[i]/n;
  }
 
  for (i=0;i<n;i++) {
    sigma1+=pow(data[i]-mean1,2);
  }
  sigma1=sqrt(sigma1/(n-1));
  
  if (sigma_clip!=0) {
    sum=0;
    for (i=0;i<n;i++) {
      marca[i]=0;
      if (abs(data[i]-mean1)<(sigma_clip*sigma1)) {
	mean_temp+=data[i];
	marca[i]=1;
	sum++;
      }
    }
    mean_temp/=sum;
    

  } else {
    mean_temp=mean1;
    sigma_temp=sigma1;
  }
 
  *mean=mean_temp;
  *sigma=sigma_temp;
  return(0);
}


/*
 * Convolution/Correlation staff
 */
int test_tr_palett(int n, float *in_data, float *temp_data, int sig, double contra, double bright, float min, float max) {
  int j;
  float a,b,c;
  // float *temp_data;
  //temp_data = (float *)malloc(n*sizeof(float));    
  if (sig==1) {
    b=(1-bright)/(contra);
    c=b/(max+min);
    a=(-1)*(c*max*max+(b-1)*max);    
    for (j=0;j<n;j++) {
      temp_data[j]=(a+b*in_data[j]+c*pow(in_data[j],2));
    }  
    
  } 
/*
else {
    b=(bright)/(1-contra);

    c=((-1)*b)/(max+min);
    a=(-1)*(c*max*max+b*max-min);    
    for (j=0;j<n;j++) {
      temp_data[j]=(a+b*in_data[j]+c*pow(in_data[j],2));
    }  

  }
*/  

  
//  *out_data=temp_data;
  return(0);
}


/*
 * TR PALETT
 * The SIGN= 1=linear, 2=LOG, 3=SQRT
 */
int tr_palett(int n, float *in_data, float **out_data, int sig, double contra, double bright, float min, float max) {
  int j;
  float a,b,c;
  float *temp_data;
  float suma;
  float x;

  float lbright,lcontra,lmax,lmin;
  float mean;
  lbright=(float)bright;
  lcontra=(float)contra;



  temp_data = (float *)malloc(n*sizeof(float));    
  suma=max+min;
  
  if (suma==0) suma=0.0001;


  switch (sig) {    
      case (1):  
	b=(1-bright)/(contra);
	c=b/(suma);
	a=(-1)*(c*max*max+(b-1)*max);    
	for (j=0;j<n;j++) {
	  temp_data[j]=(a+b*in_data[j]+c*pow(in_data[j],2));
	}  
	break;
      case (2):	
	b=(1-bright);//(contra);
	for (j=0;j<n;j++) {
	  temp_data[j]=(b*log10(in_data[j]/contra));
	}    
	break;
      case (3):	  
	b=(1-bright);//(contra);
	for (j=0;j<n;j++) {
	  temp_data[j]=(b*pow(in_data[j]/contra,2));
	}    	
	break;
      case (4):	  
	b=(1-bright);//(contra);
	for (j=0;j<n;j++) {
	  if (in_data[j]>0.0) {
	    temp_data[j]=(b*sqrt(abs(in_data[j]/contra)));
	  } else {
	    temp_data[j]=-(b*sqrt(abs(in_data[j]/contra)));
	  }
	}    	
	break;
      case (5):	  
	b=(1-bright);//(contra);
	for (j=0;j<n;j++) {
	  temp_data[j]=b*in_data[j]/contra;
	}    
	break;
      default:
	b=(1-bright)/(contra);
	c=b/(suma);
	a=(-1)*(c*max*max+(b-1)*max);    
	for (j=0;j<n;j++) {
	  temp_data[j]=(a+b*in_data[j]+c*pow(in_data[j],2));
	}  
	break;
  }


  
  if (sig==1) {
  } else {
    b=(bright)/(1-contra);
    c=((-1)*b)/(suma);
    a=(-1)*(c*max*max+b*max-min);    
    for (j=0;j<n;j++) {
      temp_data[j]=(a+b*in_data[j]+c*pow(in_data[j],2));
    }  

 }
  

  /*
  c=contra;
  lmax=max;
  lmin=min;
  b=(lmax-lmin)/(atan(c*(lmax-mean)/0.2)-atan(c*(lmin-mean)/0.2));
  a=lmin-b*atan2(c*(lmin-mean),0.2);
  for (j=0;j<n;j++) {
    temp_data[j]=(bright/0.5)*(a+b*atan((c*in_data[j]-mean)/0.2));
  }  
  */


  *out_data=temp_data;
  return(0);
}


/*
 * Reading a velocity map created my the "map.pl" script
 * and the "velocity.pl"+"velocity" programs.
 *
 * It should contain in each line:
 * id x y F e_F Wave e_Wave Delta_Wave e_Delta_Wave
 */
int read_velocity_map(char* filename, struct VELOCITY_MAP *vel_map, float sistemic, float resolution) {
    FILE *fichero;
    float temp_float;    
    int count_f,n,i,temp_int;
//    float sistemic;
//    float d_sistemic;
    float c=299792.458;
    printf("Reading data...\n");
    count_f=0;
    n=0;
    if ((fichero=fopen(filename,"r"))==NULL) {
	printf("Error opening file: %s\n",filename);
	return(-1);
    } else {
	while( fscanf(fichero, "%f", &temp_float) != EOF) {
	    count_f++;
	    if (count_f>9) {
		count_f=0;
		n++;
	    }
	}
    }
    rewind(fichero);
    vel_map->n= n;
    vel_map->id= (int *)calloc(n,sizeof(int));  
    vel_map->x= (float *)calloc(n,sizeof(float));  
    vel_map->y= (float *)calloc(n,sizeof(float));  
    vel_map->f= (float *)calloc(n,sizeof(float));  
    vel_map->e_f= (float *)calloc(n,sizeof(float));  
    vel_map->w= (float *)calloc(n,sizeof(float));  
    vel_map->e_w= (float *)calloc(n,sizeof(float));  
    vel_map->dw= (float *)calloc(n,sizeof(float));  
    vel_map->e_dw= (float *)calloc(n,sizeof(float));  
    vel_map->q= (int *)calloc(n,sizeof(int));  
    for(i=0;i<n;i++) {
	fscanf(fichero, "%d", &temp_int);
	vel_map->id[i]=temp_int;
	fscanf(fichero, "%f", &temp_float);
	vel_map->x[i]=temp_float;
	fscanf(fichero, "%f", &temp_float);
	vel_map->y[i]=temp_float;
	fscanf(fichero, "%f", &temp_float);
	vel_map->f[i]=temp_float;
	fscanf(fichero, "%f", &temp_float);
	vel_map->e_f[i]=temp_float;
	fscanf(fichero, "%f", &temp_float);
//	printf("%d %f\n",i,temp_float);
	vel_map->w[i]=temp_float;
	fscanf(fichero, "%f", &temp_float);
	vel_map->e_w[i]=temp_float;
	fscanf(fichero, "%f", &temp_float);
	vel_map->dw[i]=temp_float;
	fscanf(fichero, "%f", &temp_float);
	vel_map->e_dw[i]=temp_float;
	fscanf(fichero, "%d", &temp_int);
	vel_map->q[i]=temp_int;
    }
    /*
     * We transform to velocities
     */
//    mean_loop(vel_map->w,n,1,3,&sistemic,&d_sistemic);
    if (sistemic==0) sistemic=1.0;
    for(i=0;i<n;i++) {
//	vel_map->w[i]=((vel_map->w[i]-sistemic)/sistemic)*c;
//	printf("%d %f %f\n",i,vel_map->w[i],sistemic);
	vel_map->w[i]=(((vel_map->w[i])/sistemic)-1)*c;
	vel_map->e_w[i]=((vel_map->e_w[i])/sistemic)*c;
	vel_map->dw[i]=(sqrt(vel_map->dw[i]*vel_map->dw[i]-resolution*resolution)/sistemic)*c;
	vel_map->e_dw[i]=((vel_map->e_dw[i])/sistemic)*c;
//	printf("%d %f %f\n",i,vel_map->w[i],sistemic);
    }    

    vel_map->lambda=sistemic;
    printf("File readed\n");
    fclose(fichero);
    return(0);
}


/*
 * read_cont_map
 */

int read_cont_map(char* filename, struct VELOCITY_MAP *vel_map, float width) {
    FILE *fichero;
    float temp_float;    
    int count_f,n,i,temp_int;
//    float sistemic;
//    float d_sistemic;
    float c=299792.458;
    printf("Reading data...\n");
    count_f=0;
    n=0;
    if ((fichero=fopen(filename,"r"))==NULL) {
	printf("Error opening file: %s\n",filename);
	return(-1);
    } else {
	while( fscanf(fichero, "%f", &temp_float) != EOF) {
	    count_f++;
	    if (count_f>9) {
		count_f=0;
		n++;
	    }
	}
    }
    rewind(fichero);
    vel_map->n= n;
    vel_map->id= (int *)calloc(n,sizeof(int));  
    vel_map->x= (float *)calloc(n,sizeof(float));  
    vel_map->y= (float *)calloc(n,sizeof(float));  
    vel_map->f= (float *)calloc(n,sizeof(float));  
    vel_map->e_f= (float *)calloc(n,sizeof(float));  
    vel_map->w= (float *)calloc(n,sizeof(float));  
    vel_map->e_w= (float *)calloc(n,sizeof(float));  
    vel_map->dw= (float *)calloc(n,sizeof(float));  
    vel_map->e_dw= (float *)calloc(n,sizeof(float));  
    vel_map->q= (int *)calloc(n,sizeof(int));  
    for(i=0;i<n;i++) {
	fscanf(fichero, "%d", &temp_int);
	vel_map->id[i]=temp_int;
	fscanf(fichero, "%f", &temp_float);
	vel_map->x[i]=temp_float;
	fscanf(fichero, "%f", &temp_float);
	vel_map->y[i]=temp_float;
	fscanf(fichero, "%f", &temp_float);
	vel_map->f[i]=temp_float;
	fscanf(fichero, "%f", &temp_float);
	vel_map->e_f[i]=temp_float;
	fscanf(fichero, "%f", &temp_float);
	vel_map->w[i]=temp_float*width;
	fscanf(fichero, "%f", &temp_float);
	vel_map->e_w[i]=temp_float*width;
	fscanf(fichero, "%f", &temp_float);
	vel_map->dw[i]=temp_float;
	fscanf(fichero, "%f", &temp_float);
	vel_map->e_dw[i]=temp_float;
	fscanf(fichero, "%d", &temp_int);
	vel_map->q[i]=temp_int;
    }
    /*
     * We transform to velocities
     */
    vel_map->lambda=0.0;
    printf("File readed\n");
    fclose(fichero);
    return(0);
}

