/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
! COPYRIGHT    (c)  2002 AIP, Potsdam, Germany
! IDENT        Euro3D.c
! LANGUAGE     C
! AUTHOR       S.F.Sanchez
! KEYWORDS
! PURPOSE      Library for the Visualization Tool
! COMMENT      Teel process
! VERSION      0.1  2003-Jan-29 : Creation, SF. Sanchez
------------------------------------------------------------------------------*/

#include <stdio.h> //For Standard I/O
#include <stdlib.h> //For Functions like ATOI
#include <ctype.h> //For Type check
#include <string.h> //For String manipulation
#include <math.h> //For Math functions
#include <IFU_io.h> //For Lyon I/O Library
#include <data_io.h> //For Lyon I/O data
#include <tk.h>
#include <time.h>

/******************************
 * For Shared Memory
 ******************************/

#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/shm.h>



/******************************
 *To use PGPLOT
 *******************************/

#include <cpgplot.h> // PGPLOT
#include "tkpgplot.h"

/************************************
 * NCARG
 ************************************/
#include <ncarg/ngmath.h>
#include <ncarg/ncargC.h>
#include <ncarg/gks.h>



/*********************************************
 * Different procedures to read
 * the basic information on the SHM
 *********************************************/

/*
 * Limitation in shared memory due to the computer
 *
 */
#define MAX_SHM 33554432


int basic_stats(float *data, int n, float *mean, float *sigma, int sigma_clip);
int mark_range(float x1, float y1, float x2, float y2, char* text);
int ask_alive();
int ask_slice(float **out_raw_data, int nb_spec, int index_w);
int ask_spectrum(float **out_raw_data, int specId, int npix, int index_start);
int ask_raw_data(float **out_raw_data, int nb_spec, int npix, int delta_n);
int ask_e3d_info(E3D_file *e3d_frame, int **n_spaxels, SPAXEL ***spaxels, int **index_start, int **npts, GROUP **groups);
int ask_load_e3d(char *input_filename);
int ask_delete_shm();
int load_mem_all (E3D_file e3d_image, int *n_spaxels,SPAXEL **spaxels,int *index_start, int *npts, int **address);
int server();
int load_e3d(char *filename, E3D_file *e3d_frame, int **n_spaxels, SPAXEL ***spaxels, int **index_start, int **npts, float **raw_data, float **raw_noise);
float* read_spax_int(int i_spec, int nb_spec, int npix);
int read_mem_group(GROUP **group,int key, int size);
int load_mem_group(GROUP *grupo, int key, int size, int *address);
int read_mem_int(int **out_int, int key, int size);
int load_mem_int(int *raw_data, int key, int size, int* address);
int  read_mem_char(char **out_shm,int key, int size);
char load_mem_char(char *raw_data, int key, int size, int* address);
int read_mem_spaxels(SPAXEL ***out_spax, int* n_spaxels, int key, int size);
int load_mem_spaxels(SPAXEL **spaxels, int* n_spaxels, int key, int size, int* address);
int read_mem_float(float **out_float, int key, int size);
int load_mem_float(float *raw_data, int key, int size, int* address);
float* read_mem_staked_spectra(int key, int size);
int load_mem_staked_spectra(float *raw_data, int key, int size, int* address);
int load_mem_e3d(E3D_file e3d_file, int key, int *address);
int read_mem_e3d(E3D_file* e3d_out, int key);
int palette(int type, float contra, float bright);
float Minimun(float a, float b);
float Maximun(float a, float b);
int draw_staked_spectra(float min, float max,int **nnaxes, float **out);
void plot_spaxel(int color_index,float x_c, float y_c, GROUP group, int fill, int text_flag);
int split_text(char *text, char *splitter, int *n_words, char ***words);
int ask_change_palette(char *commands);
int image_plot(float *data, float datamin, float datamax, int n_x, int n_y, int dev_id, float *tr);
int ask_draw_raw(char *commands);

//int ask_commands(int f_cero,char *commands);
int ask_open_device(char *commands);
int ask_close_device(char *commands);
int ask_intensity(float *out_raw_data, int nb_spec, int index_w);
void spec_plot(float* data, float datamin, float datamax, int npix, double start_w, double delta_w);


int old_create_map(int nb_spec, int *n_spaxels, SPAXEL **spaxels, int *index_start, int *npts, float* slice, float dpix, float** map_data, float x_min, float x_max, float y_min, float y_max, float *ttr[8], int *nnx, int *nny);

int create_map(int nb_spec, int *n_spaxels, SPAXEL **spaxels, int *index_start, int *npts, float* slice, float dpix, float** map_data, float x_min, float x_max, float y_min, float y_max, float *ttr[8], int *nnx, int *nny);


int save_map(float* map_data, int nx, int ny, char* filename);

/*
 * Determines the principal momentum of an image.
 */

/*
 * Save a Map
 */ 

int save_map(float* map_data, int nx, int ny, char* filename) {
  IMAGE2D image;
  int npix[2];
  double start[2];
  double step[2];
  int i,j;
  npix[0]=nx;
  npix[1]=ny;
  start[0]=0.0;
  start[1]=0.0;
  step[0]=1.0;
  step[1]=1.0;
  create_frame(&image,filename,npix,start,step,FLOAT,"MAP",NULL);
  for (i=0;i<image.nx;i++) {
    for (j=0;j<image.ny;j++) {
      WR_frame(&image,i,j,map_data[i+j*nx]);
      WR_qframe(&image,i,j,(long) 0);
    }
  }
  close_frame(&image);  
  return(0);
}

/*
 * Create an interpolated Map
 * saving its WCS
 */

int create_map(int nb_spec, int *n_spaxels, SPAXEL **spaxels, int *index_start, int *npts, float* slice, float dpix, float** map_data, float x_min, float x_max, float y_min, float y_max, float *ttr[8], int *nnx, int *nny) {
  int i,j,k,l,m;
  int nx,ny;
  float *temp_data;
  float x_c,y_c,xx[4],yy[4];
  //  float tr[8];
  float tr[]={0.0,1.0,0.0,0.0,0.0,1.0};

  int ier;
  float *x,*y,*z;
  int n_tot;

  float *xo,*yo,rpnt;


  nx = floor((x_max-x_min)/dpix);
  ny = floor((y_max-y_min)/dpix);
  temp_data = (float *)malloc(nx*ny*sizeof(float));    

  /*
   * We define the points in the grid
   */
  xo = (float *)malloc(nx*ny*sizeof(float));    
  yo = (float *)malloc(nx*ny*sizeof(float));    

  k=0;
  for (i=0;i<nx;i++) {
    for (j=0;j<ny;j++) {
      xo[k]=x_min+dpix*i;
      yo[k]=y_min+dpix*j;
      //      printf("[%f,%f] \n",xo[k],yo[k]);
      k++;
    }
  }


  /*
   * We count the number of points included in the
   * SPAXELS array
   *
   */
  n_tot=0;
  for (k=0;k<nb_spec;k++) {
    for (l=0;l<n_spaxels[k];l++) {
      n_tot++;
    }
  }
  x = (float *)malloc(n_tot*sizeof(float));    
  y = (float *)malloc(n_tot*sizeof(float));    
  z = (float *)malloc(n_tot*sizeof(float));    
  /*
   * We feed with data
   */
  n_tot=0;
  for (k=0;k<nb_spec;k++) {
    for (l=0;l<n_spaxels[k];l++) {
      x[n_tot]=spaxels[k][l].xpos;
      y[n_tot]=spaxels[k][l].ypos;
      z[n_tot]=slice[k];
      //      printf("F[%f,%f]=%f\n",x[n_tot],y[n_tot],z[n_tot]);
      n_tot++;
    }
  }


  /*
   * We interpolate!
   */

  
  /*
   *  Enter single point mode.
   */
  c_nnpntinits(n_tot, x, y, z);


  //  temp_data = c_natgrids(n_tot, x, y, z, nx*ny, nx*ny, xo, yo, &ier);

  k=0;
  for (i=0;i<nx;i++) {
    for (j=0;j<ny;j++) {
      c_nnpnts(xo[k], yo[k], &rpnt);
      temp_data[i+j*nx] = rpnt;

      //      printf("[%f,%f] \n",xo[k],yo[k],temp_data[k]);
      k++;
    }
  }
  /*
  if (ier != 0) {
    printf (" Error return from c_natgrids = %d\n",ier);
  }
  */





  *nnx=nx;
  *nny=ny;
  *ttr=tr;
  *map_data=temp_data;
  return(0);
}

/*
 * Create an interpolated Map
 * saving its WCS
 */

int old_create_map(int nb_spec, int *n_spaxels, SPAXEL **spaxels, int *index_start, int *npts, float* slice, float dpix, float** map_data, float x_min, float x_max, float y_min, float y_max, float *ttr[8], int *nnx, int *nny) {
  int i,j,k,l,m;
  int nx,ny;
  float *temp_data;
  float x,y,x_c,y_c,xx[4],yy[4];
  //  float tr[8];
  float tr[]={0.0,1.0,0.0,0.0,0.0,1.0};
  float d[4],dist_temp,d_par[4][4],max_par[4];
  int index[4];
  float prod=1.0;

  nx = floor((x_max-x_min)/dpix);
  ny = floor((y_max-y_min)/dpix);
  temp_data = (float *)malloc(nx*ny*sizeof(float));    


  //  printf("%d,%d,%f,%f,%f,%f\n",nx,ny,x_min,x_max,y_min,y_max);

  //  printf("Paso= CREATE MAP\n");
  for (j=0;j<ny;j++) {
    for (i=0;i<nx;i++) {
      for (k=0;k<4;k++) {
	index[k]=0;
	d[k]=32000.0;
      }
      temp_data[i+j*nx]=0.0;
      x=((float)i)*dpix+x_min;
      y=((float)j)*dpix+y_min;
      /*
       * we move along all the SPAXELs
       * to determine the distances
       */
      for (k=0;k<nb_spec;k++) {
	for (l=0;l<n_spaxels[k];l++) {
	  x_c=spaxels[k][l].xpos;
	  y_c=spaxels[k][l].ypos;
	  dist_temp=sqrt(pow(x_c-x,2)+pow(y_c-y,2));
	  //	  printf("%f,%f,%f,%f,%f\n",dist_temp,d[0],d[1],d[2],d[3]);
	  if (dist_temp<=d[0]) {
	    d[3]=d[2];
	    d[2]=d[1];
	    d[1]=d[0];
	    d[0]=dist_temp;
	    index[3]=index[2];
	    index[2]=index[1];
	    index[1]=index[0];
	    index[0]=k;
	    xx[3]=xx[2];
	    xx[2]=xx[1];
	    xx[1]=xx[0];
	    xx[0]=x_c;
	    yy[3]=yy[2];
	    yy[2]=yy[1];
	    yy[1]=yy[0];
	    yy[0]=y_c;
	  } else if (dist_temp<=d[1]) {
	    d[3]=d[2];
	    d[2]=d[1];
	    d[1]=dist_temp;
	    index[3]=index[2];
	    index[2]=index[1];
	    index[1]=k;
	    xx[3]=xx[2];
	    xx[2]=xx[1];
	    xx[1]=x_c;
	    yy[3]=yy[2];
	    yy[2]=yy[1];
	    yy[1]=y_c;
	  }  else if (dist_temp<=d[2]) {
	    d[3]=d[2];
	    d[2]=dist_temp;
	    index[3]=index[2];
	    index[2]=k;
	    xx[3]=xx[2];
	    xx[2]=x_c;
	    yy[3]=yy[2];
	    yy[2]=y_c;
	  }   else if (dist_temp<=d[3]) {
	    d[3]=dist_temp;
	    index[3]=k;
	    xx[3]=x_c;
	    yy[3]=y_c;
	  }	  
	}	
      }

      for (l=0;l<4;l++) {
	max_par[l]=-1000;
	for (k=0;k<4;k++) {
	  d_par[l][k]=sqrt(pow(xx[l]-xx[k],2)+pow(yy[l]-yy[k],2));
	  if (d_par[l][k]>max_par[l]) max_par[l]=d_par[l][k];
	}
      }
      
      /*
       * We test if it is among the points!
       */
      if ((d[0]+d[1]+d[2])<(d_par[0][1]+d_par[0][2]+d_par[1][2])) {
	
	prod=0.0;
	for (l=0;l<3;l++) {
	  prod+=1/(1.0+d[l]);
	  temp_data[i+j*nx]+=slice[index[l]]/(1.0+d[l]);	
	  
	}
	temp_data[i+j*nx]/=prod;
      } else {
	temp_data[i+j*nx]=-1000;
      }
    }
  }

  //  printf("Paso= CREATE MAP\n");

  *nnx=nx;
  //  printf("1\n");
  *nny=ny;
  //printf("2\n");
  *ttr=tr;
  //printf("3\n");
  *map_data=temp_data;
  // printf("4\n");
  //printf("Paso\n");
  return(0);
}



/*
 * Mark a range with an arrow and a text 
 */
int mark_range(float x1, float y1, float x2, float y2, char* text) {
  float x_m,y_m;
  x_m=(x2+x1)/2;
  y_m=(y2+y1)/2;
  cpgarro(x_m,y_m,x1,y1);
  cpgarro(x_m,y_m,x2,y2);
  cpgptxt(x_m,y_m+1.0,0.0,1.5,text);
  return(0);
}

/*
 * Plot a certain spectra
 */
void spec_plot(float* data, float datamin, float datamax, int npix, double start_w, double delta_w) {
  float *x,*y;
  int i,j;

  if (start_w==0.0)
    start_w=1.0;
  if (delta_w==0.0)
    delta_w=1.0;

  x = (float *)malloc(npix*sizeof(float));  
  y = (float *)malloc(npix*sizeof(float));  
  for (i=0;i<npix;i++) {
    x[i]=(float)start_w+(float)delta_w*i;
    y[i]=data[i];
  }
  cpgsch(1.1);  
  cpgenv(x[0],x[npix-1],datamin,datamax,0,0);
  cpglab("\\gLambda", "Flux", "Spectra");	  
  cpgsls(1);  
  cpgpt(npix,x,y,1);
  cpgline(npix,x,y);
  
}




/**************************************
 * Ask to Open a device
 **************************************/
int ask_open_device(char *commands) {
  int *flag;
  int tmp_address;
  int mem_key;
  int n_bytes;
  //  printf("%s\n",commands);

  flag = (int *)malloc(15*sizeof(int));
  load_mem_char(commands,4,80,&tmp_address);
  mem_key=3;
  flag[0]=101;
  flag[1]=0;
  flag[2]=0;
  n_bytes=15;       
  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
  while(flag[1]==0) { 
    //    printf("Aqui me quedo?\n");
    read_mem_int(&flag,mem_key,n_bytes);		
  }
  return(flag[3]);
}


/**************************************
 * Ask to Close a device
 **************************************/
int ask_close_device(char *commands) {
  int *flag;
  int tmp_address;
  int mem_key;
  int n_bytes;

  flag = (int *)malloc(15*sizeof(int));
  load_mem_char(commands,4,80,&tmp_address);
  mem_key=3;
  flag[0]=102;
  flag[1]=0;
  flag[2]=0;
  n_bytes=15;       
  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
  while(flag[1]==0) { 
    read_mem_int(&flag,mem_key,n_bytes);		
  }
  return(0);
}


/**************************************
 * Ask to draw the raw data
 **************************************/
int ask_draw_raw(char *commands) {
  int *flag;
  int tmp_address;
  int mem_key;
  int n_bytes;

  flag = (int *)malloc(15*sizeof(int));
  load_mem_char(commands,4,80,&tmp_address);
  mem_key=3;
  flag[0]=103;
  flag[1]=0;
  flag[2]=0;
  n_bytes=15;       
  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
  while(flag[1]==0) { 
    read_mem_int(&flag,mem_key,n_bytes);		
  }
  return(0);
}

/**********************************************
 * Ask to change the Palette
 * Where Commands='Pal Bright Contra Sign DEV_ID'
 **********************************************/
int ask_change_palette(char *commands) {
  int *flag;
  int tmp_address;
  int mem_key;
  int n_bytes;

  flag = (int *)malloc(15*sizeof(int));
  load_mem_char(commands,4,80,&tmp_address);
  mem_key=3;
  flag[0]=100;
  flag[1]=0;
  flag[2]=0;
  n_bytes=15;       
  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
  while(flag[1]==0) { 
    read_mem_int(&flag,mem_key,n_bytes);		
  }
  return(0);
}


/**********************************************
 * Ask for a single intensity in a point
 * defined by its index
 **********************************************/
int ask_intensity(float *out_raw_data, int nb_spec, int index_w) {
  int *flag;
  int tmp_address;
  int mem_key;
  int n_bytes;
  int i,j,jj;
  float *raw_spec;
  flag = (int *)malloc(15*sizeof(int));  
  mem_key=3;
  n_bytes=15;
  flag[0]=7;
  flag[1]=0;
  flag[2]=index_w;
  flag[3]=nb_spec;
  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
  while(flag[1]==0) { 
    read_mem_int(&flag,mem_key,n_bytes);		
  }
  read_mem_float(&raw_spec,2,1);	    
  *out_raw_data=raw_spec[0];  
}

/**********************************************
 * Ask for an SLIDE of DATA at a certain Wavelength
 * defined by its index
 **********************************************/
int ask_alive() {
  int *flag;
  int tmp_address;
  int mem_key;
  int n_bytes;
  time_t t1,t2;
  int overtime=0;

  flag = (int *)malloc(15*sizeof(int));  
  mem_key=3;
  n_bytes=15;
  /*
   * We send a signal to the server
   */
  flag[0]=666;
  flag[1]=0;
  flag[2]=666;
  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
  (void) time(&t1);
  /*
   * Waint until an answer
   */
  while((flag[1]==0)&&(overtime==0)) { 
    read_mem_int(&flag,mem_key,n_bytes);		
    (void) time(&t2);
    //    printf("TIME=%d\n",(int)(t2-t1));
    if (((int)(t2-t1))>1) overtime=1;
  }
  /*
   * Read each spectra plot by the 
   */
  //  printf("Overtime=%d\n",overtime);
  return(overtime);
}


/**********************************************
 * Ask for an SLIDE of DATA at a certain Wavelength
 * defined by its index
 **********************************************/
int ask_slice(float **out_raw_data, int nb_spec, int index_w) {
  int *flag;
  int tmp_address;
  int mem_key;
  int n_bytes;
  int i,j,jj;
  float *raw_spec,*raw_data;
  flag = (int *)malloc(15*sizeof(int));  
  raw_data = (float *)malloc((nb_spec)*sizeof(float));
  mem_key=3;
  n_bytes=15;
  /*
   * We send a signal to the server
   */
  flag[0]=5;
  flag[1]=0;
  flag[2]=index_w;
  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
  /*
   * Waint until an answer
   */
  while(flag[1]==0) { 
    read_mem_int(&flag,mem_key,n_bytes);		
  }
  /*
   * Read each spectra plot by the 
   */
  read_mem_float(&raw_spec,2,nb_spec);	    
  for (i=0;i<nb_spec;i++) {
    raw_data[i]=raw_spec[i];
  }
  *out_raw_data=raw_data;  
  return(0);
}


/**********************************************
 * Ask for Only one Spectra, of specId
 **********************************************/
int ask_spectrum(float **out_raw_data, int specId, int npix, int index_start) {
  int *flag;
  int tmp_address;
  int mem_key;
  int n_bytes;
  int i,j,jj;
  float *raw_spec,*raw_data;
  flag = (int *)malloc(15*sizeof(int));  
  raw_data = (float *)malloc((npix)*sizeof(float));
  mem_key=3;
  n_bytes=15;
  j=specId-1;
  jj=j+1;
  /*
   * We send a signal to the server
   */
  flag[0]=2;
  flag[1]=0;
  flag[2]=j;
  flag[3]=jj;
  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
  /*
   * Waint until an answer
   */
  while(flag[1]==0) { 
    read_mem_int(&flag,mem_key,n_bytes);		
  }
  flag[0]=0;
  flag[1]=0;
  flag[2]=0;
  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
  /*
   * Read each spectra plot by the 
   */
  read_mem_float(&raw_spec,2,npix);	    
  for (i=0;i<npix;i++) {
    raw_data[i]=raw_spec[i+index_start];
  }
  *out_raw_data=raw_data;  
}


/**********************************************
 * Ask for the Staked Spectra as an array of floats!
 **********************************************/
int ask_raw_data(float **out_raw_data, int nb_spec, int npix, int delta_n) {
  int *flag;
  int tmp_address;
  int mem_key;
  int n_bytes;
  int i,j,jj;
  float *raw_spec,*raw_data;
  flag = (int *)malloc(15*sizeof(int));  
  raw_data = (float *)malloc((nb_spec*npix)*sizeof(float));
  mem_key=3;
  n_bytes=15;
  for (j=0;j<nb_spec;j=j+delta_n) {
    jj=j+delta_n;
    if (jj>=nb_spec) {
      jj=nb_spec;
      delta_n=jj-j;
    }
     /*
      * We send a signal to the server
      */
    flag[0]=2;
    flag[1]=0;
    flag[2]=j;
    flag[3]=jj;
    load_mem_int(flag,mem_key,n_bytes,&tmp_address);
    /*
     * Waint until an answer
     */
    while(flag[1]==0) { 
      read_mem_int(&flag,mem_key,n_bytes);		
    }
    flag[0]=0;
    flag[1]=0;
    flag[2]=0;
    load_mem_int(flag,mem_key,n_bytes,&tmp_address);
    /*
     * Read each spectra plot by the 
     */
    read_mem_float(&raw_spec,2,npix);	    
    for (i=0;i<npix*delta_n;i++) {
      raw_data[i+j*npix]=raw_spec[i];
    }
  }
  *out_raw_data=raw_data;
}

/**********************************************
 * Ask for the E3D image INFO
 **********************************************/

int ask_e3d_info(E3D_file *e3d_frame, int **n_spaxels, SPAXEL ***spaxels, int **index_start, int **npts, GROUP **groups) {

  E3D_file e3d_temp;
  int *n_spaxels_temp;
  SPAXEL **spaxels_temp;
  int *index_start_temp;
  int *npts_temp;
  GROUP *groups_temp;
  int i,j,k;
  int nb_spec, npix, specId;
  int *nol_temp, max_npts;
  float delta_w;
  int mem_key;
  int n_groups;
  double start_w,end_w;

  mem_key=1;
  //printf("We read the E3D_file from memory\n");
  if (read_mem_e3d(&e3d_temp,mem_key)!=0) {
    //printf("No E3D descriptor on SHM");
    //printf("\n");
    exit(0);
  }
  start_w=e3d_temp.common_parameters[0];
  end_w=e3d_temp.common_parameters[1];
  npix=e3d_temp.common_parameters[2];
  delta_w=(end_w-start_w)/npix;
  nb_spec=e3d_temp.nbspec;
  n_groups=e3d_temp.ngroups;
  //printf("Ok\n");

  //printf("We read the N_SPAXELS from the memory\n");
  mem_key=5;
  read_mem_int(&n_spaxels_temp,mem_key,nb_spec);
  //printf("Ok\n");
  //printf("We read the SPAXELS from the memory\n");
  mem_key=6;
  read_mem_spaxels(&spaxels_temp,n_spaxels_temp,mem_key,nb_spec);
  //printf("Ok\n");
  //printf("We read the INDEX_START from the memory\n");
  mem_key=7;	
  read_mem_int(&index_start_temp,mem_key,nb_spec);
  //printf("Ok\n");
  //printf("We read the NPTS from the memory\n");
  mem_key=8;
  read_mem_int(&npts_temp,mem_key,nb_spec);
  //printf("Ok\n");
  //printf("We read the GROUPS from the memory\n");
  mem_key=9;
  read_mem_group(&groups_temp,mem_key,nb_spec);
  //printf("Ok\n");

  *e3d_frame=e3d_temp;  
  *n_spaxels=n_spaxels_temp;
  *spaxels=spaxels_temp;
  *index_start=index_start_temp;
  *npts=npts_temp;
  *groups=groups_temp;

  return(0);
}


/**********************************************
 * ask to load a new FILE and ask for the 
 * descriptors information.
 **********************************************/
int ask_load_e3d(char *input_filename) {
  int *flag;
  int tmp_address;
  int mem_key;
  int n_bytes;

  flag = (int *)malloc(15*sizeof(int));

  // printf("Asking to load '");
  // printf(input_filename);
  // printf("'\n");
  fflush(stdout);
  /* Load the Filename in the SHM space */
  load_mem_char(input_filename,4,80,&tmp_address);
  // printf("Paso");
  mem_key=3;
  flag[0]=3;
  flag[1]=0;
  flag[2]=0;
  n_bytes=15;       
  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
  while(flag[1]==0) { 
    read_mem_int(&flag,mem_key,n_bytes);		
  }
  return(0);
}

/**********************************************
 * ask to delete all the SHM
 * Send a command to the server to delete the SHM
 **********************************************/
int ask_delete_shm() {
  int *flag;
  int tmp_address;
  int mem_key;
  int n_bytes;

  flag = (int *)malloc(15*sizeof(int));

  mem_key=3;
  flag[0]=4;
  flag[1]=0;
  flag[2]=0;
  n_bytes=15;
  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
  while(flag[1]==0) { 
    read_mem_int(&flag,mem_key,n_bytes);		
  }
  flag[0]=0;
  flag[1]=0;
  flag[2]=0;
  n_bytes=15;
  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
  return(0);
}


/*********************************************
 * Load in Memory, not in the SHM, all the information available 
 * for a certain E3D_file
 *********************************************/
int load_mem_all (E3D_file e3d_image, int *n_spaxels,SPAXEL **spaxels,int *index_start, int *npts, int **address) {
  int tmp_address;
  int npix,nb_spec,n_groups;
  int mem_key,n_bytes;

  int j;

  int caracter;

  nb_spec=e3d_image.nbspec;  
  n_groups=e3d_image.ngroups;
  npix=e3d_image.common_parameters[2];

  /*
   * We load in memory the Euro3D image
   */
  
  //  address = (int *)malloc(15*sizeof(int));  	
  
  mem_key=1;
  if (load_mem_e3d(e3d_image,mem_key,&tmp_address)!=0) {
    //printf("Error loading in SHM\n");
    return(1);
  }
  address[mem_key]=tmp_address;
  //  printf("add=%d\n",address[mem_key]);
  //printf("Load Ok\n");

  //caracter=getc(stdin);

  /*
   * We load the N_SPAXELS in Memory
   */
  //printf("\nWe store in the shared memory the N_SPAXELS\n");
  mem_key=5;
  n_bytes=nb_spec;
  if (n_bytes<MAX_SHM) {
    load_mem_int(n_spaxels,mem_key,n_bytes,&tmp_address);
    //printf("Ok\n");
  } else {
    //printf("*******************************\n");
    //printf("** TOO MUCH LARGE FOR THE SHM**\n");
    //printf("*******************************\n");
  }
  address[mem_key]=tmp_address;
  //caracter=getc(stdin);
  /*
   * We load the SPAXELS in Memory
   */
  //printf("\nWe store in the shared memory the SPAXELS\n");
  mem_key=6;
  n_bytes=nb_spec;
  if (n_bytes<MAX_SHM) {
    load_mem_spaxels(spaxels,n_spaxels,mem_key,n_bytes,&tmp_address);
    //printf("Ok\n");
  } else {
    //printf("*******************************\n");
    //printf("** TOO MUCH LARGE FOR THE SHM**\n");
    //printf("*******************************\n");
  }
  address[mem_key]=tmp_address;
  //caracter=getc(stdin);
  /*
   * We load the INDEX_START in Memory
   */
  //printf("\nWe store in the shared memory the INDEX_START\n");
  mem_key=7;
  n_bytes=nb_spec;
  if (n_bytes<MAX_SHM) {
    load_mem_int(index_start,mem_key,n_bytes,&tmp_address);
    //printf("Ok\n");
  } else {
    //printf("*******************************\n");
    //printf("** TOO MUCH LARGE FOR THE SHM**\n");
    //printf("*******************************\n");
  }
  address[mem_key]=tmp_address;
  //caracter=getc(stdin);
  /*
   * We load the NPTS in Memory
   */
  //printf("\nWe store in the shared memory the NPTS\n");
  mem_key=8;
  n_bytes=nb_spec;
  if (n_bytes<MAX_SHM) {
    load_mem_int(npts,mem_key,n_bytes,&tmp_address);
    //printf("Ok\n");
  } else {
    //printf("*******************************\n");
    //printf("** TOO MUCH LARGE FOR THE SHM**\n");
    //printf("*******************************\n");
  }
  address[mem_key]=tmp_address;
  //caracter=getc(stdin);
  //printf("\nWe store in the shared memory the GROUPS info\n");
  mem_key=9;
  n_bytes=nb_spec;
  if (n_bytes<MAX_SHM) {
    load_mem_group(e3d_image.groups,mem_key,n_bytes,&tmp_address);
    //printf("Ok\n");
  } else {
    //printf("*******************************\n");
    //printf("** TOO MUCH LARGE FOR THE SHM**\n");
    //printf("*******************************\n");
  }
  address[mem_key]=tmp_address;
  //caracter=getc(stdin);
  /*
  for (j=1;j<10;j++) {
    printf("address[%d]=%d\n",j,address[j]);
  }
  */
}



/*********************************************
 * Function that uses SHM to work as a server
 * to serve the requested information
 *
 * You have to use a FLAG inserved in the SHM
 * to allow the input of Commands in the 
 * SERVER
 * The FLAG is an array 15 elements
 * the 1st elements is the command.
 *
 * SHM key of the FLAG = 3
 * 
 * FLAG[0]=1 ask for a certain spectra, specId in FLAG[2]
 * FLAG[0]=2 ask for a secuence of spectra, specId from FLAG[2] to FLAG[3]
 *
 * FLAG[0]=3 ask for loading a new E3D_FILE
 * 
 *********************************************/
int server() {

  /*************************************************
   * Painting variables
   *
   *
   *************************************************/
  int p=2;
  float contra=0.8;
  float bright=0.4;
  float sign=1.0;
  
  float tr_now[]={0.0,1.0,0.0,0.0,0.0,1.0};




  int caracter;

    int *flag,temp_flag;

    int *address,tmp_address;
    int n_bytes;

    int window_size;
    float window_aspect;

    int n_colors,nc=200,color_index;
    float x_c,y_c;
    float size1=0.5,size2;
    



    GROUP group, *groups, fake_group;
    GROUP grupo;
    int n_groups;
    
    SPAXEL *spax, **spaxels, *order_spaxels;  
    int *n_spaxels,*index_start,*npts,max_npts;
    
    float x_min,y_min,x_max,y_max;
    
    
    E3D_file e3d_image,e3d_mem,shm;
    SPECTRUM signal,noise;
    SPECTRUM *spectra;
    SLICE s_signal;
    IMAGE2D image;
    char **argval, **arglabel;
    int nbal;
    
    int mem_key;
    int mem_size;

    short s_temp[10];
    int i_temp[10];
    float f_temp[10];
    double d_temp[10];
    char str_temp[80];
    char c_temp[10];
    
    float *raw_data, *raw_noise;
    float *raw_spec;
    float *false_data;

    int shmid;

    /*
      All the images has an initial BITPIX.
      It will be stored in this initial BITPIX,
      but the data will be internally managed as DOUBLE
      not to reduce the precision on none of them.
    */
    short bitpix; // Variable that stores the initial BITPIX value.
    char euro3d;
    
    /* Euro3D variables */
    int specId, i,j,k, npix, nb_spec, status, *nol, nfib;
    double start, step;
    short sval;
    long lval;
    float fval, *x,*y;
    double dval;
    char *format;
    /* General variables */
    char *input_filename, output_filename[80], author[80], origin[80];
    char *commands,**tokens;
    int n_tokens;
    char ref[80],date[80],date_obs[80],object[80],observer[80],telescope[80];
    float equinox;
    int error=0;
    float min,max,mean,sigma,min_i,max_i;
    int naxes[2],daxes[2];
    int npix_int;
    double start_w, end_w, delta_w;
    
    /* PGPLOT needed varibles */
    char answer_now;
    char answer[10];    
    int answer_len = sizeof(answer);
    int real_x,real_y,old_x,old_y;
    
    
    //    FILE *fichero;



    address = (int *)malloc(15*sizeof(int));  	
    /*
     * We load the N_SPAXELS in Memory
     */
    //



    //printf("\nWe store in the shared memory the FLAGs\n");

      
    mem_key=3;
    n_bytes=15;
    flag = (int *)malloc(n_bytes*sizeof(int));  
    for (j=0;j<15;j++) {
      flag[j]=0;
    }
    
    
    if (n_bytes<MAX_SHM) {
      load_mem_int(flag,mem_key,n_bytes,&tmp_address);
	//printf("Ok\n");
    } else {
      //printf("*******************************\n");
      //printf("** TOO MUCH LARGE FOR THE SHM**\n");
      //printf("*******************************\n");

    }
    address[mem_key]=tmp_address;
    
    //    printf("Waiting for the Flag to give a signal\n");
    temp_flag=-666;

    /*
     * We set some values not to crash
     */
    while (flag[0]!=666) {
      read_mem_int(&flag,mem_key,n_bytes);

      /*
       * 
       *
       */


      /*
       * Ask the server if it is alive
       * FLAG[0]=666 -> COMMAND
       * FLAG[1]=1 temp flag: indicates the client that can read
       *           the client wait until it is set to 1
       */
      if ((flag[0]==666)&&(flag[2]!=-666)) {
	/*
	 * We send the signal to the client
	 */
	flag[1]=999;
	flag[2]=-999;
	load_mem_int(flag,mem_key,n_bytes,&tmp_address);				address[mem_key]=tmp_address;
      }



      /*
       * To I/O only one spectra.
       * FLAG[0]=1 -> COMMAND
       * FLAG[1]=1 temp flag: indicates the client that can read
       *           the client wait until it is set to 1
       * FLAG[2]= Spectra to load in SHM
       */
      if ((flag[0]==1)&&(flag[2]!=-1)) {
	// printf("FLAG[0]=1\n");
	/*
	 * We read the proper information to  be
	 * loaded
	 */
 	raw_spec = (float *)malloc(npix*sizeof(float));
	for (j=0;j<npix;j++) {
	  raw_spec[j]=raw_data[j+flag[2]*npix];
	}
	load_mem_float(raw_spec,2,npix,&tmp_address);
	address[2]=tmp_address;
	// printf("load Spectra %d in SHM \n",flag[2]);
	free(raw_spec);
	/*
	 * We send the signal to the client
	 */
	flag[1]=1;
	flag[2]=-1;
	load_mem_int(flag,mem_key,n_bytes,&tmp_address);				// printf("FLAG[1]=0\n");
	address[mem_key]=tmp_address;
	//	printf("No paso\n");
      }
      /*
       * We load in SHM a series of Spectra
       * it is faster
       * FLAG[0]=2 -> COMMAND
       * FLAG[1]=1 temp flag: indicates the client that can read
       *           the client wait until it is set to 1
       * FLAG[2]= initial spectra to load in SHM
       * FLAG[3]= Last spectra to load in SHM
       */
      
      if ((flag[0]==2)&&(flag[2]!=-2)) {
	// printf("FLAG[0]=1\n");
	/*
	 * We read the proper information to  be
	 * loaded
	 */	


	shmctl(address[2], IPC_RMID, 0);
	raw_spec = (float *)malloc(npix*(flag[3]-flag[2])*sizeof(float));
	for (j=0;j<npix*(flag[3]-flag[2]);j++) {
	  raw_spec[j]=raw_data[j+flag[2]*npix];
	}
	load_mem_float(raw_spec,2,npix*(flag[3]-flag[2]),&tmp_address);
	free(raw_spec);
	address[2]=tmp_address;


	/*
	 * We send the signal to the client
	 */
	flag[1]=1;
	flag[2]=-2;
	load_mem_int(flag,mem_key,n_bytes,&tmp_address);
	address[mem_key]=tmp_address;

      }

      /*
       * This FLAG indicates the server to load a 
       * new E3D_image in the memory
       * and upload all the descriptors to the SHM
       * to be readed by the client.
       *
       * it is faster
       * FLAG[0]=3 -> COMMAND
       * FLAG[1]=1 temp flag: indicates the client that can read
       *           the client wait until it is set to 1
       */      
      if ((flag[0]==3)&&(flag[2]!=-3)) {
	/*
	 * We read the FILE_NAME from the SHM
	 * KEY=4
	 * We assume a maximun length for a file of 80 characters.
	 */
	// printf("Paso Server\n");
	if (read_mem_char(&input_filename,4,80)!=0) {
	  //printf("No file in the SHM to be upload\n");
	  flag[1]=-1;
	  flag[2]=-3;
	} else {
	  // printf("LOADING FILE=");
	  // printf(input_filename);
	  // printf("...\n");
	  load_e3d(input_filename,&e3d_image,&n_spaxels, &spaxels, &index_start, &npts, &raw_data, &raw_noise);
	  // printf("He leido el fichero\n");
	  nb_spec=e3d_image.nbspec;  
	  n_groups=e3d_image.ngroups;
	  npix=e3d_image.common_parameters[2];	 	  
	  load_mem_all(e3d_image,n_spaxels,spaxels,index_start,npts,&address);
	  // printf("He Cargado las cosas en memoria\n");
	  min=3000000;
	  max=-3000000;
	  for (j=0;j<npix*nb_spec;j++) {
	    if (min>raw_data[j]) 
	      min=raw_data[j];
	    if (max<raw_data[j]) 
	      max=raw_data[j];
	  }	      	 

	  flag[1]=1;
	  flag[2]=-3;
	}
	  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
	  address[mem_key]=tmp_address;
	  temp_flag=flag[2];

      }
      
      /*
       * FLAG[0]=4 -> CLEAN THE SHM
       * FLAG[1]=1 temp flag: indicates the client that can read
       *           the client wait until it is set to 1       
       *
       */      
      if ((flag[0]==4)&&(flag[2]!=-4)) {
	for (j=1;j<10;j++) {
	  if (!((shmid = shmget(j, 1, 0666)) < 0)) { 
	    shmctl(shmid, IPC_RMID, NULL);
	  }
	}
	flag[1]=1;
	flag[2]=-4;
	//	printf("flag={%d,%d,%d}\n",flag[0],flag[1],flag[3]);
	load_mem_int(flag,mem_key,n_bytes,&tmp_address);    
	address[mem_key]=tmp_address;
      }
    

      /*
       * We load in SHM an SLICE
       * it is faster
       * FLAG[0]=5 -> Get a SLICE at a certain wavelength
       * FLAG[1]=1 temp flag: indicates the client that can read
       *           the client wait until it is set to 1
       * FLAG[2]= The Wavelength index of the CUT
       */
      
      if ((flag[0]==5)&&(flag[2]!=-5)) {
	// printf("FLAG[0]=1\n");
	/*
	 * We read the proper information to  be
	 * loaded
	 */	
	shmctl(address[2], IPC_RMID, 0);
	raw_spec = (float *)malloc(nb_spec*sizeof(float));
	for (j=0;j<nb_spec;j++) {
	  raw_spec[j]=raw_data[flag[2]+j*npix];
	}
	load_mem_float(raw_spec,2,nb_spec,&tmp_address);
	free(raw_spec);
	address[2]=tmp_address;
	/*
	 * We send the signal to the client
	 */
	flag[1]=1;
	flag[2]=-5;
	load_mem_int(flag,mem_key,n_bytes,&tmp_address);
	address[mem_key]=tmp_address;
      }

      /*
       * We load in SHM a series of SLICES
       * it is faster
       * FLAG[0]=6 -> Get a SLICE at a certain wavelength
       * FLAG[1]=1 temp flag: indicates the client that can read
       *           the client wait until it is set to 1
       * FLAG[2]= The Wavelength index of the CUT
       */
      
      if ((flag[0]==6)&&(flag[2]!=-6)) {
	/*
	 * We read the proper information to  be
	 * loaded
	 */	
	shmctl(address[2], IPC_RMID, 0);
	raw_spec = (float *)malloc((flag[3]-flag[2])*nb_spec*sizeof(float));
	for (j=0;j<(flag[3]-flag[2])*nb_spec;j++) {
	  raw_spec[j]=raw_data[flag[2]+j*npix];
	}
	load_mem_float(raw_spec,2,(flag[3]-flag[2])*nb_spec,&tmp_address);
	free(raw_spec);
	address[2]=tmp_address;
	/*
	 * We send the signal to the client
	 */
	flag[1]=1;
	flag[2]=-6;
	load_mem_int(flag,mem_key,n_bytes,&tmp_address);
	address[mem_key]=tmp_address;
      }


      /*
       * We load in SHM a series of SLICES
       * it is faster
       * FLAG[0]=7 -> Get a certain point intensity
       * FLAG[1]=1 temp flag: indicates the client that can read
       *           the client wait until it is set to 1
       * FLAG[2]= The Wavelength index of the CUT.
       * FLAG[3]= The SpectraID.
       */
      
      if ((flag[0]==7)&&(flag[2]!=-7)) {
	/*
	 * We read the proper information to  be
	 */	
	shmctl(address[2], IPC_RMID, 0);
	raw_spec = (float *)malloc(1*sizeof(float));
	raw_spec[0]=raw_data[flag[2]+flag[3]*npix];
	load_mem_float(raw_spec,2,1,&tmp_address);
	free(raw_spec);
	address[2]=tmp_address;
	/*
	 * We send the signal to the client
	 */
	flag[1]=1;
	flag[2]=-7;
	load_mem_int(flag,mem_key,n_bytes,&tmp_address);
	address[mem_key]=tmp_address;
      }

    

      /*************************
       * Painting Flags
       *************************/

      /*
       * Change the Palette
       *
       * it is faster
       * FLAG[0]=100 -> COMMAND
       * FLAG[1]=1 temp flag: indicates the client that can read
       *           the client wait until it is set to 1
       */      
      if ((flag[0]==100)&&(flag[2]!=-100)) {
	/*
	 * We read the FILE_NAME from the SHM
	 * KEY=4
	 * We assume a maximun length for a file of 80 characters.
	 */
	if (read_mem_char(&commands,4,80)!=0) {
	  //printf("No Commands found\n");
	  flag[1]=-1;
	  flag[2]=-100;
	} else {
	  split_text(commands," ",&n_tokens,&tokens);
	  p=atoi(tokens[0]);
	  bright=atof(tokens[1]);
	  contra=atof(tokens[2]);
	  sign=atof(tokens[3]);
	  cpgslct(atoi(tokens[4]));
	  palette(p,sign*contra,bright);
	  flag[1]=1;
	  flag[2]=-100;
	}
	  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
	  address[mem_key]=tmp_address;
	  temp_flag=flag[2];

      }
      


      /*
       * Open a Device
       *
       * it is faster
       * FLAG[0]=101 -> COMMAND
       * FLAG[1]=1 temp flag: indicates the client that can read
       *           the client wait until it is set to 1
       */      
      if ((flag[0]==101)&&(flag[2]!=-101)) {
	//printf("Paso por aqui?\n");
	/*
	 * We read the FILE_NAME from the SHM
	 * KEY=4
	 * We assume a maximun length for a file of 80 characters.
	 */
	if (read_mem_char(&commands,4,80)!=0) {
	  //printf("No Commands found\n");
	  flag[1]=-1;
	  flag[2]=-101;
	} else {
	  //printf("commands='%s'\n",commands);
	  flag[3] = cpgopen(commands);
	  flag[1]=1;
	  flag[2]=-101;
	}
	  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
	  address[mem_key]=tmp_address;
	  temp_flag=flag[2];

      }
      

      /*
       * Plot Staked Spectra
       *
       * it is faster
       * FLAG[0]=102 -> COMMAND
       * FLAG[1]=1 temp flag: indicates the client that can read
       *           the client wait until it is set to 1
       */      
      if ((flag[0]==102)&&(flag[2]!=-102)) {
	/*
	 * We read the FILE_NAME from the SHM
	 * KEY=4
	 * We assume a maximun length for a file of 80 characters.
	 */
	if (read_mem_char(&commands,4,80)!=0) {
	  //printf("No Commands found\n");
	  flag[1]=-1;
	  flag[2]=-102;
	} else {
	  //	  split_text(commands," ",&n_tokens,&tokens);
	  //	  image_plot(raw_data,min,max,npix,nb_spec,atoi(commands),tr_now);
	  flag[1]=1;
	  flag[2]=-102;
	}
	  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
	  address[mem_key]=tmp_address;
	  temp_flag=flag[2];

      }
      





      /*
       * Plot Staked Spectra
       *
       * it is faster
       * FLAG[0]=103 -> COMMAND
       * FLAG[1]=1 temp flag: indicates the client that can read
       *           the client wait until it is set to 1
       */      
      if ((flag[0]==103)&&(flag[2]!=-103)) {
	/*
	 * We read the FILE_NAME from the SHM
	 * KEY=4
	 * We assume a maximun length for a file of 80 characters.
	 */
	if (read_mem_char(&commands,4,80)!=0) {
	  //printf("No Commands found\n");
	  flag[1]=-1;
	  flag[2]=-103;
	} else {
	  //printf("COMMANDS='%s'\n",commands);
	  split_text(commands," ",&n_tokens,&tokens);
	  image_plot(raw_data,min,max,npix,nb_spec,atoi(tokens[0]),tr_now);
	  flag[1]=1;
	  flag[2]=-103;
	}
	  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
	  address[mem_key]=tmp_address;
	  temp_flag=flag[2];

      }
      



      /*********************************************
       * ALL THE COMMANDS SHOULD BE BEFORE THIS!!!
       *********************************************
       * We delete the FLAG SHARED MEMORY
       * and if I try to read then I will have an error!
       */
      mem_size=15*sizeof(float);            
      shmid = shmget(mem_key, mem_size, 0666);
      shmctl(shmid, IPC_RMID, NULL);
      while (!((shmid = shmget(mem_key, mem_size, IPC_EXCL | IPC_CREAT | 0666))<0)) {
	shmctl(shmid, IPC_RMID, NULL);
	sleep(1);
      }


    }

    
    free(raw_data);
    
    /*
     * We clean the SHARED memory
     */
    shmdt(address[1]);
    shmdt(address[2]);
    shmdt(address[3]);
    shmdt(address[4]);
    shmdt(address[5]);
    shmdt(address[6]);
    shmdt(address[7]);
    shmdt(address[8]);
    shmdt(address[9]);
    shmctl(address[1], IPC_RMID, NULL);
    shmctl(address[2], IPC_RMID, 0);
    shmctl(address[3], IPC_RMID, 0);
    shmctl(address[4], IPC_RMID, 0);
    shmctl(address[5], IPC_RMID, 0);
    shmctl(address[6], IPC_RMID, 0);
    shmctl(address[7], IPC_RMID, 0);
    shmctl(address[8], IPC_RMID, 0);
    shmctl(address[9], IPC_RMID, 0);
    exit_session(0);
    return(0);	
    
}





/*********************************************
 * A basic function that load an E3D file and the
 * basic parameters needed to handle the image
 *********************************************/

int load_e3d(char *filename, E3D_file *e3d_frame, int **n_spaxels, SPAXEL ***spaxels, int **index_start, int **npts, float **raw_data, float **raw_noise) {

  E3D_file e3d_temp;
  int *n_spaxels_temp;
  SPAXEL **spaxels_temp;
  int *index_start_temp;
  int *npts_temp;
  float *raw_data_temp;
  float *raw_noise_temp;


  int i,j,k;
  int nb_spec, npix, specId;
  int *nol_temp, max_npts;
  float start_w,end_w,delta_w;
  SPECTRUM signal,noise;
  SPECTRUM *spectra_signal_temp,*spectra_noise_temp;

  // printf("LOAD_E3D\n");
  // printf("%s OPEN\n",filename); 
  open_E3D_frame(&e3d_temp,filename,"I");
  // printf("Ok\n");
  nb_spec=e3d_temp.nbspec;
  // printf("nb_spec=%d,%d Ok\n",nb_spec,e3d_temp.nbspec);
  spaxels_temp = (SPAXEL *) calloc(nb_spec, sizeof(SPAXEL));
  n_spaxels_temp = (int *) calloc(nb_spec, sizeof(int));
  index_start_temp = (int *) calloc(nb_spec, sizeof(int));
  npts_temp = (int *) calloc(nb_spec, sizeof(int));
  spectra_signal_temp = (SPECTRUM *) calloc(nb_spec, sizeof(SPECTRUM));
  spectra_noise_temp = (SPECTRUM *) calloc(nb_spec, sizeof(SPECTRUM));
  nol_temp = (int *) calloc(nb_spec, sizeof(int));

  /*
   * Check if it has common paramters!
   */

  if (has_common_bounds(&e3d_temp)) {
    // printf("Has common Bounds\n");
  } else {
    // printf("It has no common Bounds\n");
  }

  // printf("common=%d\n",has_common_bounds(&e3d_temp));
  /*
   * Reading common parameters
   */
  // printf("Reading common parameters\n");
  start_w= (float) e3d_temp.common_parameters[0];
  end_w= (float) e3d_temp.common_parameters[1];
  npix= floor(e3d_temp.common_parameters[2]);

  //  get_common_param(&e3d_temp,&npix,&start_w,&end_w);	
  delta_w=(end_w-start_w)/npix;

  // printf("%d,%f,%f\n Ok\n",npix,start_w,end_w);
  raw_data_temp = (float *) calloc(nb_spec*npix, sizeof(float));
  raw_noise_temp = (float *) calloc(nb_spec*npix, sizeof(float));




  get_spectra_ID(&e3d_temp,nol_temp);

  max_npts=0;

  /*
   * We read the spectra one by one
   */
  for (specId=1;specId<=nb_spec;specId++) {	 
    /*
      We should read the 'index_start' when available
      in the new format.
      By now it is set to 0.
    */
    index_start_temp[specId-1]=0;
    /*
      We get the spaxels information!!!
    */
    n_spaxels_temp[specId-1]=get_E3D_spaxels(&e3d_temp,specId,&spaxels_temp[specId-1]);    




    /*
      We get the spectra information.
    */
    get_E3D_spec(&e3d_temp,&signal,NULL,nol_temp[specId-1]); 
    spectra_signal_temp[specId-1]=signal;
    npts_temp[specId-1]=signal.npts;

    if (signal.npts>max_npts) {
      max_npts=signal.npts;
    }
  }



  //  printf("Ok***\n");
  /*
   * We build the RAW_DATA image
   */

  for (specId=1;specId<=nb_spec;specId++) {	 
    for (j=0;j<max_npts;j++) {
      //printf("specId=%d,j=%d\n",specId,j);
      if (j<index_start_temp[specId-1]) {
	raw_data_temp[j+(specId-1)*npix]=0.0;
      } else {
	raw_data_temp[j+(specId-1)*npix]=RD_spec(&spectra_signal_temp[specId-1],j-index_start_temp[specId]-1);
	raw_noise_temp[j+(specId-1)*npix]=sqrt(fabs(raw_data_temp[j+(specId-1)*npix]));
		
      }
    }		
  }


  // printf("Ok NOISE\n");





  *e3d_frame=e3d_temp;  
  *n_spaxels=n_spaxels_temp;
  *spaxels=spaxels_temp;
  *index_start=index_start_temp;
  *npts=npts_temp;
  *raw_data=raw_data_temp;
  *raw_noise=raw_data_temp;

  // printf("Closing File\n");
  close_E3D_frame(&e3d_temp);
  // printf("Ok\n");
  return(0);
}





/********************************
 * Program that reads an Slice of data
 * as a float array.
 * The array contain from 0, NB_SPEC-1 data
 * containing a Wavelength cut.
 ********************************/
float* read_spax_int(int i_spec, int nb_spec, int npix) {    
  int key=2; // We read from the RAW_data
  int mem_size;
  int shmid;
  int j;
  float *shm,*spax_int;
  int size;
  size=nb_spec*npix;
  mem_size=size*sizeof(float);

    /*
     * Locate the segment.
     */
    if ((shmid = shmget(key, mem_size, 0666)) < 0) {
        perror("shmget");
        return(1);
    }
    //printf("MEM=%d\n",shmid);
    /*
     * Now we attach the segment to our data space.
     */
    
    if ((shm = shmat(shmid, NULL, 0)) == (float *) -1) {
        perror("shmat");
        return(1);
    }

    spax_int = (float *)malloc(nb_spec*sizeof(float));    
    if (i_spec<0) i_spec=0;
    if (i_spec>=npix) i_spec=npix-1;

    for (j=0;j<nb_spec;j++) {
      spax_int[j]=shm[i_spec+j*npix];
    }

    return spax_int;
}







/********************************
 * This function reads the groups from
 * the SHM
 ********************************/
//GROUP* read_mem_group(int key, int size) {
int read_mem_group(GROUP **group,int key, int size) {
    
  int mem_size;
  int shmid;
  int j;
  GROUP* shm;
  
  mem_size=size*sizeof(GROUP);

    /*
     * Locate the segment.
     */
    if ((shmid = shmget(key, mem_size, 0666)) < 0) {
        perror("shmget");
        return(1);
    }
    //printf("MEM=%d\n",shmid);
    /*
     * Now we attach the segment to our data space.
     */
    
    if ((shm = shmat(shmid, NULL, 0)) == (GROUP *) -1) {
        perror("shmat");
        return(1);
    }
    //    return shm;
    *group=shm;
    return(0);
}


/********************************
 * Function that load a GROUP array to the memory
 ********************************/

int load_mem_group(GROUP *grupo, int key, int size, int *address) {

  int mem_size;
    int shmid;
    GROUP *shm,*s;
    int j;

    mem_size=size*sizeof(GROUP);
    /*
     * Creating the Segment
     */
    if ((shmid = shmget(key, mem_size, IPC_CREAT | 0666)) < 0) {
        perror("shmget");
        return(1);
    }

    *address=shmid;
    //printf("MEM=%d, key=%d\n",mem_size,key); 
    /*
     * Now we attach the segment to our data space.
     */
    if ((shm = shmat(shmid, NULL, 0)) == (GROUP *) -1) {
        perror("shmat");
        return(1);
    }

    s=shm;

    for (j=0;j<size;j++) {
      s[j]=grupo[j];
    }

    return(0);
}



/********************************
 * Function that read a INT array from the memory.
 * Used to write in memory the 2D images of staked
 * spectra for the QUALITY
 * or the N_SPAXELS.
 ********************************/
int read_mem_int(int **out_int, int key, int size) {
    
  int mem_size;
  int shmid;
  int j;
  int* shm;
  
  mem_size=size*sizeof(int);

    /*
     * Locate the segment.
     */
    if ((shmid = shmget(key, mem_size, 0666)) < 0) {
      perror("shmget");
        return(1);
    }
    //printf("MEM=%d\n",shmid);
    /*
     * Now we attach the segment to our data space.
     */
    
    if ((shm = shmat(shmid, NULL, 0)) == (int *) -1) {
              perror("shmat");
        return(2);
    }
    //    return shm;
    *out_int=shm;
    return(0);
}


/********************************
 * Function that load a int array to the memory
 ********************************/

int load_mem_int(int *raw_data, int key, int size, int* address) {

  int mem_size;
    int shmid;
    int *shm,*s;
    int j;

    mem_size=size*sizeof(int);
    /*
     * Creating the Segment
     */
    if ((shmid = shmget(key, mem_size, IPC_CREAT | 0666)) < 0) {
        perror("shmget");
        return(1);
    }
    *address=shmid;
    //printf("MEM=%d, key=%d\n",mem_size,key); 
    /*
     * Now we attach the segment to our data space.
     */
    if ((shm = shmat(shmid, NULL, 0)) == (int *) -1) {
        perror("shmat");
        return(1);
    }

    /*
     * Now put some things into the memory for the
     * other process to read.
     */
    //    shm=raw_data;

    //    raw_data=shm;
    s=shm;

    for (j=0;j<size;j++) {
      s[j]=raw_data[j];
    }

    return(0);
}



/********************************
 * Function that read a CHAR array from the memory.
 * Used to write in memory the 2D images of staked
 * spectra for the QUALITY
 * or the N_SPAXELS.
 ********************************/
int  read_mem_char(char **out_shm,int key, int size) {
    
  int mem_size;
  int shmid;
  int j;
  char* shm;
  
  mem_size=size*sizeof(char);

    /*
     * Locate the segment.
     */
    if ((shmid = shmget(key, mem_size, 0666)) < 0) {
        perror("shmget");
        return(1);
    }
    //printf("MEM=%d, key=%d\n",mem_size,key); 
    /*
     * Now we attach the segment to our data space.
     */
    
    if ((shm = shmat(shmid, NULL, 0)) == (char *) -1) {
        perror("shmat");
        return(1);
    }
    *out_shm=shm;
    return(0);
}


/********************************
 * Function that load a char array to the memory
 ********************************/

char load_mem_char(char *raw_data, int key, int size, int* address) {

  int mem_size;
  int shmid;
  char *shm,*s;
  int j;
  
  mem_size=size*sizeof(char);
  /*
   * Creating the Segment
   */

  // printf("Paso %s\n",raw_data);
  if ((shmid = shmget(key, mem_size, IPC_CREAT | 0666)) < 0) {
    perror("shmget");
    return(1);
  }
  *address=shmid;
  //printf("MEM=%d, key=%d\n",mem_size,key); 
  /*
   * Now we attach the segment to our data space.
   */
  if ((shm = shmat(shmid, NULL, 0)) == (char *) -1) {
    perror("shmat");
        return(1);
  }
  
  /*
   * Now put some things into the memory for the
   * other process to read.
   */
  //    shm=raw_data;
  
  //    raw_data=shm;
  s=shm;
  
  for (j=0;j<size;j++) {
    s[j]=raw_data[j];
  }
  
  // printf("FILE='%s','%s'\n",raw_data,s);

  return(0);
}




/********************************
 * Function that read an array of Spaxels.
 ********************************/
//SPAXEL** read_mem_spaxels(int* n_spaxels,int key, int size) {
    
int read_mem_spaxels(SPAXEL ***out_spax, int* n_spaxels, int key, int size) {

  int mem_size,real_size;
  int shmid;
  int j,i,k;
  SPAXEL *shm,**out_spaxels;

  
  real_size=1;
  for (j=0;j<size;j++) {
    for (i=0;i<n_spaxels[j-1];i++) {
      real_size++;
    }
  }
  

  
  mem_size=real_size*sizeof(SPAXEL);

    if ((shmid = shmget(key, mem_size, 0666)) < 0) {
        perror("shmget");
        return(1);
    }
    //printf("MEM=%d, key=%d\n",mem_size,key); 
    
    if ((shm = shmat(shmid, NULL, 0)) == (SPAXEL *) -1) {
        perror("shmat");
        return(1);
    }

    //    printf("size=%d,real_size=%d\n",size,real_size);

    out_spaxels = (SPAXEL **)malloc(size*sizeof(SPAXEL*));    
    k=0;
    for (j=0;j<size;j++) {
      out_spaxels[j] = (SPAXEL *)malloc(n_spaxels[j]*sizeof(SPAXEL));
      for (i=0;i<n_spaxels[j];i++) {
	out_spaxels[j][i]=shm[k];
	k++;
      }
    }
    
  
    //    return out_spaxels;
    *out_spax=out_spaxels;
    return(0);
}



/****************************************************
 * This function is used to load in memory
 * the SPAXELS.
 ****************************************************/

int load_mem_spaxels(SPAXEL **spaxels, int* n_spaxels, int key, int size, int* address) {
      
  int mem_size,real_size;
  int shmid;
  int i,j,k;
  SPAXEL *shm,*s;
  
  real_size=1;
  for (j=0;j<size;j++) {
    for (i=0;i<n_spaxels[j];i++) {
      real_size++;
    }
  }


  mem_size=real_size*sizeof(SPAXEL);
  
  /*
   * Creating the Segment
   */
  
  if ((shmid = shmget(key, mem_size, IPC_CREAT | 0666)) < 0) {
    perror("shmget");
    return(1);
  }
  
  *address=shmid;
    //printf("MEM=%d, key=%d\n",mem_size,key); 
  /*
   * Now we attach the segment to our data space.
   */
  
  if ((shm = shmat(shmid, NULL, 0)) == (SPAXEL *) -1) {
    perror("shmat");
    return(1);
  }
  
  
  /*
   * Now put some things into the memory for the
   * other process to read.
   */
  
  s=shm;
  
  k=0;

  //  s = (SPAXEL *)malloc(real_size*sizeof(SPAXEL));

  for (j=0;j<size;j++) {
    for (i=0;i<n_spaxels[j];i++) {
      s[k]=spaxels[j][i];
      k++;
    }
  }
  
  
  return(0);

}

/********************************
 * Function that read a float array from the memory.
 * Used to write in memory the 2D images of staked
 * spectra, both the SIGNAL, QUALITY, and NOISE
 ********************************/
//float* read_mem_float(int key, int size) {
int read_mem_float(float **out_float, int key, int size) {
    
  int mem_size;
  int shmid;
  int j;
  float* shm;
  
  mem_size=size*sizeof(float);

    /*
     * Locate the segment.
     */
    if ((shmid = shmget(key, mem_size, 0666)) < 0) {
        perror("shmget");
        return(1);
    }
    //printf("MEM=%d, key=%d\n",mem_size,key); 
    /*
     * Now we attach the segment to our data space.
     */
    
    if ((shm = shmat(shmid, NULL, 0)) == (float *) -1) {
        perror("shmat");
        return(1);
    }
    *out_float=shm;
    return(0);
}


/********************************
 * Function that load a float array to the memory
 ********************************/

int load_mem_float(float *raw_data, int key, int size, int* address) {

  int mem_size;
    int shmid;
    float *shm,*s;
    int j;

    mem_size=size*sizeof(float);
    /*
     * Creating the Segment
     */
    if ((shmid = shmget(key, mem_size, IPC_CREAT | 0666)) < 0) {
        perror("shmget");
        return(1);
    }

    *address=shmid;
    //printf("MEM=%d, key=%d\n",mem_size,key); 
    /*
     * Now we attach the segment to our data space.
     */
    if ((shm = shmat(shmid, NULL, 0)) == (float *) -1) {
        perror("shmat");
        return(1);
    }

    /*
     * Now put some things into the memory for the
     * other process to read.
     */
    //    shm=raw_data;

    //    raw_data=shm;
    s=shm;

    for (j=0;j<size;j++) {
      s[j]=raw_data[j];
    }

    return(0);
}



/********************************
 * Function that read an 2D image from memory
 ********************************/
float* read_mem_staked_spectra(int key, int size) {
    
  int mem_size;
  int shmid;
  int j;
  float* shm;
  
  mem_size=size*sizeof(float);
    /*
     * Locate the segment.
     */
    if ((shmid = shmget(key, mem_size, 0666)) < 0) {
        perror("shmget");
        return(1);
    }
    //printf("MEM=%d, key=%d\n",mem_size,key); 
    /*
     * Now we attach the segment to our data space.
     */
    
    if ((shm = shmat(shmid, NULL, 0)) == (float *) -1) {
        perror("shmat");
        return(1);
    }
    return shm;
}


/********************************
 * Function that load an E3D_file into memory
 ********************************/

int load_mem_staked_spectra(float *raw_data, int key, int size, int* address) {

  int mem_size;
    int shmid;
    float *shm,*s;
    int j;

    mem_size=size*sizeof(float);


    //  printf("mem_size=%d\n",mem_size);

    /*
     * Creating the Segment
     */
  //    if ((shmid = shmget(key, mem_size, IPC_CREAT | 0666)) < 0) {
  if ((shmid = shmget(key, mem_size, IPC_CREAT | 0666)) < 0) {
    perror("shmget");
    return(1);
  }

    *address=shmid;
    //printf("MEM=%d, key=%d\n",mem_size,key); 
    /*
     * Now we attach the segment to our data space.
     */
    if ((shm = shmat(shmid, NULL, 0)) == (float *) -1) {
        perror("shmat");
        return(1);
    }

    /*
     * Now put some things into the memory for the
     * other process to read.
     */
    //    shm=raw_data;

    //    raw_data=shm;
    s=shm;

    for (j=0;j<size;j++) {
      s[j]=raw_data[j];
    }

    return(0);
}



/**************************************************
 * Function that load a 2D image to the Memory
 **************************************************/

int load_mem_e3d(E3D_file e3d_file, int key, int *address) {

    int mem_size=sizeof(E3D_file);
    int shmid;
    E3D_file *shm,*s;
    int j;

    /*
     * Creating the Segment
     */
    if ((shmid = shmget(key, mem_size, IPC_CREAT | 0666)) < 0) {
        perror("shmget");
        return(1);
    }

    *address=shmid;
    //printf("MEM=%d, key=%d\n",mem_size,key); 
    /*
     * Now we attach the segment to our data space.
     */
    if ((shm = shmat(shmid, NULL, 0)) == (char *)-1) {
        perror("shmat");
        return(1);
    }

    /*
     * Now put some things into the memory for the
     * other process to read.
     */
    s=shm;
    *s=e3d_file;
    return(0);
}

/********************************
 * Function that read an E3D_file from memory
 ********************************/

int read_mem_e3d(E3D_file* e3d_out, int key) {
    
    int mem_size=sizeof(E3D_file);
    int shmid;
    E3D_file *shm;

    int j;

    if ((shmid = shmget(key, mem_size, 0666)) < 0) {
        perror("shmget");
        return(1);
    }
    //printf("MEM=%d, key=%d\n",mem_size,key); 
    
    if ((shm = shmat(shmid, NULL, SHM_RDONLY)) == (char *) -1) {
        perror("shmat");
        return(1);
    }
    
    *e3d_out=*shm;

    return(0);
}

/******************************************
 * Function to Change the palett
 ******************************************/

int palette(int type, float contra, float bright){
  
  //  printf("type=%d, %f,%f\n",type,contra,bright);

  float GL[]={0.0, 1.0};
  float GR[]={0.0, 1.0};
  float GG[]={0.0, 1.0};
  float GB[]={0.0, 1.0};
  
  float RL[]={-0.5, 0.0, 0.17, 0.33, 0.50, 0.67, 0.83, 1.0, 1.7};
  float RR[]={ 0.0, 0.0,  0.0,  0.0,  0.6,  1.0,  1.0, 1.0, 1.0};
  float RG[]={ 0.0, 0.0,  0.0,  1.0,  1.0,  1.0,  0.6, 0.0, 1.0};
  float RB[]={ 0.0, 0.3,  0.8,  1.0,  0.3,  0.0,  0.0, 0.0, 1.0};
  
  float HL[]={0.0, 0.2, 0.4, 0.6, 1.0};
  float HR[]={0.0, 0.5, 1.0, 1.0, 1.0};
  float HG[]={0.0, 0.0, 0.5, 1.0, 1.0};
  float HB[]={0.0, 0.0, 0.0, 0.3, 1.0};
  
  float WL[]={0.0, 0.5, 0.5, 0.7, 0.7, 0.85, 0.85, 0.95, 0.95, 1.0};
  float WR[]={0.0, 1.0, 0.0, 0.0, 0.3,  0.8,  0.3,  1.0,  1.0, 1.0};
  float WG[]={0.0, 0.5, 0.4, 1.0, 0.0,  0.0,  0.2,  0.7,  1.0, 1.0};
  float WB[]={0.0, 0.0, 0.0, 0.0, 0.4,  1.0,  0.0,  0.0, 0.95, 1.0};
  
  float AL[]={0.0, 0.1, 0.1, 0.2, 0.2, 0.3, 0.3, 0.4, 0.4, 0.5,
     0.5, 0.6, 0.6, 0.7, 0.7, 0.8, 0.8, 0.9, 0.9, 1.0};
  float AR[]={0.0, 0.0, 0.3, 0.3, 0.5, 0.5, 0.0, 0.0, 0.0, 0.0,
	      0.0, 0.0, 0.0, 0.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0};
  float AG[]={0.0, 0.0, 0.3, 0.3, 0.0, 0.0, 0.0, 0.0, 0.8, 0.8,
	      0.6, 0.6, 1.0, 1.0, 1.0, 1.0, 0.8, 0.8, 0.0, 0.0};
  float AB[]={0.0, 0.0, 0.3, 0.3, 0.7, 0.7, 0.7, 0.7, 0.9, 0.9,
	      0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};

  switch(type) {
  case (1): {
    /* Grey */
    cpgctab(GL, GR, GG, GB, 2, contra, bright);
    break;
  }
  case (2): {
    /* Rainbow */
    cpgctab(RL, RR, RG, RB, 9, contra, bright);
    break;
  }
  case (3): {
    /* Heat */
    cpgctab(HL, HR, HG, HB, 5, contra, bright);
    break;
  }
  case (4): {
    /* Weird IRAF */
    cpgctab(WL, WR, WG, WB, 10, contra, bright);
    break;
  }
  case (5): {
    /* AIPS */
    cpgctab(AL, AR, AG, AB, 20, contra, bright);
    break;
  }
  default: {
    cpgctab(RL, RR, RG, RB, 9, contra, bright);
    //    cpgctab(GL, GR, GG, GB, 2, contra, bright);
    break;
  }
  }

  return(0);  
}

void setVP() {
  float d, vpx1, vpx2, vpy1, vpy2;
  cpgsvp(0.0,1.0,0.0,1.0);
  cpgqvp(1,&vpx1,&vpx2,&vpy1,&vpy2);
  d=Minimun(vpx2-vpx1,vpy2-vpy1)/40.0;
  vpx1=vpx1+5.0*d;
  vpx2=vpx2-2.0*d;
  vpy1=vpy1+8.0*d;
  vpy2=vpy2-2.0*d;
  cpgvsiz(vpx1,vpx2,vpy1,vpy2);
  return;
}

float Minimun(float a, float b) {
  float c;
  if (a<b) {
    c=a;
  } else {
    c=b;
  }
  return c;
}

float Maximun(float a, float b) {
  float c;
  if (a>b) {
    c=a;
  } else {
    c=b;
  }
  return c;
}


/*****************************************
 * Plot Staked Spectra
 *****************************************/
int draw_staked_spectra(float max, float min, int **nnaxes, float **out) {

  int delta_n=512;
  E3D_file e3d_image;
  SPAXEL **spaxels;
  int *n_spaxels,*index_start,*npts;
  GROUP *groups;
  double start_w, end_w, delta_w;
  int npix;
  int nb_spec,n_groups;
  float *raw_data;
  int naxes[2];
  int j;
  float datamin,datamax;
  float tr[]={0.0,1.0,0.0,0.0,0.0,1.0};
  float minmax[2];

  ask_e3d_info(&e3d_image,&n_spaxels,&spaxels,&index_start,&npts,&groups);
  start_w=e3d_image.common_parameters[0];
  end_w=e3d_image.common_parameters[1];
  npix=e3d_image.common_parameters[2];
  delta_w=(end_w-start_w)/npix;
  nb_spec=e3d_image.nbspec;
  n_groups=e3d_image.ngroups;
  raw_data = (float *)malloc(nb_spec*n_groups*sizeof(float));
  ask_raw_data(&raw_data,nb_spec,npix,delta_n);
  naxes[0]=npix;
  naxes[1]=nb_spec;

  //  *nnaxes = (int *)malloc(2*sizeof(int));
  *nnaxes=naxes;


  
  datamin=3000000.0;
  datamax=-3000000.0;
  for (j=0;j<npix*nb_spec;j++) {
    if (datamin>raw_data[j]) 
      datamin=raw_data[j];
    if (datamax<raw_data[j]) 
      datamax=raw_data[j];
  }	      	 
  datamax=2.5*datamax;
  datamin=0.5*datamin;
  if (min!=0.0) datamin=min;
  if (max!=0.0) datamax=max;
  
  minmax[0]=datamin;
  minmax[1]=datamax;
  //  printf("MINMAX=%f,%f\n",datamin,datamax);

  *out=minmax;

  

  //  cpgslct(device_id);
  cpgpage();
  cpgvstd();
  cpgwnad(0.0,1.0+naxes[0],0.0,1.0+naxes[1]);
  cpgimag(raw_data,naxes[0],naxes[1],1,naxes[0],1,naxes[1],datamin,datamax,tr);
  cpgsch(0.6);
  cpgbox("bcntsi",0.0,0,"bcntsiv",0.0,0);
  cpgmtxt("b",3.0,1.0,1.0,"pixel number");
  cpgwedg("BI",4.0,5.0,datamin,datamax,"pixel value");
  cpgsch(1.0);
  free(raw_data);
  return(0);
}



/******************************************
 * Function to Plot an Image
 ******************************************/

int image_plot(float *data, float datamin, float datamax, int n_x, int n_y, int dev_id, float *tr) {
  //  float tr[]={0.0,1.0,0.0,0.0,0.0,1.0};
  //  cpgslct(dev_id);
  //  cpgpage();
  cpgvstd();
  //  cpgwnad(0.0,1.0+n_x,0.0,1.0+n_y);
  cpgsch(1.1);
  cpgenv(0.0,1.0+n_x,0.0,1.0+n_y,-1,0);
  cpgimag(data,n_x,n_y,1,n_x,1,n_y,datamin,datamax,tr);
  cpgsch(0.6);
  //  cpgbox("bcntsi",0.0,0,"bcntsiv",0.0,0); 
  cpgbox("bc",0.0,0,"bc",0.0,0); 
  cpgmtxt("b",3.0,1.0,1.0,"pixel number"); 
  cpgwedg("BI",4.0,5.0,datamin,datamax,"pixel value");
  cpgsch(1.0);
  return(0);
}



/******************************************
 * Function to plot spaxels
 ******************************************/
void plot_spaxel(int color_index,float x_c, float y_c, GROUP group, int fill, int text_flag) {

  int j,i;
  float angle;
  float size1,size2;
  char shape;


  /*
   * OLD
   *  size1=(float)group.size1/100.0;
   * size2=(float)group.size2/100.0;
   */

  size1=(float)group.size1;
  size2=(float)group.size2;

  if (size1==0.0) size1=0.30;

  angle=((float)group.angle)/180*3.14159;

  //  printf("%d,%f, %f, %f,%f\n",color_index,x_c,y_c,size1,size2);

  cpgsci(color_index);
  if (fill<1) fill=1;
  if (fill>4) fill=4;

  cpgsfs(fill);
  //  shape=toupper(group.shape);
  //printf("%s %s\n",shape,group.shape);

  /*
  if (shape==1075052544) {
    shape=83;
  }
  
  */

  switch(group.shape) {
  case('S'): {
      if (angle==0.0) {
	cpgrect(x_c-size1/2,x_c+size1/2,y_c-size1/2,y_c+size1/2);
      } else {
	float xp[4],yp[4];
	float x[4],y[4];
	xp[0]=-size1; yp[0]=-size1;
	xp[1]=size1; yp[1]=-size1;
	xp[2]=size1; yp[2]=size1;
	xp[3]=-size1; yp[3]=size1;
	for (i=0;i<4;i++) {
	  x[i]=x_c+xp[i]*cos(angle)-yp[i]*sin(angle);
	  y[i]=y_c+xp[i]*sin(angle)+yp[i]*cos(angle);
	}
	cpgpoly(4,x,y);
      }
      break;
    }
    case('R'): {      
      if (angle==0.0) {
	cpgrect(x_c-size1/2,x_c+size1/2,y_c-size2/2,y_c+size2/2);
      } else {
	float xp[4],yp[4];
	float x[4],y[4];
	xp[0]=-size1; yp[0]=-size2;
	xp[1]=size1; yp[1]=-size2;
	xp[2]=size1; yp[2]=size2;
	xp[3]=-size1; yp[3]=size2;
	for (i=0;i<4;i++) {
	  x[i]=x_c+xp[i]*cos(angle)-yp[i]*sin(angle);
	  y[i]=y_c+xp[i]*sin(angle)+yp[i]*cos(angle);
	}
	cpgpoly(4,x,y);
      }
      break;
    }

    case('C'): {
      cpgcirc(x_c,y_c,size1);
      break;
    }
    default: {
      //      cpgrect(x_c-size1/2,x_c+size1/2,y_c-size1/2,y_c+size1/2);
      //      cpgcirc(x_c,y_c,0.15);
      cpgcirc(x_c,y_c,size1/2);
      break;
    }
    }

  
  //  return(0);
  

}


/*********************************************
 * Utility to split a text in different tokens
 *********************************************/

int split_text(char *text, char *splitter, int *n_words, char ***words) {
  int j;
  int n_w;
  //  char *w[1000000];
  char **w;
  n_w=0;
  for (j=0;j<strlen(text);j++) {
    if (text[j]==splitter[0]) {
      n_w++;
    }
  }
  //  n_w++;
  w = (char **)malloc((n_w+1)*sizeof(char*));

  w[0]=strtok(text,splitter);
  for (j=0;j<n_w;j++) {
    w[j+1]=strtok(NULL,splitter);
  }
  *words=w;
  *n_words=(n_w+1);
  //  free(w);
  return(0);
}

/*
 * Basic Stats
 */

int basic_stats(float *data, int n, float *mean, float *sigma, int sigma_clip) {

  float mean_temp=0.0;
  float sigma_temp=0.0;
  float mean1=0.0;
  float sigma1=0.0;
  int i,sum;
  int *marca;

  marca = (int *)malloc(n*sizeof(int));  

  for (i=0;i<n;i++) {
    mean1+=data[i]/n;
  }
 
  for (i=0;i<n;i++) {
    sigma1+=pow(data[i]-mean1,2);
  }
  sigma1=sqrt(sigma1/(n-1));
  
  if (sigma_clip!=0) {
    sum=0;
    for (i=0;i<n;i++) {
      marca[i]=0;
      if (abs(data[i]-mean1)<(sigma_clip*sigma1)) {
	mean_temp+=data[i];
	marca[i]=1;
	sum++;
      }
    }
    mean_temp/=sum;
    

  } else {
    mean_temp=mean1;
    sigma_temp=sigma1;
  }
 
  *mean=mean_temp;
  *sigma=sigma_temp;
  return(0);
}
