/*-------------------------------------------------------------------------
   Perform a 2D FFT inplace given a complex 2D array
   The direction dir, 1 for forward, -1 for reverse
   The size of the array (nx,ny)
   Return false if there are memory problems or
      the dimensions are not powers of 2
*/

#include <stdlib.h>
#include <math.h>
#include "fft2d.h"

#define FALSE 0
#define TRUE 1

int FFT2D(double **c,int nx,int ny,int dir)
{
   int i,j;
   int m,twopm;
   double *real,*imag;
   int k=0;

   /* Transform the rows */
   real = (double *)malloc(nx * sizeof(double));
   imag = (double *)malloc(nx * sizeof(double));
   if (real == NULL || imag == NULL)
      return(FALSE);
   if (!Powerof2(nx,&m,&twopm) || twopm != nx)
     return(FALSE);
   for (j=0;j<ny;j++) {
      for (i=0;i<nx;i++) {
         real[i] = c[i][(int)2*j];
         imag[i] = c[i][(int)2*j+1];	 
	 //	 printf("c[%d][%d]=(%f,%f)\n",i,2*j,c[i][2*j],c[i][2*j+1]);
	 k++;
      }
      FFT(dir,m,real,imag);
      for (i=0;i<nx;i++) {
	if (j!=0) {
	  c[i][(int)2*j] = real[i];
	  c[i][(int)2*j+1] = imag[i];
	} else {
	  c[i][0] = real[i];
	  c[i][1] = imag[i];
	}
      }
   }


   free(real);
   free(imag);

   /* Transform the columns */
   real = (double *)malloc(ny * sizeof(double));
   imag = (double *)malloc(ny * sizeof(double));
   if (real == NULL || imag == NULL)
      return(FALSE);
   if (!Powerof2(ny,&m,&twopm) || twopm != ny)
      return(FALSE);

   for (i=0;i<nx;i++) {
      for (j=0;j<ny;j++) {
         real[j] = c[i][(int)2*j];
         imag[j] = c[i][(int)2*j+1];	 
      }
      FFT(dir,m,real,imag);
      for (j=0;j<ny;j++) {
	if (j!=0) {
         c[i][(int)2*j] = real[j];
         c[i][(int)2*j+1] = imag[j];
	} else {
         c[i][0] = real[j];
         c[i][1] = imag[j];
	}
      }
   }
   free(real);
   free(imag);

   return(TRUE);
}

int FFT(int dir,int m,double *x,double *y)
{
   long nn,i,i1,j,k,i2,l,l1,l2;
   double c1,c2,tx,ty,t1,t2,u1,u2,z;

   /* Calculate the number of points */
   nn = 1;
   for (i=0;i<m;i++)
      nn *= 2;

   /* Do the bit reversal */
   i2 = nn >> 1;
   j = 0;
   for (i=0;i<nn-1;i++) {
      if (i < j) {
         tx = x[i];
         ty = y[i];
         x[i] = x[j];
         y[i] = y[j];
         x[j] = tx;
         y[j] = ty;
      }
      k = i2;
      while (k <= j) {
         j -= k;
         k >>= 1;
      }
      j += k;
   }

   //   printf("%d [x,y]=%f,%f\n",nn,x[0],y[0]);


   if (x[0]!=x[0]) x[0]=0.0;
   if (y[0]!=y[0]) y[0]=0.0;

   /* Compute the FFT */
   c1 = -1.0;
   c2 = 0.0;
   l2 = 1;
   for (l=0;l<m;l++) {
      l1 = l2;
      l2 <<= 1;
      u1 = 1.0;
      u2 = 0.0;
      for (j=0;j<l1;j++) {
         for (i=j;i<nn;i+=l2) {
            i1 = i + l1;
            t1 = u1 * x[i1] - u2 * y[i1];
            t2 = u1 * y[i1] + u2 * x[i1];
            x[i1] = x[i] - t1;
            y[i1] = y[i] - t2;
            x[i] += t1;
            y[i] += t2;
         }
         z =  u1 * c1 - u2 * c2;
         u2 = u1 * c2 + u2 * c1;
         u1 = z;
      }
      c2 = sqrt((1.0 - c1) / 2.0);
      if (dir == 1)
         c2 = -c2;
      c1 = sqrt((1.0 + c1) / 2.0);
   }

   /* Scaling for forward transform */
   if (dir == 1) {
      for (i=0;i<nn;i++) {
         x[i] /= (double)nn;
         y[i] /= (double)nn;
      }
   }

   return(TRUE);
}

int Powerof2(int n,int *m,int *twopm)
{
  if (n <= 1) {
    *m = 0;
    *twopm = 1;
    return(FALSE);
  }
  *m = 1;
  *twopm = 2;
  do {
    (*m)++;
    (*twopm) *= 2;
  } while (2*(*twopm) <= n);
  
  if (*twopm != n)
    return(FALSE);
  else
    return(TRUE);
}
