/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
! COPYRIGHT    (c)  2002 AIP, Potsdam, Germany
! IDENT        any2Euro3D.c
! LANGUAGE     C
! AUTHOR       S.F.Sanchez
! KEYWORDS
! PURPOSE      Facility to transform any reduced data to Euro3D data format
! COMMENT      No GROUP information included. No header data properly copied
! VERSION      0.1  2002-Dec-17 : Creation, SF.
------------------------------------------------------------------------------*/

#include <stdlib.h>
#include <IFU_io.h>
#include "stats.h"

/*-----------------------------------------------------------------------------
!
!.blk     Transform "any" reduced data in RSS format to Euro3D data: No GROUP info included.
!
!.prog                             any2Euro3D
!
!.purp    Facility to transform any reduced data to Euro3D data format.
!
-----------------------------------------------------------------------------*/

#define NB_SPEC 256
#define NPIX 1024

int set_group(E3D_file *frame, GROUP grupo)
{
  frame->ngroups = 1;
  frame->groups = (GROUP *)malloc(sizeof(GROUP));
  frame->groups[0]=grupo;
  return 0;
}

int main(int argc, char **argv)
{
  FILE *filename;
  int *spax_id,*type_id;
  float *spax_x,*spax_y;
  int type_save=-1;
  int nueva_cuenta=0;
  float g_angle=0;
  GROUP grupo; /* GROUPS */
  SPAXEL spax, *spaxels; /* SPAXELS */
  E3D_file e3d_image;
  SPECTRUM signal;
  IMAGE2D image;
  char **argval, **arglabel;
  char string[80];
  float *data;
#if 0
  short bitpix; /* Variable that stores the initial BITPIX value. */
#endif
  int specId, i,j,k, nb_spec=0, status, *nol; /* Euro3D variables */
  double start=0.0, step=1.0;
  float *x,*y;
  int npix_p1, npix_p2; /* image variables */
  double start_p[2],step_p[2];
  double rotang=0;
  double airmass=5;
  double parang=0, parang1=0, parang2=0;
  double pressure=0, pressure1=0, pressure2=0;
  double temperature=0;
  float size_x=1.0;
  float size_y=1.0;
  char shape[20];
  /* General variables */
  char input_filename[256], output_filename[256], author[80], origin[80];
  char position_file[256];
  char ref[80],date[80],date_obs[80],object[80],observer[80],telescope[80];
  float equinox;
  int error=0;

  start_p[0]=1; start_p[1]=1;
  step_p[0]=1; step_p[1]=1;

  /* before starting, override any IFU_DEFAULT_FMT variable, *
   * within E3D we only need Euro3D                          */
  setenv("IFU_DEFAULT_FMT","Euro3D",1);

  /* Read a file header */
  set_arglist("-if|inputfile none -of|outputfile outputfile -pf none -nb_spec 0 -au|author sanchez -or|origin AIP -ref|reference void -type 1 -rotang 0");
  set_version(VERSION);
  init_session(argv,argc,&arglabel,&argval);
#ifdef DEBUG_e3d
    set_control_level(WARNING);
#else
    set_control_level(NONE);
#endif

  for (i=0; i<9; i++) {
    if ((strstr(arglabel[i],"-if"))||(strstr(arglabel[i],"-inputfile"))) {
      strcpy(input_filename,argval[i]);
    }
    if ((strstr(arglabel[i],"-of"))||(strstr(arglabel[i],"-outfile"))) {
      strcpy(output_filename,argval[i]);
    }
    if (strstr(arglabel[i],"-pf")) {
      strcpy(position_file,argval[i]);
    }
    if (strstr(arglabel[i],"-nb_spec")) {
      nb_spec=atoi(argval[i]);
    }
    if ((strstr(arglabel[i],"-au"))||(strstr(arglabel[i],"-author"))) {
      strcpy(author,argval[i]);
    }
    if ((strstr(arglabel[i],"-or"))||(strstr(arglabel[i],"-origin"))) {
      strcpy(origin,argval[i]);
    }
    if ((strstr(arglabel[i],"-ref"))||(strstr(arglabel[i],"-reference"))) {
      strcpy(ref,argval[i]);
    }
    if (strstr(arglabel[i],"-type")) {
      type_save=atoi(argval[i]);
    }
    if (strstr(arglabel[i],"-rotang")) {
      rotang=(double)atof(argval[i]);
    }
  }

  /*
   * open the file and read some of the header information
   */
  printf("Open the file %s, read header information\n",input_filename);
  if (open_frame(&image,input_filename,"I") != 0) {
    printf("Error: wrong input format (this program can only convert RSS data"
           "with NAXIS=2)!\n");
    exit_session(0);
    exit(-1);
  }

  if (RD_desc(&image,"NAXIS1",INT,1,&npix_p1) < 0 ||
      RD_desc(&image,"NAXIS2",INT,1,&npix_p2) < 0) {
    printf("Some NAXISn keyword missing, error\n");
    exit_session(0);
    exit(-1);
  }

#if 0
  /* bitpix is never used anywhere */
  if (RD_desc(&image,"BITPIX",SHORT,1,&bitpix) < 0) {
    printf("No BITPIX defined, error\n");
    exit_session(0);
    exit(-1);
  }
#endif

  if (RD_desc(&image,"CRVAL1",DOUBLE,1,&start) < 0) {
    printf("No CRVAL1 defined, error\n");
    exit_session(0);
    exit(-1);
  }

  if (RD_desc(&image,"CDELT1",DOUBLE,1,&step) < 0) {
    print_msg("FITS header CDELT1 missing (trying to use CD1_1 instead)!");
    if (RD_desc(&image,"CD1_1",DOUBLE,1,&step) < 0) {
      printf("No CRDELT1 defined, error\n");
      exit_session(0);
      exit(-1);
    }
  }

  if (RD_desc(&image,"DATE",CHAR,80,date) < 0 ||
      RD_desc(&image,"DATE-OBS",CHAR,80,date_obs) < 0) {
    printf("DATE or DATE-OBS missing, error\n");
    exit_session(0);
    exit(-1);
  }

  if (RD_desc(&image,"EQUINOX",FLOAT,1,&equinox) < 0 &&
      RD_desc(&image,"HIERARCH CAHA TEL POS SET EQUINOX",
        FLOAT,1,&equinox) < 0) {
    printf("  Warning: no EQUINOX defined, using J2000\n");
    equinox = 2000;
  }

  if (RD_desc(&image,"OBJECT",CHAR,80,object) < 0) {
    printf("  Warning: no OBJECT defined\n");
  }

  if (RD_desc(&image,"OBSERVER",CHAR,80,observer) < 0) {
    printf("  Warning: no OBSERVER defined\n");
  }

  if (RD_desc(&image,"TELESCOP",CHAR,80,telescope) < 0) {
    printf("  Warning: no TELESCOP defined\n");
  }

  if (RD_desc(&image,"AIRMASS",DOUBLE,1,&airmass) < 0) {
    printf("  Warning: no AIRMASS defined\n");
  }

  if (RD_desc(&image,"PRES",DOUBLE,1,&pressure) < 0) {
    pressure = 0; /* misuse this as return value checker */
    pressure = RD_desc(&image,"HIERARCH ESO TEL AMBI PRES START",
                       DOUBLE,1,&pressure1);
    pressure = RD_desc(&image,"HIERARCH ESO TEL AMBI PRES END",
                       DOUBLE,1,&pressure2);
    if (pressure > 0) {
      pressure = (pressure1 + pressure2) / 2;
    } else if (RD_desc(&image,"HIERARCH CAHA GEN AMBI PRES",
                       DOUBLE,1,&pressure) < 0) {
      printf("  Warning: no PRESSURE defined\n");
      pressure = 0;
    }
  }

  if (RD_desc(&image,"TEMP",DOUBLE,1,&temperature) < 0 &&
      RD_desc(&image,"HIERARCH ESO TEL AMBI TEMP",DOUBLE,1,&temperature) < 0 &&
      RD_desc(&image,"HIERARCH CAHA GEN AMBI TEMP",DOUBLE,1,&temperature) < 0) {
    printf("  Warning: no TEMPERATURE defined\n");
  }

  if (RD_desc(&image,"PARANGLE",DOUBLE,1,&parang) < 0) {
    parang = 0; /* misuse this as return value checker */
    parang = RD_desc(&image,"HIERARCH ESO TEL PARANG START",DOUBLE,1,&parang1);
    parang = RD_desc(&image,"HIERARCH ESO TEL PARANG END",DOUBLE,1,&parang2);
    if (parang > 0) {
      parang = (parang1 + parang2) / 2;
    } else {
      printf("  Warning: no PARANG defined\n");
      parang = 0;
    }
  }

  /* Perhaps also include rotation angle on sky (FITS keyword
   * possibly ROTSKYPA and then recompute parang as
   * parang=parang-rotang;
   */

#ifdef DEBUG_e3d
  printf("input image: %dx%d, start=%lf, step=%lf\n",
         npix_p1, npix_p2, start, step);
#endif

  /*
   * Read the data from the image.
   */
  image.nx = npix_p1;
  image.ny = npix_p2;
  data = (float *)malloc(image.nx*image.ny*sizeof(float));
  for (j=0;j<image.ny;j++) {
    for (i=0;i<image.nx;i++) {
      data[i+j*image.nx] = RD_frame(&image,i,j);
    }
  }
  close_frame(&image);
  if (nb_spec == 0) {
    nb_spec = npix_p2;
  }

  /*
   * Read the position table
   */
  type_id = (int *)malloc((nb_spec+1)*sizeof(int));
  spax_id = (int *)malloc((nb_spec+1)*sizeof(int));
  spax_x = (float *)malloc((nb_spec+1)*sizeof(float));
  spax_y = (float *)malloc((nb_spec+1)*sizeof(float));
  if ((filename=fopen(position_file,"r"))==NULL) {
    printf("\nError opening position file %s\n", position_file);
    free(type_id);
    free(spax_id);
    free(spax_x);
    free(spax_y);
    exit_session(0);
    exit(1);
  } else {
    printf("Read the position table from %s\n", position_file);
    fscanf(filename, "%s %f %f %f", string, &size_x, &size_y, &g_angle);
    strcpy(shape,string);
    printf("  (shape: %s, elements: %5.3fx%5.3f)\n",string,size_x,size_y);
    j=0;
    while (fscanf(filename, "%d %f %f %d",
                  spax_id+j, spax_x+j, spax_y+j,
                  type_id+j) != EOF) {
      if (j>nb_spec) {
        print_msg("Position table has more entries than there are spectra!\n"
                  "Exiting\n");
        free(type_id);
        free(spax_id);
        free(spax_x);
        free(spax_y);
        exit_session(0);
        exit(2);
      }
		j++;
    }
  }
  if (j!=nb_spec)
    print_msg("Warning, -nb_spec=%d and linecount in -pf=%d do not match\n",
              nb_spec, j);

  /*
   * Creating the single GROUP
   */
#ifdef DEBUG_e3d
  printf("group1: shape=%s, %fx%f\n",shape, size_x,size_y);
#endif
  grupo.groupId=1;
  grupo.size1=1.0*size_x;
  grupo.size2=1.0*size_y;
  grupo.shape=shape[0];
  grupo.angle=g_angle;
  grupo.poswav=(float)start;
  grupo.airmass=(float)airmass;
  grupo.parang=(float)parang;
  grupo.pressure=(float)pressure;
  grupo.temperature=(float)temperature;

  /*
   * Creating the spatial structure of the data
   * NOTE: By now the relative positions of the fibers
   * are arbitrary (?), they are so called instrumental
   * We assume a square distribution...
   */
  i=1;
  k=1;
  nb_spec=j+1;
  nol = (int *)malloc(nb_spec*sizeof(int));
  x = (float *)malloc(nb_spec*sizeof(float));
  y = (float *)malloc(nb_spec*sizeof(float));
  for (j=0;j<nb_spec;j++) {
    nol[j]=spax_id[j];
    x[j]=spax_x[j];
    y[j]=spax_y[j];
    i++;
    if (i>sqrt(nb_spec)) {
      i=1;
      k++;
    }
  }

  /*
   * We create now the new Euro3D image and same the data.
   */
  printf("Write the new Euro3D file (%d spectra of %dpx length)  ",
         nb_spec, npix_p1);
  create_E3D_frame(&e3d_image,output_filename,npix_p1,start,step,FLOAT,"test",
                   NULL);
  for (specId=1; specId<=nb_spec; specId++) {
    printf("."), fflush(stdout);
    if ((type_save==-1) || (type_save==type_id[specId-1])) {
      nueva_cuenta++;
      /* Changed call to have no. of pixels in spectral instead of spatial *
       * direction */
      init_new_E3D_spec(&e3d_image,&signal,npix_p1,start);
      /* SPAXELS */
      spax.specId=nueva_cuenta;
      spax.group=1;
      spax.xpos=x[specId-1]*cos(rotang*.01745327)
               -y[specId-1]*sin(rotang*.01745327);
      spax.ypos=y[specId-1]*cos(rotang*.01745327)
               +x[specId-1]*sin(rotang*.01745327);

      for (j=0;j<signal.npts;j++) {
        /* XXX don't understand this, */
        /* it definitely overflows the data[] array for large datasets */
        WR_spec(&signal,j,data[j+(specId-1)*npix_p1]);
      }
      put_E3D_spec(&e3d_image,&signal,NULL,nueva_cuenta);
      put_E3D_spaxels(&e3d_image,nueva_cuenta,1,&spax);
    }
  }

  /*
   * Write the GROUPS information
   */
  set_group(&e3d_image,grupo);
  close_E3D_frame(&e3d_image);
  printf(" Done\n");
#ifdef DEBUG_e3d
  printf("n_count=%d\n",nueva_cuenta);
#endif

  /*
   * Reading the saved data on the output file...
   */
  printf("Checking the saved image, comparing spectrum by spectrum ");
  open_E3D_frame(&e3d_image,output_filename,"I");

  get_spectra_ID(&e3d_image,nol);
  nb_spec=nueva_cuenta;
  for (specId=1; specId<=nb_spec; specId++) {
    printf("."), fflush(stdout);
    get_E3D_spec(&e3d_image,&signal,NULL,nol[specId-1]);
    for (j=0;j<signal.npts;j++) {
      if (RD_spec(&signal,j)!=data[j+(specId-1)*npix_p1])
        error=1;
    }

    status=get_E3D_spaxels(&e3d_image,specId,&spaxels);
    if (spaxels[0].specId!=specId) error=2;
    if (spaxels[0].group!=1) error=3;
    if (spaxels[0].xpos!=x[specId-1]) error=4;
    if (spaxels[0].ypos!=y[specId-1]) error=5;
  }
  if (error==0) {
    printf(" OK\n");
  } else {
    printf("\nWarning: not all spectra are matched in the output file but "
           "this may\n         be OK if your input data included"
           " calibration spectra (error=%d)\n", error);
  }

  /* Testing the groups */
#ifdef DEBUG_e3d
  printf("nbSpec=%d\n",e3d_image.nbspec);
  printf("nGroups=%d\n",e3d_image.ngroups);
  printf("Type=%c\n",    e3d_image.groups[0].shape);
  printf("Size1=%f\n",   e3d_image.groups[0].size1);
  printf("Size2=%f\n",   e3d_image.groups[0].size2);
  printf("Airmass=%f\n", e3d_image.groups[0].airmass);
#endif

  close_E3D_frame(&e3d_image);

  free(type_id);
  free(spax_id);
  free(spax_x);
  free(spax_y);
  free(data);
  free(nol);
  free(x);
  free(y);

  exit_session(0);
  exit(0);
}
