/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
! COPYRIGHT    (c)  2004 AIP, Potsdam, Germany
! IDENT        flamesArgus2Euro3D.c
! LANGUAGE     C
! AUTHOR       S.F.Sanchez
! KEYWORDS     
! PURPOSE      Facility to transform FLAMES reduced data to Euro3D data format.
! COMMENT      
! VERSION      0.1  2004-Mar-15 : Creation, SF.
------------------------------------------------------------------------------*/

#include <stdlib.h>
#include <IFU_io.h>

#include "stats.h"

/*-----------------------------------------------------------------------------
!
!.blk            Transform the FLAMES reduced data to Euro3D data: No GROUP info included.
!
!.prog                             flamesArgus2Euro3D()
!
!.purp         Facility to transform FLAMES reduced data to Euro3D data format.
!
-----------------------------------------------------------------------------*/

#define NB_SPEC 256
#define NPIX 1024

int set_flames_group(E3D_file *frame, GROUP grupo, int n_groups, float *spax_angle)
{
  int j;
  frame->ngroups = n_groups+1;
  frame->groups = (GROUP *)malloc((n_groups+1)*sizeof(GROUP));
  for (j=0;j<=n_groups;j++) {
    grupo.angle=spax_angle[j]*180/3.141592654;
    frame->groups[j]=grupo;
  }
  return 0;
}

int set_single_flames_group(E3D_file *frame, GROUP grupo, int n_groups, float *spax_angle)
{
  frame->ngroups = 1;
  frame->groups = (GROUP *)malloc((n_groups+1)*sizeof(GROUP));
  grupo.angle=spax_angle[n_groups]*180/3.141592654;
  frame->groups[0]=grupo;
  return 0;
}

int main(int argc, char **argv)
{
  int *spax_id,*type_id,*def_spax,*group_id,button,which_button;
  float ra,dec,orient,xind,yind;
  float *spax_x,*spax_y,*spax_angle;
  float mean_x,mean_y;
  double temp_double;
  int nueva_cuenta=0;
  /* GROUPS */
  GROUP grupo;
  int n_groups,group_now;
  /* SPAXELS */
  SPAXEL spax, *spaxels;
  TABLE table; 
  int colno, colx, coly, color, colra, coldec, colbut;
  E3D_file e3d_image;
  SPECTRUM signal;
  IMAGE2D image;
  char **argval, **arglabel;
  float f_temp[10];
  double d_temp[10];
  char str_temp[80];
  float *data;
  int msize;
  /* Euro3D variables */
  int specId, i,j,k, nb_spec, status, *nol, n_cont;
  short sval;
  float start, step;
  float *x,*y;
  /* FLAMES image variables */
  int npix_p[2];
  double rotang;
  double airmass;
  double parang;
  double pressure;
  double temperature;
  float size;
  /* General variables */
  char input_filename[80], output_filename[80];
  char position_file[80];
  char date[80],date_obs[80];
  float equinox;
  int error=0;

  /* before starting, override any IFU_DEFAULT_FMT variable, *
   * within E3D we only need Euro3D                          */
  setenv("IFU_DEFAULT_FMT","Euro3D",1);

  /* Read a FLAMES file header */
  set_arglist("-if|inputfile none -of|outputfile -size 0.52 -button -1");
  set_version(VERSION);
  init_session(argv,argc,&arglabel,&argval);
  set_control_level(WARNING);

  get_argval(0,"%s",input_filename);
  get_argval(0,"%s",position_file);
  get_argval(1,"%s",output_filename);
  get_argval(2,"%f",&size);
  get_argval(3,"%d",&which_button);
  
  /*
	* We open the file to read the headers
   */
  printf("Opening the data file %s\n",input_filename);
  open_frame(&image,input_filename,"IO");

  /*
   * We read some of the header information
   */
  printf("Reading Header information\n");

  start = image.starty;
  start *= 10; /* convert to Angstroem */

  step = image.stepy;
  step *= 10; /* convert to Angstroem */

  if(!RD_desc(&image,"DATE",CHAR,80,str_temp)) {
    printf("No DATE defined, error\n");
    exit_session(0);
    return(0);
  }
  strcpy(date,str_temp);

  if(!RD_desc(&image,"DATE-OBS",CHAR,80,str_temp)) {
    printf("No DATE-OBS defined, error\n");
    exit_session(0);
    return(0);
  }
  strcpy(date_obs,str_temp);

  if(!RD_desc(&image,"EQUINOX",FLOAT,1,f_temp)) {
    printf("No EQUINOX defined, error\n");
    exit_session(0);
    return(0);
  }
  equinox=f_temp[0];

  if(!RD_desc(&image,"HIERARCH ESO TEL AIRM END",DOUBLE,1,d_temp)) {
    printf("No AIRMASS defined, error\n");
    exit_session(0);
    return(0);
  } else {
    airmass=d_temp[0];
  }

  if(!RD_desc(&image,"HIERARCH ESO TEL AMBI PRES END",DOUBLE,1,d_temp)) {
    printf("No PRESSURE defined, error\n");
    exit_session(0);
    return(0);
  } else {
    pressure=d_temp[0];
  }

  if(!RD_desc(&image,"HIERARCH ESO TEL AMBI TEMP",DOUBLE,1,d_temp)) {
    printf("No TEMPERATURE defined, error\n");
    exit_session(0);
    return(0);
  } else {
    temperature=d_temp[0];
  }

  if(!RD_desc(&image,"HIERARCH ESO TEL PARANG END",DOUBLE,1,d_temp)) {
    printf("No PARANGLE defined, error\n");
    exit_session(0);
    return(0);
  } else {
    parang=d_temp[0];
  }

  if(!RD_desc(&image,"HIERARCH ESO ADA ABSROT END",DOUBLE,1,d_temp)) {
    printf("No ROTSKYPA defined, error\n");
    exit_session(0);
    return(0);
  } else {
    rotang=d_temp[0];
  }
  parang=parang-rotang;

  printf("Reading FLAMES image data\n");
  /*
    We read the data from the image.
  */
  printf("[nx,ny]=%d,%d\n",image.nx,image.ny);
  msize = image.nx*image.ny*sizeof(float);  
  data = (float *)malloc(msize);

  memcpy ((char *)data, (char *)image.data.f_data, msize);

  mean_x=0;
  mean_y=0;

  type_id = (int *)malloc((image.ny)*sizeof(int));          
  spax_id = (int *)malloc((image.ny)*sizeof(int));          
  def_spax = (int *)malloc((image.ny)*sizeof(int));          
  spax_x = (float *)malloc((image.ny)*sizeof(float));          
  spax_y = (float *)malloc((image.ny)*sizeof(float));          
  spax_angle = (float *)malloc((image.ny)*sizeof(float));          
  group_id = (int *)malloc((image.ny)*sizeof(int));          

  close_frame(&image);

  strcat(position_file,"[3]");
  printf("POSITION TABLE='%s'\n",position_file);
  open_table(&table,strcat(input_filename,"[3]"),"I");

  colno=get_col_ref(&table,"NSPEC");
  colra=get_col_ref(&table,"RA");
  coldec=get_col_ref(&table,"DEC");
  colx=get_col_ref(&table,"XIND");
  coly=get_col_ref(&table,"YIND");
  color=get_col_ref(&table,"ORIENT");
  colbut=get_col_ref(&table,"button");

  nb_spec=table.row;

  n_groups=0;        
  n_cont=0;

  for (j=0;j<nb_spec;j++) {

    RD_tbl(&table,j,colno,&k);
    def_spax[k]=0;

    RD_tbl(&table,j,colra,&temp_double);
    ra=(float)temp_double;
    RD_tbl(&table,j,coldec,&temp_double);
    dec=(float)temp_double;

    RD_tbl(&table,j,colx,&sval);
    xind=(float)sval;
    xind=xind*size*2*1.02;
    RD_tbl(&table,j,coly,&sval);
    yind=(float)sval;
    yind=yind*size*2*1.02;

    RD_tbl(&table,j,color,&temp_double);
    orient=(float)temp_double;           

    RD_tbl(&table,j,colbut,&i);
    button=i;
    if ((button>0)&&(button<65535)) {
      if (n_cont==0) {
          group_now=button;
      } else {
          if (group_now!=button) {
              group_now=button;
              n_groups++;
          }
      }
      spax_angle[n_groups]=0.01745*((315-orient));        
      spax_id[j]=-1;        
    }
	else {

    group_id[j]=n_groups;
    def_spax[j]=1;
    spax_x[j]=ra*3600+(xind*cos(spax_angle[n_groups])-yind*sin(spax_angle[n_groups]));
    spax_y[j]=dec*3600+(yind*cos(spax_angle[n_groups])+xind*sin(spax_angle[n_groups]));
    spax_id[j]=k;        
    mean_x+=spax_x[j];
    mean_y+=spax_y[j];                    
	}
    n_cont++;
  }
  close_table(&table);

  if ((which_button<1)||(which_button>(n_groups+1))) {
      printf("Wrong button selection: %d [0,%d]\n",which_button,n_groups);
      which_button=-1;
  }

  /*
   * We recenter the data
   */
  mean_x /= n_cont;
  mean_y /= n_cont;
  for (j=0;j<=nb_spec;j++) {
    spax_x[j] -= mean_x;
    spax_y[j] -= mean_y;
  }

  printf("NB_SPEC=%d, %d\n",nb_spec,image.ny);

  /*
    Creating the single FLAMES GROUP
  */
  grupo.groupId=1;
  grupo.shape='R';
  grupo.size1=size; 
  grupo.size2=size;
  grupo.angle=0;
  grupo.poswav=(float)start;
  grupo.airmass=(float)airmass;
  grupo.parang=(float)parang;
  grupo.pressure=(float)pressure;
  grupo.temperature=(float)temperature;

  i=1;
  k=1;
  nol = (int *)malloc(nb_spec*sizeof(int));  
  x = (float *)malloc(nb_spec*sizeof(float));  
  y = (float *)malloc(nb_spec*sizeof(float));  
  for (j=0;j<nb_spec;j++) {
    nol[j]=spax_id[j];
    x[j]=spax_x[j];
    y[j]=spax_y[j];
    i++;
    if (i>sqrt(nb_spec)) {
      i=1;
      k++;
    }
  }

  npix_p[0]=image.ny;
  npix_p[1]=nb_spec;
  /*
    We create now the new Euro3D image and same the data.
  */
  printf("Opening the Euro3D image\n");
  create_E3D_frame(&e3d_image,output_filename,npix_p[0],start,step,FLOAT,"test",NULL);

  printf("Writing into created Euro3D file ...\n");
  for (specId=0;specId<nb_spec;specId++) {
    if (spax_id[specId]>0) {
      if (which_button==-1) {
        nueva_cuenta++;
        init_new_E3D_spec(&e3d_image,&signal,npix_p[0],start); 
        /* SPAXELS */
        spax.specId=nueva_cuenta;
        spax.group=group_id[specId];
        spax.xpos=x[specId];
  //      spax.xpos=x[specId]*cos(rotang*.01745327)-y[specId]*sin(rotang*.01745327);
        spax.ypos=y[specId];
   //     spax.ypos=y[specId]*cos(rotang*.01745327)+x[specId]*sin(rotang*.01745327);
        for (j=0;j<signal.npts;j++) {
            WR_spec(&signal,j,data[specId+j*npix_p[1]]);
        }                    
        put_E3D_spec(&e3d_image,&signal,NULL,nueva_cuenta);
        put_E3D_spaxels(&e3d_image,nueva_cuenta,1,&spax);

      } else if (which_button==group_id[specId]) {

        nueva_cuenta++;
        init_new_E3D_spec(&e3d_image,&signal,npix_p[0],start); 
        /* SPAXELS */
        spax.specId=nueva_cuenta;
        spax.group=1;
        //spax.xpos=x[specId]*cos(rotang*.01745327)-y[specId]*sin(rotang*.01745327);
        spax.xpos=x[specId];
        // spax.ypos=y[specId]*cos(rotang*.01745327)+x[specId]*sin(rotang*.01745327);
        spax.ypos=y[specId];
        for (j=0;j<signal.npts;j++) {
            WR_spec(&signal,j,data[specId+j*npix_p[1]]);
        }                    
        put_E3D_spec(&e3d_image,&signal,NULL,nueva_cuenta);
        put_E3D_spaxels(&e3d_image,nueva_cuenta,1,&spax);
      }
    }
  }

  printf("N_COUNT=%d, Ok\n",nueva_cuenta);
  /*
    Writing the GROUPS information 
  */
  printf("****We assume HIERARCH CAHA ADA POSANG to be the paralactic angle*********\n");
  printf("Writing the GROUPS information\n");
  
  if (which_button==-1) {
      set_flames_group(&e3d_image,grupo,n_groups,spax_angle);
  } else {
      set_single_flames_group(&e3d_image,grupo,which_button-1,spax_angle);
  }
  printf("NGROUPS=%d\n",e3d_image.ngroups);
  printf("Saving created Euro3D file ...\n");
  close_E3D_frame(&e3d_image);

  /*
    Reading the saved data on the output file...
  */
  printf("Checking the saved image: %s\n",output_filename);
  open_E3D_frame(&e3d_image,output_filename,"I");

  get_spectra_ID(&e3d_image,nol);
  printf("Reading Euro3D values spectrum by spectrum and Spaxels...");
  npix_p[1]=nueva_cuenta;
  
  for (specId=1;specId<=npix_p[1];specId++) {
          get_E3D_spec(&e3d_image,&signal,NULL,nol[specId-1]); 
          for (j=0;j<signal.npts;j++) {
            if (RD_spec(&signal,j)!=data[j+(specId-1)*npix_p[0]])
              error=1;
          }
          status=get_E3D_spaxels(&e3d_image,specId,&spaxels);
          if (spaxels[0].specId!=specId) error=2;
          if (spaxels[0].group <1) error=3;
          if (spaxels[0].xpos!=x[specId-1]) error=4;
          if (spaxels[0].ypos!=y[specId-1]) error=5;
  }
  if (error==0) {
    printf("Ok\n");
  } else {
    printf("\n***** THE EURO3D FILES DOES NOT MATCH THE ORIGINAL FILE *******\n");
    printf("ERROR=%d\n",error);
    printf("******\n");
  }

  /* Testing the groups */
  printf("NBSPEC=%d\n",e3d_image.nbspec);
  printf("NGROUPS=%d\n",e3d_image.ngroups);
  printf("TYPE=%d\n",e3d_image.groups[0].shape);
  printf("SIZE1=%f\n",        e3d_image.groups[0].size1); 
  printf("SIZE2=%f\n",        e3d_image.groups[0].size2); 
  printf("airmass=%f\n",        e3d_image.groups[0].airmass); 

  printf("closing previous datacube ...\n");
  close_E3D_frame(&e3d_image);

  free(data);
  exit_session(0);
  return(0);
}

