/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
! COPYRIGHT    (c)  2002 AIP, Potsdam, Germany
! IDENT        pmas2Euro3D.c
! LANGUAGE     C
! AUTHOR       S.F.Sanchez
! KEYWORDS
! PURPOSE      Facility to Visualize Euro3D data
! COMMENT      1st prototype
! VERSION      0.1  2002-Dec-17 : Creation, SF.
------------------------------------------------------------------------------*/

#include <stdio.h> //For Standard I/O
#include <stdlib.h> //For Functions like ATOI
#include <ctype.h> //For Type check
#include <string.h> //For String manipulation
#include <math.h> //For Math functions
#include <IFU_io.h> //For Lyon I/O Library
#include <data_io.h> //For Lyon I/O data
#include <time.h>
#include "Euro3D.h"

/*
 To use PGPLOT
 */
#include <cpgplot.h> // PGPLOT

/******************************
 * For Shared Memory
 ******************************/
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/shm.h>

#define MAX_SHM 33554432

/*-----------------------------------------------------------------------------
!
!.blk          Reads an Euro3D data file and plot a raw frame
!
!.prog                             VEuro3D()
!
!.purp         Facility to Visualize Euro3D data
!
-----------------------------------------------------------------------------*/

#define NB_SPEC 256
#define NPIX 1024

void imageplot(float* data,float datamin, float datamax, int l_naxes[2],int id, char *title);
void specplot(float* data, float datamin, float datamax, int npix, double start_w, double delta_w);
void setvp(void);
float minimum(float a, float b);
float maximum(float a, float b);
void palett(int type, float contra, float bright);
void fiddle(int l_naxes[2], char* ch, int* real_x, int* real_y, int old_x, int old_y);

int id1,id2,id3,id4;
int id_w;
int p=2;
float contra=0.8;
float bright=0.4;
float sign=1.0;
int mode=0;

int get_group(E3D_file *frame, GROUP *grupo, int j)
{
  if ((frame->ngroups!=0)&&(j<=frame->ngroups)) {
    grupo->groupId=frame->groups[j].groupId;
    grupo->shape=frame->groups[j].shape;
    grupo->size1=frame->groups[j].size1;
    grupo->size2=frame->groups[j].size2;
    grupo->angle=frame->groups[j].angle;
    grupo->poswav=frame->groups[j].poswav;
    grupo->parang=frame->groups[j].parang;
    grupo->pressure=frame->groups[j].pressure;
    grupo->temperature=frame->groups[j].temperature;
  } else {
    grupo->groupId=1;
    grupo->shape=2;
    grupo->size1=50; //miliArcseconds?
    grupo->size2=50; //miliArcseconds??
    grupo->angle=0;
    grupo->poswav=0.0;
    grupo->airmass=1.0;
    grupo->parang=0.0;
    grupo->pressure=700;
    grupo->temperature=30;
  }
  return(0);
}

void plot_spaxels(int color_index,float x_c, float y_c, GROUP group) {
  int i;
  float angle;
  float size1,size2;
  int shape;

  size1=(float)group.size1/100.0;
  size2=(float)group.size2/100.0;
  if (size1==0.0) size1=2.0;
  angle=((float)group.angle)/180*3.14159;

  cpgsci(color_index);
  shape=toupper(group.shape);
  /* ????
   * if (shape==1075052544) {
   *   shape=83;
   * }
   */
  switch (shape) {
  case (83):
    if (angle==0.0) {
      cpgrect(x_c-size1/2,x_c+size1/2,y_c-size1/2,y_c+size1/2);
    } else {
      float xp[4],yp[4];
      float x[4],y[4];
      xp[0]=-size1; yp[0]=-size1;
      xp[1]=size1; yp[1]=-size1;
      xp[2]=size1; yp[2]=size1;
      xp[3]=-size1; yp[3]=size1;
      for (i=0;i<4;i++) {
        x[i]=x_c+xp[i]*cos(angle)-yp[i]*sin(angle);
        y[i]=y_c+xp[i]*sin(angle)+yp[i]*cos(angle);
      }
      cpgpoly(4,x,y);
    }
    break;
  case ('R'):
    if (angle==0.0) {
      cpgrect(x_c-size1/2,x_c+size1/2,y_c-size2/2,y_c+size2/2);
    } else {
      float xp[4],yp[4];
      float x[4],y[4];
      xp[0]=-size1; yp[0]=-size2;
      xp[1]=size1; yp[1]=-size2;
      xp[2]=size1; yp[2]=size2;
      xp[3]=-size1; yp[3]=size2;
      for (i=0;i<4;i++) {
        x[i]=x_c+xp[i]*cos(angle)-yp[i]*sin(angle);
        y[i]=y_c+xp[i]*sin(angle)+yp[i]*cos(angle);
      }
      cpgpoly(4,x,y);
    }
    break;
  case ('C'):
    cpgcirc(x_c,y_c,size1);
    break;
  default:
    cpgcirc(x_c,y_c,0.15);
    break;
  } /* switch */
}

int main(int argc, char **argv)
{
  time_t t1,t2;
  int delta_n=256;
  int flag[15];
  int tmp_address;
  int n_bytes=15;
  int mem_key=3;
  int n_colors,nc=200,color_index;
  float x_c,y_c;
  GROUP *groups;
  GROUP grupo;
  int n_groups;
  SPAXEL **spaxels;
  int *n_spaxels,*index_start,*npts;
  float x_min,y_min,x_max,y_max;
  E3D_file e3d_image;
  SPECTRUM *spectra;
  char **argval, **arglabel;
  float *raw_data;
  float *raw_spec;
  float *raw_spax_int;
  /* Euro3D variables */
  int specId, i,j, npix, nb_spec, nfib;
  /* General variables */
  float min,max,min_i,max_i;
  int naxes[2];
  int npix_int;
  double start_w, end_w, delta_w;
  /* PGPLOT needed varibles */
  char answer_now;
  int real_x,real_y,old_x,old_y;

  /* before starting, override any IFU_DEFAULT_FMT variable, *
   * within E3D we only need Euro3D                          */
  setenv("IFU_DEFAULT_FMT","Euro3D",1);

  set_arglist("");
  time(&t1);
  set_version(VERSION);
  init_session(argv,argc,&arglabel,&argval);
  set_control_level(WARNING);

  nfib=0;
  /* We delete everything present before in the
   * SHM
   */
  printf("We ask for the Descriptors information about the FILE\n");
  ask_e3d_info(&e3d_image,&n_spaxels,&spaxels,&index_start,&npts,&groups);

  start_w=e3d_image.common_parameters[0];
  end_w=e3d_image.common_parameters[1];
  npix_int=e3d_image.common_parameters[2];
  delta_w=(end_w-start_w)/npix_int;
  npix=npix_int;
  nb_spec=e3d_image.nbspec;
  n_groups=e3d_image.ngroups;

  ask_raw_data(&raw_data,nb_spec,npix,delta_n);

  time(&t2);
  printf("Elapsed time = %ld seconds\n", t2-t1);

  /*
   * End of reading the RAW_DATA
   */
  printf("We determine the MIN and MAX values\n");

  spectra = (SPECTRUM *)malloc(nb_spec*sizeof(SPECTRUM));
  min=3000000;
  max=-3000000;
  x_min=10000;
  y_min=10000;
  x_max=-10000;
  y_max=-10000;
  for (specId=1;specId<=nb_spec;specId++) {
    for (j=0;j<n_spaxels[specId-1];j++) {
      if (x_min>spaxels[specId-1][j].xpos) x_min=spaxels[specId-1][j].xpos;
      if (x_max<spaxels[specId-1][j].xpos) x_max=spaxels[specId-1][j].xpos;
      if (y_min>spaxels[specId-1][j].ypos) y_min=spaxels[specId-1][j].ypos;
      if (y_max<spaxels[specId-1][j].ypos) y_max=spaxels[specId-1][j].ypos;
    }
    for (j=0;j<npix_int;j++) {
      if (min>raw_data[j+(specId-1)*npix])
        min=raw_data[j+(specId-1)*npix];
      if (max<raw_data[j+(specId-1)*npix])
        max=raw_data[j+(specId-1)*npix];
    }
  }
  if (min_i!=0.0)
    min=min_i;
  if (max_i!=0.0)
    max=max_i;
  printf("min=%f,max=%f\n",min,max);

  naxes[0]=npix_int;
  naxes[1]=nb_spec;
  id1=cpgopen("/xwin");
  cpgpap(10,0.5);
  cpgask(0);
  cpgsubp(1,1);
  cpgpanl(1,1);
  imageplot(raw_data,min_i,max_i,naxes,id1,"Euro3D raw data");

  printf("Options:\n");
  printf("Press 'Q' to quit\n");
  printf("Press 'R' to read the intensity in a point\n");
  printf("Press 'P' to change the palette\n");
  printf("Press 'F' to change the bright and contrast and reaload the image\n");
  printf("Press 'G' to change the bright and contrast (Only in PseudoColor Computers)\n");
  printf("Press 'S' to plot a Spectrum\n");
  printf("Press 'V' to plot a Spectrum accesing the SHM\n");
  printf("Press 'M' to plot a Monochromatic Image\n");
  printf("Press 'H' to plot a Monochromatic Image, using direct access to SHM\n");
  printf("Press 'N' to plot a Monochromatic Image\n");
  printf("Press 'O' to delete Share Memory\n");
  answer_now=' ';
  while ((toupper(answer_now)!='Q')) {
    fiddle(naxes, &answer_now,&real_x,&real_y, old_x, old_y);
    switch (toupper(answer_now)) {
    case('O'): {
          flag[0]=4;
          flag[1]=0;
          flag[2]=0;
          load_mem_int(flag,mem_key,n_bytes,&tmp_address);
          while(flag[1]==0) {
            read_mem_int(flag,mem_key,n_bytes);
          }
      break;
    }

    case('R'): {
      if ((real_x<npix)&&(real_x>0)&&(real_y<nb_spec)&&(real_y>0)) {
        printf("I(%d,%d)=%f\n",real_x,real_y,raw_data[real_x+real_y*npix]);
      } else {
        printf("Out of borders\n");
      }
      break;
    }
    case('P'): {
      p++;
      if (p>5) {
        p=1;
      }
      color_palett(p,sign*contra,bright,nc);
      palett(p,sign*contra,bright);
      imageplot(raw_data,min,max,naxes,id1,"Euro3D raw data");
      break;
    }
    case('F'): {
      color_palett(p,sign*contra,bright,nc);
      palett(p,sign*contra,bright);
      imageplot(raw_data,min,max,naxes,id1,"Euro3D raw data");
      cpgmove((float)old_x/naxes[0],(float)old_y/naxes[1]);
      break;
    }

    case('S'): {
      if(!id2) {
        id2=cpgopen("/xterm");
        cpgask(0);
      } else {
        cpgslct(id2);
      }
      if ((real_y<nb_spec)&&(real_y>=0)) {
        raw_spec = (float *)malloc(npts[real_y]*sizeof(float));
        for (j=0;j<npts[real_y];j++) {
          raw_spec[j]=raw_data[j+index_start[real_y]+real_y*npix];
        }
        specplot(raw_spec, min, max, npix, start_w, delta_w);
        free(raw_spec);
      } else {
        printf("Out of borders\n");
      }
      cpgslct(id1);
      break;
    }

    case('V'): {
      if(!id2) {
        id2=cpgopen("/xterm");
        cpgask(0);
      } else {
        cpgslct(id2);
      }
      if ((real_y<nb_spec)&&(real_y>=0)) {
        raw_spec = (float *)malloc(npts[real_y]*sizeof(float));
        ask_spectrum(&raw_spec,real_y+1,npts[real_y],index_start[real_y]);
        specplot(raw_spec, min, max, npix, start_w, delta_w);
        free(raw_spec);
      } else {
        printf("Out of borders\n");
      }
      cpgslct(id1);
      break;
    }

    case('M'): {
      if(!id3) {
        id3=cpgopen("/xwin");
        cpgpap(5,1);
        cpgask(0);
      } else {
        cpgslct(id3);
      }
      cpgenv(x_min-1.0,x_max+1.0,y_min-1.0,y_max+1.0,0,0);
      cpglab("X", "Y", "test");
      color_palett(p,sign*contra,bright,nc);
      for (specId=1;specId<=nb_spec;specId++) {
        n_colors=nc*contra;
        color_index=16+floor((raw_data[real_x+(specId-1)*npix]-min)/(max-min)*(n_colors-1));
        cpgsci(color_index);

        for (j=0;j<n_spaxels[specId-1];j++) {
          x_c=spaxels[specId-1][j].xpos;
          y_c=spaxels[specId-1][j].ypos;
          grupo=groups[spaxels[specId-1][j].group-1];
          plot_spaxels(color_index,x_c,y_c,grupo);
        }
      }
      cpgslct(id1);
      break;
    }

    case('H'): {
      if(!id3) {
        id3=cpgopen("/xwin");
        cpgpap(5,1);
        cpgask(0);
      } else {
        cpgslct(id3);
      }
      cpgenv(x_min-1.0,x_max+1.0,y_min-1.0,y_max+1.0,0,0);
      cpglab("X", "Y", "test");
      color_palett(p,sign*contra,bright,nc);
      ask_slice(&raw_spax_int,nb_spec,real_x);

      for (specId=1;specId<=nb_spec;specId++) {
        n_colors=nc*contra;
        color_index=16+floor((raw_spax_int[specId-1]-min)/(max-min)*(n_colors-1));
        cpgsci(color_index);

        for (j=0;j<n_spaxels[specId-1];j++) {
          x_c=spaxels[specId-1][j].xpos;
          y_c=spaxels[specId-1][j].ypos;
          grupo=groups[spaxels[specId-1][j].group-1];
          plot_spaxels(color_index,x_c,y_c,grupo);
        }
      }
      cpgslct(id1);
      break;
    }

    case('D'): {
      if(!id3) {
        id3=cpgopen("/xwin");
        cpgpap(5,1);
        cpgask(0);
      } else {
        cpgslct(id3);
      }
      cpgenv(x_min-1.0,x_max+1.0,y_min-1.0,y_max+1.0,0,0);
      cpglab("X", "Y", "test");
      color_palett(p,sign*contra,bright,nc);
      for (i=0;i<npix;i++) {
        ask_slice(&raw_spax_int,nb_spec,i);
        for (specId=1;specId<=nb_spec;specId++) {
          n_colors=nc*contra;
          color_index=16+floor((raw_spax_int[specId-1]-min)/(max-min)*(n_colors-1));
          cpgsci(color_index);

          for (j=0;j<n_spaxels[specId-1];j++) {
            x_c=spaxels[specId-1][j].xpos;
            y_c=spaxels[specId-1][j].ypos;
            grupo=groups[spaxels[specId-1][j].group-1];
            plot_spaxels(color_index,x_c,y_c,grupo);
          }
        }
      }
      cpgslct(id1);
      break;
    }

    case('N'): {
      if(!id4) {
        id4=cpgopen("/CPS");
        cpgpap(5,1);
        cpgask(0);
      } else {
        cpgslct(id4);
      }
      cpgenv(x_min-1.0,x_max+1.0,y_min-1.0,y_max+1.0,0,0);
      cpglab("X", "Y", "test");
      color_palett(p,sign*contra,bright,nc);
      for (specId=1;specId<=nb_spec;specId++) {
        n_colors=nc*contra;
        color_index=16+floor((raw_data[real_x+(specId-1)*npix]-min)/(max-min)*(n_colors-1));
        cpgsci(color_index);

        for (j=0;j<n_spaxels[specId-1];j++) {
          x_c=spaxels[specId-1][j].xpos;
          y_c=spaxels[specId-1][j].ypos;
          grupo=groups[spaxels[specId-1][j].group-1];
          plot_spaxels(color_index,x_c,y_c,grupo);
        }

      }
      cpgslct(id1);
      break;
    }

    case('L'): {
      if(!id3) {
        id3=cpgopen("/xwin");
        cpgpap(5,1);
        cpgask(0);
      } else {
        cpgslct(id3);
      }
      cpgenv(x_min-1.0,x_max+1.0,y_min-1.0,y_max+1.0,0,0);
      cpglab("X", "Y", "test");
      color_palett(p,sign*contra,bright,nc);
      for (i=0;i<npix;i++) {
        for (specId=1;specId<=nb_spec;specId++) {
          n_colors=nc*contra;
          color_index=16+floor((raw_data[i+(specId-1)*npix]-min)/(max-min)*(n_colors-1));
          cpgsci(color_index);

          for (j=0;j<n_spaxels[specId-1];j++) {
            x_c=spaxels[specId-1][j].xpos;
            y_c=spaxels[specId-1][j].ypos;
            grupo=groups[spaxels[specId-1][j].group-1];
            plot_spaxels(color_index,x_c,y_c,grupo);
          }

        }
      }
      cpgslct(id1);
      break;
    }
    } /* switch */

    old_x=real_x;
    old_y=real_y;
  }

  cpgclos();
  cpgend();
  //        free(raw_data);
  exit_session(0);
  return(0);
}

/*
 Subroutine that plots a spectrum
*/

void specplot(float* data, float datamin, float datamax, int npix, double start_w, double delta_w) {
  float *x,*y;
  int i;

  if (start_w==0.0)
    start_w=1.0;
  if (delta_w==0.0)
    delta_w=1.0;

  x = (float *)malloc(npix*sizeof(float));
  y = (float *)malloc(npix*sizeof(float));
  for (i=0;i<npix;i++) {
    x[i]=(float)start_w+(float)delta_w*i;
    y[i]=data[i];
  }

  cpgpage();
  cpgenv(x[0],x[npix-1],datamin,datamax,0,0);
  cpglab("\\gLambda", "Flux", "Spectra");
  cpgsls(1);
  cpgpt(npix,x,y,1);
  cpgline(npix,x,y);

}


/*
 Subroutine that plot the image.
*/
void imageplot(float* data, float datamin, float datamax, int l_naxes[2], int id, char *title) {
  float tr[]={0.0,1.0,0.0,0.0,0.0,1.0};
  /*
   * We select a certain device
   */
  cpgslct(id);
  cpgpage();
  setvp();
  cpgwnad(0.0,1.0+l_naxes[0],0.0,1.0+l_naxes[1]);
  /* Defines the original palette */
  palett(p,sign*contra,bright);
  /* Plot the image */
  cpgimag(data,l_naxes[0],l_naxes[1],1,l_naxes[0],1,l_naxes[1],datamin,datamax,tr);
  cpgmtxt("t",1.0,0.0,0.0,title);
  cpgsch(0.6);
  cpgbox("bcntsi",0.0,0,"bcntsiv",0.0,0);
  cpgmtxt("b",3.0,1.0,1.0,"pixel number");
  /* Draws the Scale */
  cpgwedg("BI",4.0,5.0,datamin,datamax,"pixel value");
  cpgsch(1.0);
}

void setvp() {
  float d, vpx1, vpx2, vpy1, vpy2;
  cpgsvp(0.0,1.0,0.0,1.0);
  cpgqvp(1,&vpx1,&vpx2,&vpy1,&vpy2);
  d=minimum(vpx2-vpx1,vpy2-vpy1)/40.0;
  vpx1=vpx1+5.0*d;
  vpx2=vpx2-2.0*d;
  vpy1=vpy1+8.0*d;
  vpy2=vpy2-2.0*d;
  cpgvsiz(vpx1,vpx2,vpy1,vpy2);
  return;
}

float minimum(float a, float b) {
  float c;
  if (a<b) {
    c=a;
  } else {
    c=b;
  }
  return c;
}

float maximum(float a, float b) {
  float c;
  if (a>b) {
    c=a;
  } else {
    c=b;
  }
  return c;
}

/*
 * This subrutine changes the palette.
 * It is very useful, since it has the colour ranges.
 */
void palett(int type, float contra, float bright){
  float GL[]={0.0, 1.0};
  float GR[]={0.0, 1.0};
  float GG[]={0.0, 1.0};
  float GB[]={0.0, 1.0};

  float RL[]={-0.5, 0.0, 0.17, 0.33, 0.50, 0.67, 0.83, 1.0, 1.7};
  float RR[]={ 0.0, 0.0,  0.0,  0.0,  0.6,  1.0,  1.0, 1.0, 1.0};
  float RG[]={ 0.0, 0.0,  0.0,  1.0,  1.0,  1.0,  0.6, 0.0, 1.0};
  float RB[]={ 0.0, 0.3,  0.8,  1.0,  0.3,  0.0,  0.0, 0.0, 1.0};

  float HL[]={0.0, 0.2, 0.4, 0.6, 1.0};
  float HR[]={0.0, 0.5, 1.0, 1.0, 1.0};
  float HG[]={0.0, 0.0, 0.5, 1.0, 1.0};
  float HB[]={0.0, 0.0, 0.0, 0.3, 1.0};

  float WL[]={0.0, 0.5, 0.5, 0.7, 0.7, 0.85, 0.85, 0.95, 0.95, 1.0};
  float WR[]={0.0, 1.0, 0.0, 0.0, 0.3,  0.8,  0.3,  1.0,  1.0, 1.0};
  float WG[]={0.0, 0.5, 0.4, 1.0, 0.0,  0.0,  0.2,  0.7,  1.0, 1.0};
  float WB[]={0.0, 0.0, 0.0, 0.0, 0.4,  1.0,  0.0,  0.0, 0.95, 1.0};

  float AL[]={0.0, 0.1, 0.1, 0.2, 0.2, 0.3, 0.3, 0.4, 0.4, 0.5,
     0.5, 0.6, 0.6, 0.7, 0.7, 0.8, 0.8, 0.9, 0.9, 1.0};
  float AR[]={0.0, 0.0, 0.3, 0.3, 0.5, 0.5, 0.0, 0.0, 0.0, 0.0,
              0.0, 0.0, 0.0, 0.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0};
  float AG[]={0.0, 0.0, 0.3, 0.3, 0.0, 0.0, 0.0, 0.0, 0.8, 0.8,
              0.6, 0.6, 1.0, 1.0, 1.0, 1.0, 0.8, 0.8, 0.0, 0.0};
  float AB[]={0.0, 0.0, 0.3, 0.3, 0.7, 0.7, 0.7, 0.7, 0.9, 0.9,
              0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};

  switch(type) {
  case (1): {
    /* Grey */
    cpgctab(GL, GR, GG, GB, 2, contra, bright);
    break;
  }
  case (2): {
    /* Rainbow */
    cpgctab(RL, RR, RG, RB, 9, contra, bright);
    break;
  }
  case (3): {
    /* Heat */
    cpgctab(HL, HR, HG, HB, 5, contra, bright);
    break;
  }
  case (4): {
    /* Weird IRAF */
    cpgctab(WL, WR, WG, WB, 10, contra, bright);
    break;
  }
  case (5): {
    /* AIPS */
    cpgctab(AL, AR, AG, AB, 20, contra, bright);
    break;
  }
  default: {
    cpgctab(GL, GR, GG, GB, 2, contra, bright);
    break;
  }
  }
}

void fiddle(int l_naxes[2], char* ch, int* real_x, int* real_y, int old_x, int old_y) {
  float x=0.5,y=1.0;
  float x1,y1,x2,y2;
  float b1=0.0,b2=1.0,c1=0.0,c2=1.0;
  int fuera=0;

  sign=1;
  cpgqwin(&x1,&x2,&y1,&y2);
  cpgswin(b1,b2,c1,c2);
  while ((toupper(ch[0])!='Q')&&(fuera==0)) {
    if (cpgband(mode,0,(float)(old_x/l_naxes[0]),(float)(old_y/l_naxes[1]),&x,&y,ch)) {
      *real_x=(int) (l_naxes[0]*x);
      *real_y=(int) (l_naxes[1]*y);

      if (*real_x<0) *real_x=0;
      if (*real_x>l_naxes[0]) *real_x=l_naxes[0];
      if (*real_y<0) *real_y=0;
      if (*real_y>l_naxes[1]) *real_y=l_naxes[1];

      switch(toupper(ch[0])) {
      case('H'): {
        fuera=1;
        break;
      }
      case('Z'): {
        fuera=1;
        break;
      }
      case('W'): {
        fuera=1;
        break;
      }
      case('O'): {
        fuera=1;
        break;
      }
      case('R'): {
        fuera=1;
        break;
      }
      case('S'): {
        fuera=1;
        break;
      }
      case('V'): {
        fuera=1;
        break;
      }
      case('M'): {
        fuera=1;
        break;
      }
      case('L'): {
        fuera=1;
        break;
      }
      case('D'): {
        fuera=1;
        break;
      }
      case('B'): {
        fuera=1;
        break;
      }
      case('N'): {
        fuera=1;
        break;
      }
      case('E'): {
        fuera=1;
        break;
      }
      case('C'): {
        fuera=1;
        break;
      }
      case('G'): {
        bright= maximum(b1,minimum(b2,x));
        contra= maximum(c1,minimum(c2,y));
        palett(p,sign*contra,bright);
        break;
      }
      case('I'): {
        fuera=1;
        break;
      }
      case('Q'): {
        cpgswin(x1,x2,y1,y2);
        break;
      }
      case('F'): {
        bright= maximum(b1,minimum(b2,x));
        contra= maximum(c1,minimum(c2,y));
        fuera=1;
        break;
      }
      case(';'): {
        contra=1.0;
        y=1.0;
        bright=0.5;
        x=0.5;
        break;
      }
      case('U'): {
        printf("c=%f,b=%f\n",contra,bright);
        break;
      }
      case('-'): {
        sign=-sign;
        break;
      }
      case('1'): {
        p=1;
        break;
      }
      case('2'): {
        p=2;
        break;
      }
      case('3'): {
        p=3;
        break;
      }
      case('4'): {
        p=4;
        break;
      }
      case('5'): {
        p=5;
        break;
      }
      case('P'): {
        fuera=1;
        break;
      }
      }
      palett(p,sign*contra,bright);
    }
  }
}
