/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
! COPYRIGHT    (c)  2002 AIP, Potsdam, Germany
! IDENT        pmas2Euro3D.c
! LANGUAGE     C
! AUTHOR       S.F.Sanchez
! KEYWORDS
! PURPOSE      Facility to Visualize Euro3D data
! COMMENT      1st prototype
! VERSION      0.1  2002-Dec-17 : Creation, SF.
------------------------------------------------------------------------------*/

#include <stdio.h> //For Standard I/O
#include <stdlib.h> //For Functions like ATOI
#include <ctype.h> //For Type check
#include <string.h> //For String manipulation
#include <math.h> //For Math functions
#include <IFU_io.h> //For Lyon I/O Library
#include <data_io.h> //For Lyon I/O data
#include "Euro3D.h"

/*
 To use PGPLOT
 */
#include <cpgplot.h> // PGPLOT

/******************************
 * For Shared Memory
 ******************************/
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/shm.h>

#define MAX_SHM 33554432

/*-----------------------------------------------------------------------------
!
!.blk          Reads an Euro3D data file and plot a raw frame
!
!.prog                             VEuro3D()
!
!.purp         Facility to Visualize Euro3D data
!
-----------------------------------------------------------------------------*/

#define NB_SPEC 256
#define NPIX 1024

void imageplot(float* data,float datamin, float datamax, int l_naxes[2],int id, char *title);
void specplot(float* data, float datamin, float datamax, int npix, double start_w, double delta_w);
void setvp(void);
float minimum(float a, float b);
float maximum(float a, float b);
void palett(int type, float contra, float bright);
void fiddle(int l_naxes[2], char* ch, int* real_x, int* real_y, int old_x, int old_y);

int id1,id2,id3,id4;
int id_w;
int p=2;
float contra=0.8;
float bright=0.4;
float sign=1.0;
int mode=0;

int get_group(E3D_file *frame, GROUP *grupo, int j)
{
  if ((frame->ngroups!=0)&&(j<=frame->ngroups)) {
    grupo->groupId=frame->groups[j].groupId;
    grupo->shape=frame->groups[j].shape;
    grupo->size1=frame->groups[j].size1;
    grupo->size2=frame->groups[j].size2;
    grupo->angle=frame->groups[j].angle;
    grupo->poswav=frame->groups[j].poswav;
    grupo->parang=frame->groups[j].parang;
    grupo->pressure=frame->groups[j].pressure;
    grupo->temperature=frame->groups[j].temperature;
  } else {
    grupo->groupId=1;
    grupo->shape=2;
    grupo->size1=50; //miliArcseconds?
    grupo->size2=50; //miliArcseconds??
    grupo->angle=0;
    grupo->poswav=0.0;
    grupo->airmass=1.0;
    grupo->parang=0.0;
    grupo->pressure=700;
    grupo->temperature=30;
  }
  return(0);
}

void plot_spaxels(int color_index,float x_c, float y_c, GROUP group) {
  int i;
  float angle;
  float size1,size2;
  int shape;

  size1=(float)group.size1/100.0;
  size2=(float)group.size2/100.0;
  if (size1==0.0) size1=0.5;
  angle=((float)group.angle)/180*3.14159;

  cpgsci(color_index);
  shape=toupper(group.shape);
  /* ????
   * if (shape==1075052544) {
   *   shape=83;
   * }
   */
  switch (shape) {
  case (83):
    if (angle==0.0) {
      cpgrect(x_c-size1/2,x_c+size1/2,y_c-size1/2,y_c+size1/2);
    } else {
      float xp[4],yp[4];
      float x[4],y[4];
      xp[0]=-size1; yp[0]=-size1;
      xp[1]=size1; yp[1]=-size1;
      xp[2]=size1; yp[2]=size1;
      xp[3]=-size1; yp[3]=size1;
      for (i=0;i<4;i++) {
        x[i]=x_c+xp[i]*cos(angle)-yp[i]*sin(angle);
        y[i]=y_c+xp[i]*sin(angle)+yp[i]*cos(angle);
      }
      cpgpoly(4,x,y);
    }
    break;
  case ('R'):
    if (angle==0.0) {
      cpgrect(x_c-size1/2,x_c+size1/2,y_c-size2/2,y_c+size2/2);
    } else {
      float xp[4],yp[4];
      float x[4],y[4];
      xp[0]=-size1; yp[0]=-size2;
      xp[1]=size1; yp[1]=-size2;
      xp[2]=size1; yp[2]=size2;
      xp[3]=-size1; yp[3]=size2;
      for (i=0;i<4;i++) {
        x[i]=x_c+xp[i]*cos(angle)-yp[i]*sin(angle);
        y[i]=y_c+xp[i]*sin(angle)+yp[i]*cos(angle);
      }
      cpgpoly(4,x,y);
    }
    break;
  case ('C'):
    cpgcirc(x_c,y_c,size1);
    break;
  default:
    cpgcirc(x_c,y_c,size1/2);
    break;
  } /* switch */
}

int main(int argc, char **argv)
{
  int *flag,temp_flag;
  int address[15],tmp_address;
  int n_bytes;
  SPAXEL **spaxels;
  int *n_spaxels,*index_start,*npts,max_npts;
  float x_min,y_min,x_max,y_max;
  E3D_file e3d_image,e3d_mem;
  SPECTRUM signal;
  SPECTRUM *spectra;
  IMAGE2D image;
  char **argval, **arglabel;
  int mem_key;
  short s_temp[10];
  float *raw_data;
  float *raw_spec;
  char euro3d;
  /* Euro3D variables */
  int specId, i,j, npix, nb_spec, *nol, nfib;
  float *x,*y;
  char *format;
  /* General variables */
  char input_filename[80];
  float min,max,min_i,max_i;
  int naxes[2];
  int npix_int;
  double start_w, end_w, delta_w;
  /* PGPLOT needed varibles */

  /* before starting, override any IFU_DEFAULT_FMT variable, *
   * within E3D we only need Euro3D                          */
  setenv("IFU_DEFAULT_FMT","Euro3D",1);

  set_arglist("-if|inputfile none -nfib|nfibers 0 -npalette 2 -min 0 -max 0");
  set_version(VERSION);
  init_session(argv,argc,&arglabel,&argval);
  set_control_level(WARNING);

  nfib=0;
  for (i=0; i<5; i++) {
    if ((strstr(arglabel[i],"-if"))||(strstr(arglabel[i],"-inputfile"))) {
      strcpy(input_filename,argval[i]);
    }
    if ((strstr(arglabel[i],"-nfib"))||(strstr(arglabel[i],"-nfibers"))) {
      nfib=atoi(argval[i]);
    }
    if (strstr(arglabel[i],"-npalette")) {
      p=atoi(argval[i]);
      if (p>5)
        p=1;
      if (p<1)
        p=5;
    }
    if (strstr(arglabel[i],"-min")) {
      min_i=atof(argval[i]);
    }
    if (strstr(arglabel[i],"-max")) {
      max_i=atof(argval[i]);
    }
  }

  /*
    checking the format of the file
  */
  format = file_format(argval[0]);
  if ((format[0]!=FITS_B_FORMAT)&&(format[1]!=T_TABLE)) {
    printf("The file does not seem to conform the Euro3D data format\n");
  }

  /*
    Reading the data on the output file...
  */
  printf("Opening the image: %s\n",input_filename);
  open_E3D_frame(&e3d_mem,input_filename,"I");
  /* Now we write in memory
   * and we try to read from it!!
   */
  printf("We load the FILE to the memory\n");

  mem_key=1;
  if (load_mem_e3d(e3d_mem,mem_key,&tmp_address)!=0) {
    printf("Error loading in SHM\n");
    exit(1);
  }
  address[mem_key]=tmp_address;
  printf("add=%d\n",address[mem_key]);
  read_mem_e3d(&e3d_image,mem_key);
  nb_spec=e3d_image.nbspec;
  printf("GROUPS information:\n");
  printf("NGROUPS=%d\n",e3d_image.ngroups);
  for (j=0;j<e3d_image.ngroups;j++) {
    printf("TYPE=%d\n",    e3d_image.groups[j].shape);
    printf("SIZE1=%lf\n",  e3d_image.groups[j].size1);
    printf("SIZE2=%lf\n",  e3d_image.groups[j].size2);
    printf("airmass=%f\n", e3d_image.groups[j].airmass);
  }
  printf("Reading the Header of the file %s\n",input_filename);

  if(!RD_desc(&image,"EURO3D",CHAR,10,s_temp)) {
     printf("No EURO3D definied, error\n");
     exit_session(0);
     return(0);
   }
  euro3d=s_temp[0];
  printf("Euro3D = %c\n",euro3d);

  if ((nfib!=0)||(nb_spec==0)) {
    nb_spec=nfib;
    if (nb_spec==0) {
      nb_spec=256;
    }
    printf("nb_spec forced to %d\n",nb_spec);
  }

  nol = (int *)malloc(nb_spec*sizeof(int));
  x = (float *)malloc(nb_spec*sizeof(float));
  y = (float *)malloc(nb_spec*sizeof(float));

  spaxels = (SPAXEL **)malloc(nb_spec*sizeof(SPAXEL));
  n_spaxels = (int *)malloc(nb_spec*sizeof(int));
  index_start = (int *)malloc(nb_spec*sizeof(int));
  npts = (int *)malloc(nb_spec*sizeof(int));

  printf("Reading common parameters\n");
  get_common_param(&e3d_image,&npix_int,&start_w,&end_w);
  printf("%d,%f,%f\n",npix_int,start_w,end_w);
  delta_w=(end_w-start_w)/npix_int;

  get_spectra_ID(&e3d_image,nol);
  printf("Reading Euro3D values spectrum by spectrum ...\n");
  min=3000000;
  max=-3000000;
  naxes[1]=nb_spec;
  spectra = (SPECTRUM *)malloc(nb_spec*sizeof(SPECTRUM));
  x_min=10000;
  y_min=10000;
  x_max=-10000;
  y_max=-10000;
  max_npts=0;
  for (specId=1;specId<=nb_spec;specId++) {
    /*
      We should read the 'index_start' when available
      in the new format.
      By now it is set to 0.
    */
    index_start[specId-1]=0;
    /*
      We get the spaxels information!!!
    */
    n_spaxels[specId-1]=get_E3D_spaxels(&e3d_image,specId,&spaxels[specId-1]);

    for (j=0;j<n_spaxels[specId-1];j++) {
      if (x_min>spaxels[specId-1][j].xpos) x_min=spaxels[specId-1][j].xpos;
      if (x_max<spaxels[specId-1][j].xpos) x_max=spaxels[specId-1][j].xpos;
      if (y_min>spaxels[specId-1][j].ypos) y_min=spaxels[specId-1][j].ypos;
      if (y_max<spaxels[specId-1][j].ypos) y_max=spaxels[specId-1][j].ypos;

    }
    /*
      We get the spectra information.
    */
    get_E3D_spec(&e3d_image,&signal,NULL,nol[specId-1]);
    spectra[specId-1]=signal;
    npts[specId-1]=signal.npts;
    if (signal.npts>max_npts) {
      max_npts=signal.npts;
    }
  }

  /*
   * Now we feed the Raw_data
   */
  npix=max_npts;
  naxes[0]=npix;
  raw_data = (float *)malloc((npix*nb_spec)*sizeof(float));
  if (npix_int!=npix) {
    printf("There is some error in the determination of the number of pixels\n");
    printf("npix=%d!=%d\n",npix,npix_int);
  }
  for (specId=1;specId<=nb_spec;specId++) {
    for (j=0;j<max_npts;j++) {
      if (j<index_start[specId-1]) {
        raw_data[j+(specId-1)*npix]=0;
      } else {
        raw_data[j+(specId-1)*npix]=RD_spec(&spectra[specId-1],j-index_start[specId]-1);
      }
      if (min>raw_data[j+(specId-1)*npix])
        min=raw_data[j+(specId-1)*npix];
      if (max<raw_data[j+(specId-1)*npix])
        max=raw_data[j+(specId-1)*npix];
    }
  }
  if (min_i!=0.0)
    min=min_i;
  if (max_i!=0.0)
    max=max_i;
  printf("min=%f,max=%f\n",min,max);

  /*******************************************************
   * LOAD MEMORY STUFF
   * SHM
   *******************************************************/
  /*
   * We load the STAKED SPECTRA in SHM
   */
  /*
   * mem_key=2;
   * n_bytes=naxes[0]*naxes[1];
   * printf("\nWe store in the shared memory the RAW_DATA\n");
   * if (n_bytes<MAX_SHM) {
   *   load_mem_float(raw_data,mem_key,(naxes[0]*naxes[1]),&tmp_address);
   * } else {
   *   printf("*******************************\n");
   *   printf("** TOO MUCH LARGE FOR THE SHM**\n");
   *   printf("*******************************\n");
   * }
   * address[mem_key]=tmp_address;
   */

  /*
   * We load the N_SPAXELS in Memory
   */
  printf("\nWe store in the shared memory the N_SPAXELS\n");
  mem_key=5;
  n_bytes=nb_spec;
  if (n_bytes<MAX_SHM) {
    load_mem_int(n_spaxels,mem_key,n_bytes,&tmp_address);
  } else {
    printf("*******************************\n");
    printf("** TOO MUCH LARGE FOR THE SHM**\n");
    printf("*******************************\n");
  }
  address[mem_key]=tmp_address;
  /*
   * We load the SPAXELS in Memory
   */
  printf("\nWe store in the shared memory the SPAXELS\n");
  mem_key=6;
  n_bytes=nb_spec;
  if (n_bytes<MAX_SHM) {
    load_mem_spaxels(spaxels,n_spaxels,mem_key,n_bytes,&tmp_address);
  } else {
    printf("*******************************\n");
    printf("** TOO MUCH LARGE FOR THE SHM**\n");
    printf("*******************************\n");
  }
  address[mem_key]=tmp_address;
  /*
   * We load the INDEX_START in Memory
   */
  printf("\nWe store in the shared memory the INDEX_START\n");
  mem_key=7;
  n_bytes=nb_spec;
  if (n_bytes<MAX_SHM) {
    load_mem_int(index_start,mem_key,n_bytes,&tmp_address);
  } else {
    printf("*******************************\n");
    printf("** TOO MUCH LARGE FOR THE SHM**\n");
    printf("*******************************\n");
  }
  address[mem_key]=tmp_address;
  /*
   * We load the NPTS in Memory
   */
  printf("\nWe store in the shared memory the NPTS\n");
  mem_key=8;
  n_bytes=nb_spec;
  if (n_bytes<MAX_SHM) {
    load_mem_int(npts,mem_key,n_bytes,&tmp_address);
  } else {
    printf("*******************************\n");
    printf("** TOO MUCH LARGE FOR THE SHM**\n");
    printf("*******************************\n");
  }
  address[mem_key]=tmp_address;
  printf("\nWe store in the shared memory the GROUPS info\n");
  mem_key=9;
  n_bytes=nb_spec;
  if (n_bytes<MAX_SHM) {
    load_mem_group(e3d_image.groups,mem_key,n_bytes,&tmp_address);
  } else {
    printf("*******************************\n");
    printf("** TOO MUCH LARGE FOR THE SHM**\n");
    printf("*******************************\n");
  }
  address[mem_key]=tmp_address;

  printf("closing previous datacube ...\n");
  close_E3D_frame(&e3d_image);

  /*
   * We load the N_SPAXELS in Memory
   */
  printf("\nWe store in the shared memory the FLAG\n");
  mem_key=3;
  n_bytes=15;
  flag = (int *)malloc(n_bytes*sizeof(int));
  for (j=0;j<15;j++) {
      flag[j]=0;
  }

  if (n_bytes<MAX_SHM) {
    load_mem_int(flag,mem_key,n_bytes,&tmp_address);
  } else {
    printf("*******************************\n");
    printf("** TOO MUCH LARGE FOR THE SHM**\n");
    printf("*******************************\n");
  }
  address[mem_key]=tmp_address;

  printf("Waiting for the Flag to give a signal\n");
  temp_flag=-666;
  while (flag[0]!=666) {
    read_mem_int(flag,mem_key,n_bytes);
    /*
     * To I/O only one spectra.
     * FLAG=1
     */
    if ((flag[0]==1)&&(temp_flag!=flag[2])) {
      /*
       * We read the proper information to  be
       * loaded
       */
      raw_spec = (float *)malloc(npix*sizeof(float));
      for (j=0;j<npix;j++) {
          raw_spec[j]=raw_data[j+flag[2]*npix];
      }
      load_mem_float(raw_spec,2,npix,&tmp_address);
      free(raw_spec);
      /*
       * We send the signal to the client
       */
      flag[1]=1;
      load_mem_int(flag,mem_key,n_bytes,&tmp_address);
      temp_flag=flag[2];
      printf("No paso\n");
    }
    /*
     * To read a set of spectra
     * FLAG=2
     */
    if ((flag[0]==2)&&(temp_flag!=flag[2])) {
      /*
       * We read the proper information to  be
       * loaded
       */
      shmctl(address[2], IPC_RMID, 0);
      raw_spec = (float *)malloc(npix*(flag[3]-flag[2])*sizeof(float));
      for (j=0;j<npix*(flag[3]-flag[2]);j++) {
          raw_spec[j]=raw_data[j+flag[2]*npix];
      }
      load_mem_float(raw_spec,2,npix*(flag[3]-flag[2]),&tmp_address);
      free(raw_spec);
      address[2]=tmp_address;
      /*
       * We send the signal to the client
       */
      flag[1]=1;
      load_mem_int(flag,mem_key,n_bytes,&tmp_address);
      temp_flag=flag[2];
    }
  }

  free(raw_data);

  /*
   * We clean the SHARED memory
   */
  shmdt(&address[1]);
  shmdt(&address[2]);
  shmdt(&address[3]);
  shmdt(&address[5]);
  shmdt(&address[6]);
  shmdt(&address[7]);
  shmdt(&address[8]);
  shmdt(&address[9]);
  shmctl(address[1], IPC_RMID, NULL);
  shmctl(address[2], IPC_RMID, 0);
  shmctl(address[3], IPC_RMID, 0);
  shmctl(address[5], IPC_RMID, 0);
  shmctl(address[6], IPC_RMID, 0);
  shmctl(address[7], IPC_RMID, 0);
  shmctl(address[8], IPC_RMID, 0);
  shmctl(address[9], IPC_RMID, 0);
  exit_session(0);
  return(0);
}

/*
 Subroutine that plots a spectrum
*/

void specplot(float* data, float datamin, float datamax, int npix, double start_w, double delta_w) {
  float *x,*y;
  int i;

  if (start_w==0.0)
    start_w=1.0;
  if (delta_w==0.0)
    delta_w=1.0;

  x = (float *)malloc(npix*sizeof(float));
  y = (float *)malloc(npix*sizeof(float));
  for (i=0;i<npix;i++) {
    x[i]=(float)start_w+(float)delta_w*i;
    y[i]=data[i];
  }

  cpgpage();
  cpgenv(x[0],x[npix-1],datamin,datamax,0,0);
  cpglab("\\gLambda", "Flux", "Spectra");
  cpgsls(1);
  cpgpt(npix,x,y,1);
  cpgline(npix,x,y);

}


/*
 Subroutine that plot the image.
*/
void imageplot(float* data, float datamin, float datamax, int l_naxes[2], int id, char *title) {
  float tr[]={0.0,1.0,0.0,0.0,0.0,1.0};
  /*
   * We select a certain device
   */
  cpgslct(id);
  cpgpage();
  setvp();
  cpgwnad(0.0,1.0+l_naxes[0],0.0,1.0+l_naxes[1]);
  /* Defines the original palette */
  palett(p,sign*contra,bright);
  /* Plot the image */
  cpgimag(data,l_naxes[0],l_naxes[1],1,l_naxes[0],1,l_naxes[1],datamin,datamax,tr);
  cpgmtxt("t",1.0,0.0,0.0,title);
  cpgsch(0.6);
  cpgbox("bcntsi",0.0,0,"bcntsiv",0.0,0);
  cpgmtxt("b",3.0,1.0,1.0,"pixel number");
  /* Draws the Scale */
  cpgwedg("BI",4.0,5.0,datamin,datamax,"pixel value");
  cpgsch(1.0);
}

void setvp() {
  float d, vpx1, vpx2, vpy1, vpy2;
  cpgsvp(0.0,1.0,0.0,1.0);
  cpgqvp(1,&vpx1,&vpx2,&vpy1,&vpy2);
  d=minimum(vpx2-vpx1,vpy2-vpy1)/40.0;
  vpx1=vpx1+5.0*d;
  vpx2=vpx2-2.0*d;
  vpy1=vpy1+8.0*d;
  vpy2=vpy2-2.0*d;
  cpgvsiz(vpx1,vpx2,vpy1,vpy2);
  return;
}

float minimum(float a, float b) {
  float c;
  if (a<b) {
    c=a;
  } else {
    c=b;
  }
  return c;
}

float maximum(float a, float b) {
  float c;
  if (a>b) {
    c=a;
  } else {
    c=b;
  }
  return c;
}

/*
 * This subrutine changes the palette.
 * It is very useful, since it has the colour ranges.
 */
void palett(int type, float contra, float bright){
  float GL[]={0.0, 1.0};
  float GR[]={0.0, 1.0};
  float GG[]={0.0, 1.0};
  float GB[]={0.0, 1.0};

  float RL[]={-0.5, 0.0, 0.17, 0.33, 0.50, 0.67, 0.83, 1.0, 1.7};
  float RR[]={ 0.0, 0.0,  0.0,  0.0,  0.6,  1.0,  1.0, 1.0, 1.0};
  float RG[]={ 0.0, 0.0,  0.0,  1.0,  1.0,  1.0,  0.6, 0.0, 1.0};
  float RB[]={ 0.0, 0.3,  0.8,  1.0,  0.3,  0.0,  0.0, 0.0, 1.0};

  float HL[]={0.0, 0.2, 0.4, 0.6, 1.0};
  float HR[]={0.0, 0.5, 1.0, 1.0, 1.0};
  float HG[]={0.0, 0.0, 0.5, 1.0, 1.0};
  float HB[]={0.0, 0.0, 0.0, 0.3, 1.0};

  float WL[]={0.0, 0.5, 0.5, 0.7, 0.7, 0.85, 0.85, 0.95, 0.95, 1.0};
  float WR[]={0.0, 1.0, 0.0, 0.0, 0.3,  0.8,  0.3,  1.0,  1.0, 1.0};
  float WG[]={0.0, 0.5, 0.4, 1.0, 0.0,  0.0,  0.2,  0.7,  1.0, 1.0};
  float WB[]={0.0, 0.0, 0.0, 0.0, 0.4,  1.0,  0.0,  0.0, 0.95, 1.0};

  float AL[]={0.0, 0.1, 0.1, 0.2, 0.2, 0.3, 0.3, 0.4, 0.4, 0.5,
     0.5, 0.6, 0.6, 0.7, 0.7, 0.8, 0.8, 0.9, 0.9, 1.0};
  float AR[]={0.0, 0.0, 0.3, 0.3, 0.5, 0.5, 0.0, 0.0, 0.0, 0.0,
              0.0, 0.0, 0.0, 0.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0};
  float AG[]={0.0, 0.0, 0.3, 0.3, 0.0, 0.0, 0.0, 0.0, 0.8, 0.8,
              0.6, 0.6, 1.0, 1.0, 1.0, 1.0, 0.8, 0.8, 0.0, 0.0};
  float AB[]={0.0, 0.0, 0.3, 0.3, 0.7, 0.7, 0.7, 0.7, 0.9, 0.9,
              0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};

  switch(type) {
  case (1): {
    /* Grey */
    cpgctab(GL, GR, GG, GB, 2, contra, bright);
    break;
  }
  case (2): {
    /* Rainbow */
    cpgctab(RL, RR, RG, RB, 9, contra, bright);
    break;
  }
  case (3): {
    /* Heat */
    cpgctab(HL, HR, HG, HB, 5, contra, bright);
    break;
  }
  case (4): {
    /* Weird IRAF */
    cpgctab(WL, WR, WG, WB, 10, contra, bright);
    break;
  }
  case (5): {
    /* AIPS */
    cpgctab(AL, AR, AG, AB, 20, contra, bright);
    break;
  }
  default: {
    cpgctab(GL, GR, GG, GB, 2, contra, bright);
    break;
  }
  }
}

void fiddle(int l_naxes[2], char* ch, int* real_x, int* real_y, int old_x, int old_y) {
  float x=0.5,y=1.0;
  float x1,y1,x2,y2;
  float b1=0.0,b2=1.0,c1=0.0,c2=1.0;
  int fuera=0;

  sign=1;
  cpgqwin(&x1,&x2,&y1,&y2);
  cpgswin(b1,b2,c1,c2);
  while ((toupper(ch[0])!='Q')&&(fuera==0)) {
    if (cpgband(mode,0,(float)(old_x/l_naxes[0]),(float)(old_y/l_naxes[1]),&x,&y,ch)) {
      *real_x=(int) (l_naxes[0]*x);
      *real_y=(int) (l_naxes[1]*y);

      if (*real_x<0) *real_x=0;
      if (*real_x>l_naxes[0]) *real_x=l_naxes[0];
      if (*real_y<0) *real_y=0;
      if (*real_y>l_naxes[1]) *real_y=l_naxes[1];

      switch(toupper(ch[0])) {
      case('H'): {
        fuera=1;
        break;
      }
      case('Z'): {
        fuera=1;
        break;
      }
      case('W'): {
        fuera=1;
        break;
      }
      case('O'): {
        fuera=1;
        break;
      }
      case('R'): {
        fuera=1;
        break;
      }
      case('S'): {
        fuera=1;
        break;
      }
      case('V'): {
        fuera=1;
        break;
      }
      case('M'): {
        fuera=1;
        break;
      }
      case('L'): {
        fuera=1;
        break;
      }
      case('D'): {
        fuera=1;
        break;
      }
      case('B'): {
        fuera=1;
        break;
      }
      case('N'): {
        fuera=1;
        break;
      }
      case('E'): {
        fuera=1;
        break;
      }
      case('C'): {
        fuera=1;
        break;
      }
      case('G'): {
        bright= maximum(b1,minimum(b2,x));
        contra= maximum(c1,minimum(c2,y));
        palett(p,sign*contra,bright);
        break;
      }
      case('I'): {
        fuera=1;
        break;
      }
      case('Q'): {
        cpgswin(x1,x2,y1,y2);
        break;
      }
      case('F'): {
        bright= maximum(b1,minimum(b2,x));
        contra= maximum(c1,minimum(c2,y));
        fuera=1;
        break;
      }
      case(';'): {
        contra=1.0;
        y=1.0;
        bright=0.5;
        x=0.5;
        break;
      }
      case('U'): {
        printf("c=%f,b=%f\n",contra,bright);
        break;
      }
      case('-'): {
        sign=-sign;
        break;
      }
      case('1'): {
        p=1;
        break;
      }
      case('2'): {
        p=2;
        break;
      }
      case('3'): {
        p=3;
        break;
      }
      case('4'): {
        p=4;
        break;
      }
      case('5'): {
        p=5;
        break;
      }
      case('P'): {
        fuera=1;
        break;
      }
      }
      palett(p,sign*contra,bright);
    }
  }
}
