/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
! COPYRIGHT    (c)  2002 AIP, Potsdam, Germany
! IDENT        tk_e3d
! LANGUAGE     C
! AUTHOR       S.F.Sanchez
! KEYWORDS
! PURPOSE      TK interpreter of the Visualization tool
! COMMENT      Full version
! VERSION      1.1  2003-Dec-22 : Creation, SF. Sanchez
------------------------------------------------------------------------------*/
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "my_math.h"
#include <ctype.h> //For Type check
#include <tk.h>
#include <time.h>

#include <IFU_io.h>  //For Lyon I/O Library
#include <data_io.h> //For Lyon I/O data

/******************************
 * For Shared Memory
 ******************************/
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/shm.h>

#include "tkpgplot.h"
#include "cpgplot.h"

//#if defined(USE_THREADS)
//#include <pthread.h> //POSIX Threads
//#endif

#include "my_grid.h"
#include "csa.h"
#include "nn.h"
#include "nan.h"

#include <Euro3D.h>
#include <fft2d.h>
#include <my_math.h>
#include <my_convolve.h>
#include <gl_lsq.h>
#include <stats.h>

#define MAX_SHM 33554432

#define MAX_NUM_OBJ 10
#define MAX_NUM_SPEC 10
#define MAX_NUM_MAPS 10
#define MAX_NUM_VEL_MAPS 20

/**********************************************************
 * PGPLOT routines
 **********************************************************/
static int pgenv(ClientData data, Tcl_Interp *interp, int argc, char *argv[]);
static int create_env(ClientData data, Tcl_Interp *interp, int argc, char *argv[]);

static int Demo_AppInit(Tcl_Interp *interp);
static void display_message(int device_id, char *text);
int open_device(Tcl_Interp *interp, char *device);
static int close_device(int device_id);
int new_window(Tcl_Interp *interp, char *caller, char *cmd, char *device);

/**********************************************************************
 * Different funcions associated with the TK_Commands
**********************************************************************/
static int open_device_command(Tcl_Interp *interp, int argc, char *argv[]);
static int window_instance_command(ClientData data, Tcl_Interp *interp,
                                   int argc, char *argv[]);
static int message_command(Tcl_Interp *interp, int argc, char *argv[]);
static int close_device_command(Tcl_Interp *interp, int argc, char *argv[]);
static int clear_device_command(Tcl_Interp *interp, int argc, char *argv[]);
static int start_server_command(Tcl_Interp *interp, int argc, char *argv[]);
static int stop_server_command(Tcl_Interp *interp, int argc, char *argv[]);
static int clean_server_command(Tcl_Interp *interp, int argc, char *argv[]);
static int load_file_command(Tcl_Interp *interp, int argc, char *argv[]);
static int save_file_command(Tcl_Interp *interp, int argc, char *argv[]);
static int save_obj_file_command(Tcl_Interp *interp, int argc, char *argv[]);
static int draw_raw_command(Tcl_Interp *interp, int argc, char *argv[]);
static int recolor_image_command(Tcl_Interp *interp, int argc, char *argv[]);
static int ask_e3d_info_command(Tcl_Interp *interp, int argc, char *argv[]);
static int intensity_command(Tcl_Interp *interp, int argc, char *argv[]);
static int bc_command(Tcl_Interp *interp, int argc, char *argv[]);
static int plot_line_command(Tcl_Interp *interp, int argc, char *argv[]);
static int plot_spaxels_command(Tcl_Interp *interp, int argc, char *argv[]);
static int create_slice_command(Tcl_Interp *interp, int argc, char *argv[]);
static int plot_spectra_command(Tcl_Interp *interp, int argc, char *argv[]);
static int save_spectra_sel_command(Tcl_Interp *interp, int argc, char *argv[]);
static int load_spectra_sel_command(Tcl_Interp *interp, int argc, char *argv[]);
static int load_n_spectra_sel_command(Tcl_Interp *interp, int argc, char *argv[]);
static int nearest_spax_command(Tcl_Interp *interp, int argc, char *argv[]);
static int mark_range_command(Tcl_Interp *interp, int argc, char *argv[]);
static int create_obj_command(Tcl_Interp *interp, int argc, char *argv[]);
static int reverse_obj_command(Tcl_Interp *interp, int argc, char *argv[]);
static int load_obj_command(Tcl_Interp *interp, int argc, char *argv[]);
static int create_spec_command(Tcl_Interp *interp, int argc, char *argv[]);
static int load_spec_command(Tcl_Interp *interp, int argc, char *argv[]);
static int plot_spectra_mem_command(Tcl_Interp *interp, int argc, char *argv[]);
static int sub_spec_command(Tcl_Interp *interp, int argc, char *argv[]);
static int skysub_command(Tcl_Interp *interp, int argc, char *argv[]);
static int div_spec_command(Tcl_Interp *interp, int argc, char *argv[]);

static int specarith_command(Tcl_Interp *interp, int argc, char *argv[]);
static int slicearith_command(Tcl_Interp *interp, int argc, char *argv[]);

static int create_map_command(Tcl_Interp *interp, int argc, char *argv[]);
static int plot_map_command(Tcl_Interp *interp, int argc, char *argv[]);

static int save_map_command(Tcl_Interp *interp, int argc, char *argv[]);
static int save_slice_command(Tcl_Interp *interp, int argc, char *argv[]);
static int theoretical_dar_command(Tcl_Interp *interp, int argc, char *argv[]);
static int correct_dar_command(Tcl_Interp *interp, int argc, char *argv[]);
static int empirical_dar_command(Tcl_Interp *interp, int argc, char *argv[]);
static int simple_dar_command(Tcl_Interp *interp, int argc, char *argv[]);
static int pm_dar_command(Tcl_Interp *interp, int argc, char *argv[]);
static int peak_dar_command(Tcl_Interp *interp, int argc, char *argv[]);

static int zoom_raw_command(Tcl_Interp *interp, int argc, char *argv[]);
static int release_raw_command(Tcl_Interp *interp, int argc, char *argv[]);
static int tcl_pgptxt_command(Tcl_Interp *interp, int argc, char *argv[]);

static int save_spec_command(Tcl_Interp *interp, int argc, char *argv[]);
static int save_spec_table_command(Tcl_Interp *interp, int argc, char *argv[]);

static int set_first_last_command(Tcl_Interp *interp, int argc, char *argv[]);
static int set_grid_command(Tcl_Interp *interp, int argc, char *argv[]);

static int save_raw_command(Tcl_Interp *interp, int argc, char *argv[]);
static int save_cut_command(Tcl_Interp *interp, int argc, char *argv[]);
static int read_vel_map_command(Tcl_Interp *interp, int argc, char *argv[]);
static int read_cont_map_command(Tcl_Interp *interp, int argc, char *argv[]);
static int plot_scale_command(Tcl_Interp *interp, int argc, char *argv[]);

static int set_flag_spec_command(Tcl_Interp *interp, int argc, char *argv[]);
static int save_PT_command(Tcl_Interp *interp, int argc, char *argv[]);
static int export_rss_command(Tcl_Interp *interp, int argc, char *argv[]);
static int import_rss_command(Tcl_Interp *interp, int argc, char *argv[]);
static int export_cube_command(Tcl_Interp *interp, int argc, char *argv[]);
static int import_cube_command(Tcl_Interp *interp, int argc, char *argv[]);
static int import_map_command(Tcl_Interp *interp, int argc, char *argv[]);
static int import_spectra_command(Tcl_Interp *interp, int argc, char *argv[]);
static int encircled_spax_command(Tcl_Interp *interp, int argc, char *argv[]);
static int flux_spax_command(Tcl_Interp *interp, int argc, char *argv[]);
static int stats_spax_command(Tcl_Interp *interp, int argc, char *argv[]);
static int clean_device_command(Tcl_Interp *interp, int argc, char *argv[]);
static int set_type_spec_rep_command(Tcl_Interp *interp, int argc, char *argv[]);
static int fit_single_line_command(Tcl_Interp *interp, int argc, char *argv[]);
static int guess_single_line_command(Tcl_Interp *interp, int argc, char *argv[]);
static int feed_slice_command(Tcl_Interp *interp, int argc, char *argv[]);
static int plot_flux_spax_command(Tcl_Interp *interp, int argc, char *argv[]);
static int zoom_spax_command(Tcl_Interp *interp, int argc, char *argv[]);
static int unzoom_spax_command(Tcl_Interp *interp, int argc, char *argv[]);
// 22.06.04
static int save_dar_command(Tcl_Interp *interp, int argc, char *argv[]);
static int load_dar_command(Tcl_Interp *interp, int argc, char *argv[]);
// 09.08.04
static int set_log_file_command(Tcl_Interp *interp, int argc, char *argv[]);
static int get_log_file_command(Tcl_Interp *interp, int argc, char *argv[]);
static int set_write_log_command(Tcl_Interp *interp, int argc, char *argv[]);
static int write_in_log_command(Tcl_Interp *interp, int argc, char *argv[]);

/*************************
 * Auxiliar Silly functions
 *************************/
int save_def_max();
int restore_def_max();
int func_create_map(int i0,int i1, float dpix1, float dpix2, int square, int n_m);

/********************
 * Euro3d Definitions
 ********************/
int delta_n=512;
E3D_file e3d_image;
SPAXEL **spaxels;
int *n_spaxels,*index_start,*npts;
GROUP *groups;
double start_w, end_w, delta_w;
int npix,npix_w1,npix_w2,old_npix_w1,old_npix_w2;
int nb_spec,n_groups;
float *raw_data,*raw_noise;
int j;
float datamin,datamax;
float tr[]={0.0,1.0,0.0,0.0,0.0,1.0};
float minmax[2];
float x_min,y_min,x_max,y_max;
int aspect=1;

float x_min_def,y_min_def,x_max_def,y_max_def,max_size_def;

int i_slices[8][2];
int last_i0,last_i1;
int *i_spax; // Actual Selection of Spaxels == Obj
int n_i_spax=0; //Number of elements in 'i_spax'
int *j_spax[MAX_NUM_OBJ]; // Array of Selection of Spaxels == Obj
int n_j_spax[MAX_NUM_OBJ]; // Array of their Number of elements
int que_j_spax=0; // Actual Obj selected.
int slice_index=0;
int i0_slice=0;

int flag_spec=0;
float *f_spec_now; // Array of Selection of Spectra
float *f_spec[MAX_NUM_SPEC]; // Array of Selection of Spectra
int n_f_spec=0; //Index of the actual selected spectra
int start_n_f_spec=0;

/* Slice Buffer */
float *f_slice_now; // Array of Selection of Slices
float *f_slice[MAX_NUM_MAPS]; // Array of Selection of Spectra
int f_slice_def[MAX_NUM_MAPS];

float *map_data[MAX_NUM_MAPS];
int nx_map[MAX_NUM_MAPS],ny_map[MAX_NUM_MAPS];
int n_map=0;
float tr_map[8][MAX_NUM_MAPS];
int map_created[MAX_NUM_MAPS];
float dpix1=1.0, dpix2=1.0;
float max_size;
int grid_func=2;
float grid_opt=0.0;

int server_started=0;

struct DAR *dars;
int dars_defined=0;
int dars_valid=0; // If 1 the DAR is valid.

int loaded=0;
int spax_painted=0;
int dar_corrected=0;

int pintados[30];
int n_pintados=30;
int old_p[30];
int old_text_flag[30];
int n_sel_spec=0;
float old_max[30],old_min[30];
int old_spax_id[30];

int n_contours=15;
int contours_color=1;

/* Variables for plotting only a certain range of the spectra */
float *tmp_data;
int first_spec=0;
int last_spec=256;

/* Velocity Maps */
int n_vel_map=0;
int vel_map_fill[MAX_NUM_VEL_MAPS];
struct VELOCITY_MAP vel_map[MAX_NUM_VEL_MAPS];

SPECTRUM signal_tot, noise_tot;

int type_spec_rep=0;
int type_spec_save=0;
int type_spec_n_buffer=-1;

/* Logs definitions */
int write_log=0;
char log_file[80];
int event=0;
int new_log=0;

int is_e3d=0;
int is_cube=0;
int is_rss=0;

/*.......................................................................
 * After presenting a warning if the first argument is not the name
 * of the demo Tcl script, main() simply calls the standard Tk_Main()
 * to initialize Tcl/Tk and the demo package.
 * Input:
 *  argc     int    The number of command line arguments.
 *  argv    char*[] The array of command-line argument strings.
 * Output:
 *  return   int    0 - OK.
 *                  1 - Error.
 */
int main(int argc, char *argv[])
{
  char **argval, **arglabel;
  //char *usage = "Usage: tk_test tk_test.tcl [tk-options].\n";

  /* before starting, override any IFU_DEFAULT_FMT variable, *
   * within E3D we only need Euro3D                          */
  setenv("IFU_DEFAULT_FMT","Euro3D",1);

  set_version(VERSION);
  set_arglist("-log|logfile e3d.log");
  init_session(argv,argc,&arglabel,&argval);
  set_control_level(NONE);
  get_argval(0,"%s",log_file);

  Tk_Main(argc, argv, Demo_AppInit);

  exit_session(0);
  return 0;
}

/*.......................................................................
 * This dummy fortran main allows pgtkdemo to be linked with the
 * f2c-compiled pgplot library.
 */
int MAIN__(void)
{
  return 0;
}

/*.......................................................................
 * This is the application initialization file that is called by Tk_Main().
 */
static int Demo_AppInit(Tcl_Interp *interp)
{
  /*
   * Create the standard Tcl and Tk packages, plus the TkPgplot package.
   */
  if (Tcl_Init(interp)      == TCL_ERROR ||
      Tk_Init(interp)       == TCL_ERROR ||
      Tkpgplot_Init(interp) == TCL_ERROR)
    return 1;
  /*
   * Create the TCL command used to initialization the demo.
   */
  Tcl_CreateCommand(interp, "create_env", (Tcl_CmdProc *)create_env,
                    (ClientData)Tk_MainWindow(interp), (Tcl_CmdDeleteProc *)NULL);
  Tcl_CreateCommand(interp, "pgenv", (Tcl_CmdProc *)pgenv, (ClientData)Tk_MainWindow(interp),
                    (Tcl_CmdDeleteProc *)NULL);
  return 0;
}

/*.......................................................................
 * This function provides the TCL command that creates a pgdemo
 * manipulation command. This opens the two given PGPLOT widgets to
 * PGPLOT, establishes a cursor handler and records the state of the
 * demo in a dynamically allocated container.
 *
 * Input:
 *  data      ClientData    The main window cast to ClientData.
 *  interp    Tcl_Interp *  The TCL intrepreter of the demo.
 *  argc             int    The number of command arguments.
 *  argv            char ** The array of 'argc' command arguments.
 *                          argv[0] = "create_env"
 *                          argv[1] = The name to give the new command.
 *                          argv[2] = The name of the image widget.
 *                          argv[3] = The name of the slice widget.
 * Output:
 *  return           int    TCL_OK    - Success.
 *                          TCL_ERROR - Failure.
 */
static int create_env(ClientData data, Tcl_Interp *interp, int argc, char *argv[])
{
  int dev_id;
  int i;
  /*
   * Check that the right number of arguments was provided.
   */
  if (argc != 3) {
    Tcl_AppendResult(interp,
             argv[0], ": Wrong number of arguments - should be \'",
             argv[0], " new_command_name image_widget\'", NULL);
    return TCL_ERROR;
  };
  /*
   * Allocate a context object for the command.
   */
  for (i=0;i<MAX_NUM_VEL_MAPS;i++) {
      vel_map_fill[i]=0;
  }
  dev_id=new_window(interp, argv[0], argv[1], argv[2]);

  if (!dev_id)
    return TCL_ERROR;
  return TCL_OK;
}

/********************************************************************
 * Creates a new device
 ********************************************************************/
int open_device(Tcl_Interp *interp, char *device) {
  int dev_id;

  if (!(dev_id = cpgopen(device))) {
    Tcl_AppendResult(interp, "Unable to open widgets: ", device, NULL);
    return close_device(dev_id);
  };
  return dev_id;
}

/****************************************************
 * Close the device
 ****************************************************/
static int close_device(int device_id) {
  cpgslct(device_id);
  cpgclos();
  return 0;
}

/*.......................................................................
 * Display a "Please wait" message in the slice window.
 *
 * Input:
 *  demo     Pgdemo *   The demo instance object.
 */
static void display_message(int device_id, char *text)
{
  /*
   * Clear the slice plot and replace it with instructional text.
   */
  cpgslct(device_id);
  cpgask(0);
  cpgpage();
  cpgsch(3.0f);
  cpgsvp(0.0, 1.0, 0.0, 1.0);
  cpgswin(0.0, 1.0, 0.0, 1.0);
  cpgmtxt("T", -2.0, 0.5, 0.5, text);
}

int new_window(Tcl_Interp *interp, char *caller, char *cmd, char *image_device) {
  int dev_id;

  if ((dev_id=open_device(interp, image_device))<=0) {
    Tcl_AppendResult(interp, "Unable to open widgets: ", image_device, NULL);
  }
  cpgslct(dev_id);

  Tcl_CreateCommand(interp, cmd, (Tcl_CmdProc *)window_instance_command,NULL,NULL);
  /*
   * Return the command name.
   */
  Tcl_AppendResult(interp, cmd, NULL);

  return dev_id;
}

/**************************************************************
 * Open a Device via the Server
 **************************************************************/
static
int open_device_command(Tcl_Interp *interp, int argc, char *argv[]) {
  int dev_id;
  if (argc < 1) {
    Tcl_SetResult(interp, "USE: open_device DEVICE", TCL_STATIC);
    return TCL_ERROR;
  };
  dev_id=open_device(interp,argv[0]);
  sprintf(interp->result,"%d",dev_id);
  return TCL_OK;
}

static int window_instance_command(ClientData data, Tcl_Interp *interp, int argc, char *argv[]) {
  char *command;
  /*
   * We must have at least one command argument.
   */
  if (argc < 2) {
    Tcl_SetResult(interp, "USE: create_env NAME_ENV PGPLOT_DEVICE_NAME", TCL_STATIC);
    return TCL_ERROR;
  };

  command = argv[1];
  if ((strcmp(command, "intensity")    != 0) &&
      (strcmp(command, "nearest_spax") != 0) &&
      (strcmp(command, "load_file")    != 0) &&
      (strcmp(command, "import_cube")  != 0) &&
      (strcmp(command, "import_rss")   != 0) &&
      (strcmp(command, "flux_spax")    != 0)) {
    if (write_log==1) {
      if (new_log==1) {
        add_to_file(log_file,&argc,argv);
      } else {
        write_to_file(log_file,&argc,argv);
        new_log=1;
      }
    }
  }

  /*
   * Get the command-name argument.
   */
  if (strcmp(command, "message") == 0)
    return message_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "close_device") == 0)
    return close_device_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "clear_device") == 0)
    return clear_device_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "start_server") == 0)
    return start_server_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "stop_server") == 0)
    return stop_server_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "clean_server") == 0)
    return clean_server_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "load_file") == 0)
    return load_file_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "save_file") == 0)
    return save_file_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "save_obj_file") == 0)
    return save_obj_file_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "draw_raw") == 0)
    return draw_raw_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "recolor_image") == 0)
    return recolor_image_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "ask_e3d_info") == 0)
    return ask_e3d_info_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "open_device") == 0)
    return open_device_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "intensity") == 0)
    return intensity_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "bc") == 0)
    return bc_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "plot_line") == 0)
    return plot_line_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "plot_spaxels") == 0)
    return plot_spaxels_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "create_slice") == 0)
    return create_slice_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "plot_spectra") == 0)
    return plot_spectra_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "plot_spectra_mem") == 0)
    return plot_spectra_mem_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "save_spectra_sel") == 0)
    return save_spectra_sel_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "load_spectra_sel") == 0)
    return load_spectra_sel_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "load_n_spectra_sel") == 0)
    return load_n_spectra_sel_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "nearest_spax") == 0)
    return nearest_spax_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "flux_spax") == 0)
    return flux_spax_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "plot_flux_spax") == 0)
    return plot_flux_spax_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "stats_spax") == 0)
    return stats_spax_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "encircled_spax") == 0)
    return encircled_spax_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "mark_range") == 0)
    return mark_range_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "create_obj") == 0)
    return create_obj_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "reverse_obj") == 0)
    return reverse_obj_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "load_obj") == 0)
    return load_obj_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "create_spec") == 0)
    return create_spec_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "load_spec") == 0)
    return load_spec_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "sub_spec") == 0)
    return sub_spec_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "skysub") == 0)
    return skysub_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "specarith") == 0)
    return specarith_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "slicearith") == 0)
    return slicearith_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "div_spec") == 0)
    return div_spec_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "create_map") == 0)
    return create_map_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "export_cube") == 0)
    return export_cube_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "save_map") == 0)
    return save_map_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "save_spec") == 0)
    return save_spec_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "save_raw") == 0)
    return save_raw_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "save_cut") == 0)
    return save_cut_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "save_spec_table") == 0)
    return save_spec_table_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "save_slice") == 0)
    return save_slice_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "save_PT") == 0)
    return save_PT_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "export_rss") == 0)
    return export_rss_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "import_rss") == 0)
    return import_rss_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "import_spectra") == 0)
    return import_spectra_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "import_cube") == 0)
    return import_cube_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "import_map") == 0)
    return import_map_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "plot_map") == 0)
    return plot_map_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "theoretical_dar") == 0)
    return theoretical_dar_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "correct_dar") == 0)
    return correct_dar_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "empirical_dar") == 0)
    return empirical_dar_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "simple_dar") == 0)
    return simple_dar_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "pm_dar") == 0)
    return pm_dar_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "peak_dar") == 0)
    return peak_dar_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "zoom_raw") == 0)
    return zoom_raw_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "release_raw") == 0)
    return release_raw_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "clean_device") == 0)
    return clean_device_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "tcl_pgptxt") == 0)
    return tcl_pgptxt_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "set_first_last") == 0)
    return set_first_last_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "set_grid") == 0)
    return set_grid_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "set_flag_spec") == 0)
    return set_flag_spec_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "read_cont_map") == 0)
    return read_cont_map_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "read_vel_map") == 0)
    return read_vel_map_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "plot_scale") == 0)
    return plot_scale_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "set_type_spec_rep") == 0)
    return set_type_spec_rep_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "fit_single_line") == 0)
    return fit_single_line_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "guess_single_line") == 0)
    return guess_single_line_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "feed_slice") == 0)
    return feed_slice_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "zoom_spax") == 0)
    return zoom_spax_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "unzoom_spax") == 0)
    return unzoom_spax_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "save_dar") == 0)
    return save_dar_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "load_dar") == 0)
    return load_dar_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "set_log_file") == 0)
    return set_log_file_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "get_log_file") == 0)
    return get_log_file_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "set_write_log") == 0)
    return set_write_log_command(interp, argc - 2, argv + 2);
  else if (strcmp(command, "write_in_log") == 0)
    return write_in_log_command(interp, argc - 2, argv + 2);

  /* Unknown command name.  */
  Tcl_AppendResult(interp, argv[0], ": Unknown command \"",
                   argv[1], "\"", NULL);
  return TCL_ERROR;
}

/*******************************************************
 * Defines the name of the Log File variable
 *******************************************************/
static int set_log_file_command(Tcl_Interp *interp, int argc, char *argv[]) {
  if (argc < 1) {
    Tcl_AppendResult(interp, "USE: set_log_file log_file", NULL);
    return TCL_ERROR;
  }
  strcpy(log_file,argv[0]);
  return TCL_OK;
}

/*******************************************************
 * Gets the current log file
 *******************************************************/
static int get_log_file_command(Tcl_Interp *interp, int argc, char *argv[]) {
  sprintf(interp->result,"%s",log_file);
  return TCL_OK;
}

/*******************************************************
 * Defines if to write a log file or not
 *******************************************************/
static int set_write_log_command(Tcl_Interp *interp, int argc, char *argv[]) {
  if (argc < 1) {
    Tcl_AppendResult(interp, "USE: set_write_log 0/1", NULL);
    return TCL_ERROR;
  }
  write_log=atoi(argv[0]);
  if ((write_log!=0)&&(write_log!=1)) {
      write_log=0;
  }
  return TCL_OK;
}

/*******************************************************
 * Write in the log file
 *******************************************************/
static int write_in_log_command(Tcl_Interp *interp, int argc, char *argv[]) {
  if (argc < 1) {
    Tcl_AppendResult(interp, "USE: write_in_log ARGs", NULL);
    return TCL_ERROR;
  }
  add_to_file(log_file,&argc,argv);
  return TCL_OK;
}

/*******************************************************
 * Load a DAR correction file (ID,DX,DY)
 *******************************************************/
static int load_dar_command(Tcl_Interp *interp, int argc, char *argv[]) {
  if (argc < 1) {
    Tcl_AppendResult(interp, "USE: load_dar filename", NULL);
    return TCL_ERROR;
  }
  load_dar(argv[0],dars,npix,n_groups);
  print_msg("DAR file %s loaded",argv[0]);
  dars_defined=1;
  return TCL_OK;
}

/*******************************************************
 * Save a DAR correction file (ID,DX,DY)
 *******************************************************/
static int save_dar_command(Tcl_Interp *interp, int argc, char *argv[]) {
  if (argc < 1) {
    Tcl_AppendResult(interp, "USE: save_dar filename", NULL);
    return TCL_ERROR;
  }
  if ((strlen(argv[0])==0)||(strcmp(argv[0],".fits")==0)||(strcmp(argv[0],".fit")==0)||(strcmp(argv[0],".e3d")==0)) {
      Tcl_AppendResult(interp, "No output file defined", NULL);
      return TCL_ERROR;
  }
  if (dars_defined==1) {
      save_dar(argv[0],dars,npix,n_groups);
      print_msg("DAR file %s saved",argv[0]);
  } else {
      print_msg("DAR is not defined");
  }
  return TCL_OK;
}

/*******************************************************
 * Redefine the Plotting area of the Spaxels
 * to de Default values
 *******************************************************/
static int unzoom_spax_command(Tcl_Interp *interp, int argc, char *argv[]) {
  int id;

  if (argc < 1) {
    Tcl_AppendResult(interp, "USE: unzoom_spax dev_id", NULL);
    return TCL_ERROR;
  }
  id=atoi(argv[0]);
  restore_def_max();
  pintados[id]=0;
  aspect=1;
  return TCL_OK;
}

/*******************************************************
 * Redefine the Plotting area of the Spaxels, producing
 * a Zooming effect (I hope!).
 *******************************************************/
static int zoom_spax_command(Tcl_Interp *interp, int argc, char *argv[]) {
  float lx_min,lx_max,ly_min,ly_max,lmax_size,ftmp;
  int id;

  if (argc < 6) {
    Tcl_AppendResult(interp, "USE: zoom_spax dev_id x_min x_max y_min y_max bordersize [aspect]", NULL);
    return TCL_ERROR;
  }

  id=atoi(argv[0]);
  lx_min=atof(argv[1]);
  lx_max=atof(argv[2]);
  ly_min=atof(argv[3]);
  ly_max=atof(argv[4]);
  lmax_size=atof(argv[5]);
  if (argc==7) {
    aspect=atoi(argv[6]);
  }
  if (lx_min>lx_max) {
    ftmp=lx_max;
    lx_max=lx_min;
    lx_min=ftmp;
  }

  if (ly_min>ly_max) {
    ftmp=ly_max;
    ly_max=ly_min;
    ly_min=ftmp;
  }
  x_max=lx_max;
  y_max=ly_max;
  x_min=lx_min;
  y_min=ly_min;
  /*
   * We define the scales
   */
  if (aspect==1) {
    if (fabs(x_max-x_min)>fabs(y_max-y_min)) {
      y_max=(y_max+y_min)/2+fabs(x_max-x_min)/2;
      y_min=(y_max+y_min)/2-fabs(x_max-x_min)/2;
    } else {
      x_max=(x_max+x_min)/2+fabs(y_max-y_min)/2;
      x_min=(x_max+x_min)/2-fabs(y_max-y_min)/2;
    }
  }
  max_size=lmax_size;
  pintados[id]=0;

  return TCL_OK;
}

/*******************************************************
 * Read VEL_MAP created by "map.pl" and the Fitting tool
 *******************************************************/
static int read_vel_map_command(Tcl_Interp *interp, int argc, char *argv[]) {
    float resolution;
    float sistemic;
  if (argc < 2) {
    Tcl_AppendResult(interp, "USE: read_vel_map MAPFILE sistemic_Angstroms [resolution]", NULL);
    return TCL_ERROR;
  }
  if (argc==3) {
      resolution=atof(argv[2]);
  } else {
      resolution=(float)delta_w;
  }
  sistemic=atof(argv[1]);
  read_velocity_map(argv[0],&vel_map[n_vel_map],sistemic,resolution);
  sprintf(interp->result,"%d %f %f",n_vel_map,vel_map[n_vel_map].lambda,resolution);
  vel_map_fill[n_vel_map]=1;
  n_vel_map++;
  if (n_vel_map>=MAX_NUM_VEL_MAPS) {
      n_vel_map=0;
  }
  return TCL_OK;
}

/*******************************************************
 * Read VEL_MAP created by "map.pl" and the Fitting tool
 *******************************************************/
static int read_cont_map_command(Tcl_Interp *interp, int argc, char *argv[]) {
  if (argc < 2) {
    Tcl_AppendResult(interp, "USE: read_cont_map MAPFILE WIDTH", NULL);
    return TCL_ERROR;
  }
  read_cont_map(argv[0],&vel_map[n_vel_map],atof(argv[1]));
  sprintf(interp->result,"%d",n_vel_map);
  vel_map_fill[n_vel_map]=1;
  n_vel_map++;
  if (n_vel_map>=MAX_NUM_VEL_MAPS) {
      n_vel_map=0;
  }
  return TCL_OK;
}

/*******************************************************
 * Set GRID Interpolation Options
 *******************************************************/
static int set_grid_command(Tcl_Interp *interp, int argc, char *argv[]) {
  if (argc < 3) {
    Tcl_AppendResult(interp, "USE: set_grid grid_func grid_opt pixel_size", NULL);
    return TCL_ERROR;
  }
  grid_func=atoi(argv[0]);
  grid_opt=atof(argv[1]);
  dpix1=atof(argv[2]);
  dpix2=atof(argv[2]);
#ifdef DEBUG_dpix
  printf("set_grid_command: dpix=%f/%f\n", dpix1,dpix2);
#endif
  return TCL_OK;
}

/*******************************************************
 * Set Spectral Representation Type (type_spec_rep)
 *******************************************************/
static int set_type_spec_rep_command(Tcl_Interp *interp, int argc, char *argv[]) {
  if (argc < 1) {
    /* Tcl_AppendResult(interp, "USE: set_type_spec_rep type save n_buffer", NULL); */
    Tcl_AppendResult(interp, "USE: set_type_spec_rep type", NULL);
    return TCL_ERROR;
  }
  type_spec_rep=atoi(argv[0]);
  /*
   * type_spec_save=atoi(argv[1]);
   * type_spec_n_buffer=atoi(argv[2]);
   */
  type_spec_save=0;
  type_spec_n_buffer=-1;
  /*
   * if ((type_spec_n_buffer!=-1)&&(n_rows[type_spec_n_buffer]==0)) {
   *   type_spec_save=0;
   *   type_spec_n_buffer=-1;
   *   sprintf(interp->result,"-1");
   * } else {
   *   sprintf(interp->result,"0");
   * }
   */
  return TCL_OK;
}

/*******************************************************
 * Set FLAG_SPEC
 *******************************************************/
static int set_flag_spec_command(Tcl_Interp *interp, int argc, char *argv[]) {
  if (argc < 1) {
    Tcl_AppendResult(interp, "USE: set_flag_spec flag_spec", NULL);
    return TCL_ERROR;
  }
  flag_spec=atoi(argv[0]);
  if (flag_spec<0) flag_spec=0;
  if (flag_spec>1) flag_spec=1;
  return TCL_OK;
}

/*******************************************************
 * Set FIRST and LAST spectra Plotted
 *******************************************************/
static int set_first_last_command(Tcl_Interp *interp, int argc, char *argv[]) {
  if (argc < 2) {
    Tcl_AppendResult(interp, "USE: set_first_last first last", NULL);
    return TCL_ERROR;
  }
  first_spec=atoi(argv[0]);
  last_spec=atoi(argv[1]);
  return TCL_OK;
}

/*
 * PGTXT
 */
static int tcl_pgptxt_command(Tcl_Interp *interp, int argc, char *argv[]) {
  float x,y,j,f,size,width;
  int color;
  char text[80];
  if (argc != 8) {
    Tcl_AppendResult(interp,"USE: tcl_pgptxt x y j f text size color width", NULL);
    return TCL_ERROR;
  };
  x=atof(argv[0]);
  y=atof(argv[1]);
  j=atof(argv[2]);
  f=atof(argv[3]);
  strcpy(text,argv[4]);
  size=atof(argv[5]);
  color=floor(atof(argv[6]));
  width=atof(argv[6]);
  cpgsch(size);
  cpgsci(color);
  cpgsfs(3.0);
  cpgptxt(x,y,j,f,text);
  cpgsci(1);
  cpgsch(1.0);
  cpgsfs(1);
  return TCL_OK;
}



/*******************************************************
 *
 * Zoom the RAW data within certain pixels
 *
 *******************************************************/
static int zoom_raw_command(Tcl_Interp *interp, int argc, char *argv[]) {
  int i;
  int id;
  if (argc < 3 || *argv[1] == '-') {
    Tcl_AppendResult(interp, "USE: zoom_raw pix1 pix2 dev_id", NULL);
    return TCL_ERROR;
  }
  npix_w1=atoi(argv[0]);
  npix_w2=atoi(argv[1]);
  id=atoi(argv[2]);
  if (npix_w1>npix_w2) {
    i=npix_w2;
    npix_w2=npix_w1;
    npix_w1=i;
  }
  if (npix_w1<0) npix_w1=0;
  if (npix_w2>npix) npix_w2=npix;
  pintados[id]=0;
  return TCL_OK;
}

/*******************************************************
 *
 * Release the Zoom the RAW data within certain pixels
 *
 *******************************************************/
static int release_raw_command(Tcl_Interp *interp, int argc, char *argv[]) {
  int id;
  if (argc < 1) {
    Tcl_AppendResult(interp, "USE: release_raw dev_id", NULL);
    return TCL_ERROR;
  }
  id=atoi(argv[0]);
  npix_w1=0;
  npix_w2=npix;

  first_spec=0;
  last_spec=256;

  if (last_spec>nb_spec) last_spec=nb_spec;

  pintados[id]=0;
  return TCL_OK;
}

/*******************************************************
 *
 * Set Pintados to 0, for cleaning a device the next time used
 *
 *******************************************************/
static int clean_device_command(Tcl_Interp *interp, int argc, char *argv[]) {
  int id;
  if (argc < 1) {
    Tcl_AppendResult(interp, "USE: clean_device dev_id", NULL);
    return TCL_ERROR;
  }
  id=atoi(argv[0]);
  pintados[id]=0;
  return TCL_OK;
}

/**********************************
 * EMPIRICAL D.A.R. Correction
 *********************************/
static int empirical_dar_command(Tcl_Interp *interp, int argc, char *argv[]) {
  int j,i,k,jj;
  float *slice;
  float tmp_tr[8];
  int nx_t,ny_t;
  float *map_tmp,*map_zero;
  int comp_index,delta_index;
  double *img_data,*psf_data;
  int nx_peak,ny_peak;
  double peak;
  float dpix_tt=0.2;
  int *dnx,*dny,*i_lambda;
  float *fdnx,*fdny,*e_fdnx,*e_fdny,*fi_lambda,*fdnr;
  int nnn;
  int dnx0,dny0;
  float sig_dnx,sig_dny;
  int plot_all=0;
  double *a_x,*a_y;
  int *ia;
  double **covar;
  float chisq;
  int ma=3;
  float *x,*y1,*y2,*r;
  float min_dn,max_dn;
  double *dx,*e_dx,*dy,*e_dy,*xx;
  int n_d;
  int nulo_x=0;
  int nulo_y=0;

  if (argc < 2 ) {
    Tcl_AppendResult(interp, "USE: empirical_dar INDEX_OF_COMPARISON_WAVELENGTH DELTA_INDEX", NULL);
    return TCL_ERROR;
  }

  comp_index=atoi(argv[0]);
  delta_index=atoi(argv[1]);
  if (dars_defined==1) {
    dnx = (int *)malloc(((int)(npix/delta_index))*sizeof(int));
    dny = (int *)malloc(((int)(npix/delta_index))*sizeof(int));
    i_lambda = (int *)malloc(((int)(npix/delta_index))*sizeof(int));

    fdnx = (float *)malloc(((float)(npix/delta_index))*sizeof(float));
    fdny = (float *)malloc(((float)(npix/delta_index))*sizeof(float));
    fdnr = (float *)malloc(((float)(npix/delta_index))*sizeof(float));
    fi_lambda = (float *)malloc(((float)(npix/delta_index))*sizeof(float));

    e_fdnx = (float *)malloc(((float)(npix/delta_index))*sizeof(float));
    e_fdny = (float *)malloc(((float)(npix/delta_index))*sizeof(float));
    nnn=0;

    if (argc>2) {
      dpix_tt=atof(argv[2]);
    }
    if (argc>3) {
      plot_all=atoi(argv[3]);
    }

    if ((comp_index>-1)&&(comp_index<npix-delta_index)) {
      slice = (float *)malloc((nb_spec)*sizeof(float));
      for (i=0;i<nb_spec;i++) {
        slice[i]=0.0;
        for (jj=0;jj<delta_index;jj++) {
          slice[i]=slice[i]+raw_data[comp_index+jj+i*npix];
        }
      }
      create_map(nb_spec,n_spaxels,spaxels,npts,slice,dpix_tt,dpix_tt,&map_zero,x_min,x_max,y_min,y_max,tmp_tr,&nx_t,&ny_t,grid_func,grid_opt);
      img_data = (double *)malloc(nx_t*ny_t*sizeof(double));
      psf_data = (double *)malloc(nx_t*ny_t*sizeof(double));

      if (plot_all>1) {
        cpgslct(1);
        cpgpap(5,1.0);
        cpgask(0);
        cpgpage();
        cpgsch(0.9);
        cpgenv(0,nx_t,0,ny_t,1,0);
        cpgimag(map_zero,nx_t,ny_t,1,nx_t,1,ny_t,0,600,tr);
      }
      for (k=0;k<(nx_t*ny_t);k++) {
        psf_data[k]=(double)map_zero[k]*1.0;
      }

      cpgsch(0.9);
      for (j=0;j<npix-delta_index;j=j+delta_index) {
        i_lambda[nnn]=j+(int)(delta_index/2);

        for (i=0;i<nb_spec;i++) {
          for (jj=0;jj<delta_index;jj++) {
            slice[i]=raw_data[j+jj+i*npix];
          }
        }
        create_map(nb_spec,n_spaxels,spaxels,npts,slice,dpix_tt,dpix_tt,&map_tmp,x_min,x_max,y_min,y_max,tmp_tr,&nx_t,&ny_t,grid_func,grid_opt);
        for (k=0;k<(nx_t*ny_t);k++) {
          img_data[k]=(double)map_tmp[k]*1.0;
        }
        /*
         * Now we crosscorrelate
         */
        if (plot_all>1) {
          cpgimag(map_tmp,nx_t,ny_t,1,nx_t,1,ny_t,0,600,tr);
        }
        my_crosscor(img_data,nx_t,ny_t,psf_data,nx_t,ny_t);
        if (plot_all>1) {
        }
        /*
         * We look for the peak
         */
        nx_peak=0;
        ny_peak=0;
        peak=img_data[0];
        for (i=0;i<ny_t;i++) {
          for (k=0;k<nx_t;k++) {
            if (peak<img_data[k+i*nx_t]) {
              peak=img_data[k+i*nx_t];
              nx_peak=k;
              ny_peak=i;
            }
          }
        }
        dnx[nnn]=nx_peak;
        dny[nnn]=ny_peak;
        nnn++;
        if (plot_all>1) {
          for (k=0;k<(nx_t*ny_t);k++) {
            map_tmp[k]=(float)img_data[k]*1.0;
          }
          cpgimag(map_tmp,nx_t,ny_t,1,nx_t,1,ny_t,0.25*peak,peak*1.5,tr);
        }
      }
      dnx0=dnx[(int)(((float)comp_index/delta_index+0.5))];
      dny0=dny[(int)(((float)comp_index/delta_index+0.5))];

      min_dn=300.0;
      max_dn=-300.0;
      for (i=0;i<nnn;i++) {
        dnx[i]=dnx[i]-dnx0;
        dny[i]=dny[i]-dny0;
        fdnx[i]=(dnx[i])*dpix_tt;
        fdny[i]=(dny[i])*dpix_tt;
        fdnr[i]=sqrt(fdny[i]*fdny[i]+fdnx[i]*fdnx[i]);
        if (min_dn>fdnx[i]) min_dn=fdnx[i];
        if (max_dn<fdnx[i]) max_dn=fdnx[i];
        if (min_dn>fdny[i]) min_dn=fdny[i];
        if (max_dn<fdny[i]) max_dn=fdny[i];
        e_fdnx[i]=dpix_tt/2;
        e_fdny[i]=dpix_tt/2;
        fi_lambda[i]=i_lambda[i]*delta_w+start_w;
      }
      min_dn-=dpix_tt;
      max_dn+=dpix_tt;

      sig_dnx=sigma(fdnx,nnn,0.0);
      sig_dny=sigma(fdny,nnn,0.0);

      /*
       * We clean the data from 1sigma
       */
      xx=vector(1,nnn+1);
      dx=vector(1,nnn+1);
      dy=vector(1,nnn+1);
      e_dx=vector(1,nnn+1);
      e_dy=vector(1,nnn+1);
      n_d=0;
      dx[0]=0.0;
      dy[0]=0.0;
      for (i=0;i<nnn;i++) {
        dx[n_d]=fdnx[i];
        dy[n_d]=fdny[i];
        xx[n_d]=fi_lambda[i];
        if (i>0) {
          if (dx[n_d]!=dx[n_d-1]) nulo_x=1;
          if (dy[n_d]!=dy[n_d-1]) nulo_y=1;
        }
        n_d++;
      }
      /*
       * We update the information of the DAR
       */
      if ((nulo_x==1)&&(nulo_y==1)) {
        covar=matrix(1,ma,1,ma);
        ia=ivector(1,ma);
        a_x=vector(1,ma);
        a_y=vector(1,ma);
        for (k=1;k<=ma;k++) {
          a_x[k]=1.0/k;
          a_y[k]=1.0/k;
          ia[k]=1;
        }

        /*
         * We take first the linear version...
         */
        nulo_x=0;
        nulo_y=0;
        for(j=0;j<2;j++) {
          gl_lfit((float *)xx,(float *)dx,e_fdnx,n_d-1,(float *)a_x,ia,2,(float **)covar,&chisq,poly_serie);
          gl_lfit((float *)xx,(float *)dy,e_fdny,n_d-1,(float *)a_y,ia,2,(float **)covar,&chisq,poly_serie);
          //    free(covar);
          /*
           * We make another clipping taking away 1 sigma
           * from the relation.
           */
          n_d=0;
          for (i=0;i<nnn;i++) {
            if ((fabs(fdnx[i]-a_x[1]-a_x[2]*fi_lambda[i])<sig_dnx)&&(fabs(fdny[i]-a_y[1]-a_y[2]*fi_lambda[i])<sig_dny)) {
              dx[n_d]=fdnx[i];
              dy[n_d]=fdny[i];
              xx[n_d]=fi_lambda[i];
              if (i>0) {
                if (dx[n_d]!=dx[n_d-1]) nulo_x=1;
                if (dy[n_d]!=dy[n_d-1]) nulo_y=1;
              }
              n_d++;
            }
          }
        }

        if ((nulo_x==1)&&(nulo_y==1)) {
          gl_lfit((float *)xx,(float *)dx,e_fdnx,n_d-1,(float *)a_x,ia,ma,(float **)covar,&chisq,poly_serie);
          gl_lfit((float *)xx,(float *)dy,e_fdny,n_d-1,(float *)a_y,ia,ma,(float **)covar,&chisq,poly_serie);
          x = (float *)malloc(npix*sizeof(float));
          y1 = (float *)malloc(npix*sizeof(float));
          y2 = (float *)malloc(npix*sizeof(float));
          r = (float *)malloc(npix*sizeof(float));
          if (plot_all==1) {
            cpgopen("/xterm");
            cpgask(0);
            cpgsch(1.6);
            cpgsubp(1,2);
            cpgenv(start_w,npix*delta_w+start_w,(-1)*sig_dny,sig_dny,0,0);
            cpglab("\\gl (Angstroms)", "\\gDy (arcsec)", "D.A.R");
            cpgpt(nnn,fi_lambda,fdny,4);
            cpgerrb(2,nnn,fi_lambda,fdny,e_fdny,0.2);
            cpgerrb(4,nnn,fi_lambda,fdny,e_fdny,0.4);
            for (i=0;i<npix;i++) {
              x[i]=start_w+delta_w*i;
              y1[i]=0.0;
              y2[i]=0.0;
              for (j=0;j<ma;j++) {
                y1[i]+=a_y[j+1]*pow(x[i],j);
              }

            }
            cpgline(npix,x,y1);
            if (dar_corrected!=0) {
              for (j=0;j<n_groups;j++) {
                cpgsls(j+2);
                cpgline(dars[j].n,x,dars[j].dy);
              }
            }
            cpgsls(1);
            cpgenv(start_w,npix*delta_w+start_w,(-1)*sig_dnx,sig_dnx,0,0);
            cpglab("\\gl (Angstroms)", "\\gDx (arcsec)", "");
            cpgpt(nnn,fi_lambda,fdnx,4);
            cpgerrb(2,nnn,fi_lambda,fdnx,e_fdnx,0.2);
            cpgerrb(4,nnn,fi_lambda,fdnx,e_fdnx,0.4);
            for (i=0;i<npix;i++) {
              x[i]=start_w+delta_w*i;
              y2[i]=0.0;
              for (j=0;j<ma;j++) {
                y2[i]+=a_x[j+1]*pow(x[i],j);
              }
            }
            cpgline(npix,x,y2);
            if (dar_corrected!=0) {
              for (j=0;j<n_groups;j++) {
                cpgsls(j+2);
                cpgline(dars[j].n,x,dars[j].dx);
              }
            }
            cpgsls(1);
            cpgsubp(1,1);

            cpgclos();
          }

          for (j=0;j<n_groups;j++) {
            dars[j].n=npix;
            for (i=0;i<npix;i++) {
              dars[j].dx[i]=y2[i];
              dars[j].dy[i]=y1[i];
            }
          }

          free(x);
          free(r);
          free(y1);
          free(y2);
        }
        free(covar);
        free(a_x);
        free(a_y);
        free(ia);
      }

      free(xx);
      free(dx);
      free(e_dx);
      free(dy);
      free(e_dy);
      //    free(r);
      free(i_lambda);
      free(dnx);
      free(dny);
      free(fi_lambda);
      free(fdnx);
      free(fdny);
      free(fdnr);
      free(e_fdnx);
      free(e_fdny);
      free(img_data);
      free(psf_data);
      free(map_zero);
      free(map_tmp);
      free(slice);
      sprintf(interp->result,"Done");
    } else {
      sprintf(interp->result,"Bad selection of comparison index");
    }
  } else {
    sprintf(interp->result,"DARs are not defined");
  }

  return TCL_OK;
}

/**********************************
 * SIMPLE EMPIRICAL D.A.R. Correction
 *********************************/
static int simple_dar_command(Tcl_Interp *interp, int argc, char *argv[]) {
  int j,i,k,jj;
  float *slice;
  float tmp_tr[8];
  int nx_t,ny_t;
  float *map_tmp;
  int comp_index,delta_index;
  int nx_peak,ny_peak;
  float peak;
  float dpix_tt=0.2;
  int *dnx,*dny,*i_lambda;
  float *fdnx,*fdny,*e_fdnx,*e_fdny,*fi_lambda,*fdnr;
  int nnn;
  int dnx0,dny0;
  float sig_dnx,sig_dny;
  int plot_all=1;
  double *a_x,*a_y;
  int *ia;
  double **covar;
  float chisq;
  int ma=4;
  float *x,*y1,*y2,*r;
  float min_dn,max_dn;
  double *dx,*e_dx,*dy,*e_dy,*xx;
  int n_d;
  int nulo_x=0;
  int nulo_y=0;

  if (argc < 2 ) {
    Tcl_AppendResult(interp, "USE: simple_dar COMP_INDEX DELTA_INDEX", NULL);
    return TCL_ERROR;
  }

  comp_index=atoi(argv[0]);
  delta_index=atoi(argv[1]);

  if (dars_defined==1) {
  dnx = (int *)malloc(((int)(npix/delta_index))*sizeof(int));
  dny = (int *)malloc(((int)(npix/delta_index))*sizeof(int));
  i_lambda = (int *)malloc(((int)(npix/delta_index))*sizeof(int));

  fdnx = (float *)malloc(((float)(npix/delta_index))*sizeof(float));
  fdny = (float *)malloc(((float)(npix/delta_index))*sizeof(float));
  fdnr = (float *)malloc(((float)(npix/delta_index))*sizeof(float));
  fi_lambda = (float *)malloc(((float)(npix/delta_index))*sizeof(float));

  e_fdnx = (float *)malloc(((float)(npix/delta_index))*sizeof(float));
  e_fdny = (float *)malloc(((float)(npix/delta_index))*sizeof(float));
  nnn=0;

  if (argc>2) {
    dpix_tt=atof(argv[2]);
  }
  if (argc>3) {
    plot_all=atoi(argv[3]);
  }

  if ((comp_index>-1)&&(comp_index<npix-delta_index)) {
    slice = (float *)malloc((nb_spec)*sizeof(float));
    for (j=0;j<npix-delta_index;j=j+delta_index) {
      i_lambda[nnn]=j+(int)(delta_index/2);
      for (i=0;i<nb_spec;i++) {
        for (jj=0;jj<delta_index;jj++) {
          slice[i]=raw_data[j+jj+i*npix];
        }
      }
      create_map(nb_spec,n_spaxels,spaxels,npts,slice,dpix_tt,dpix_tt,&map_tmp,x_min,x_max,y_min,y_max,tmp_tr,&nx_t,&ny_t,grid_func,grid_opt);
      /*
       * We look for the peak
       */
      nx_peak=0;
      ny_peak=0;
      peak=map_tmp[0];
      for (i=0;i<ny_t;i++) {
        for (k=0;k<nx_t;k++) {
          if (peak<map_tmp[k+i*nx_t]) {
            peak=map_tmp[k+i*nx_t];
            nx_peak=k;
            ny_peak=i;
          }

        }
      }
      dnx[nnn]=nx_peak;
      dny[nnn]=ny_peak;
      nnn++;
    }
    dnx0=dnx[(int)(((float)npix/delta_index)*0.5)];
    dny0=dny[(int)(((float)npix/delta_index)*0.5)];

    min_dn=300.0;
    max_dn=-300.0;
    for (i=0;i<nnn;i++) {
      dnx[i]=dnx[i]-dnx0;
      dny[i]=dny[i]-dny0;
      fdnx[i]=(dnx[i])*dpix_tt;
      fdny[i]=(dny[i])*dpix_tt;
      fdnr[i]=sqrt(fdny[i]*fdny[i]+fdnx[i]*fdnx[i]);
      if (min_dn>fdnx[i]) min_dn=fdnx[i];
      if (max_dn<fdnx[i]) max_dn=fdnx[i];
      if (min_dn>fdny[i]) min_dn=fdny[i];
      if (max_dn<fdny[i]) max_dn=fdny[i];
      e_fdnx[i]=dpix_tt/2;
      e_fdny[i]=dpix_tt/2;
      fi_lambda[i]=i_lambda[i]*delta_w+start_w;
    }
    min_dn-=dpix_tt;
    max_dn+=dpix_tt;

    sig_dnx=3*sigma(fdnx,nnn,0.0);
    sig_dny=3*sigma(fdny,nnn,0.0);
//    if (sig_dnx>1.5) sig_dnx=3.5;
//    if (sig_dny>1.5) sig_dny=3.5;

    /*
     * We clean the data from 1sigma
     */
    xx=vector(1,nnn+1);
    dx=vector(1,nnn+1);
    dy=vector(1,nnn+1);
    e_dx=vector(1,nnn+1);
    e_dy=vector(1,nnn+1);
    n_d=0;
    dx[0]=0.0;
    dy[0]=0.0;
    for (i=0;i<nnn;i++) {
      dx[n_d]=fdnx[i];
      dy[n_d]=fdny[i];
      xx[n_d]=fi_lambda[i];
      if (i>0) {
        if (dx[n_d]!=dx[n_d-1]) nulo_x=1;
        if (dy[n_d]!=dy[n_d-1]) nulo_y=1;
      }
      n_d++;
    }
    /*
     * We update the information of the DAR
     */
    if ((nulo_x==1)&&(nulo_y==1)) {
      covar=matrix(1,ma,1,ma);
      ia=ivector(1,ma);
      a_x=vector(1,ma);
      a_y=vector(1,ma);
      for (k=1;k<=ma;k++) {
        a_x[k]=1.0/k;
        a_y[k]=1.0/k;
        ia[k]=1;
      }

      /*
       * We take first the linear version...
       */
      nulo_x=0;
      nulo_y=0;
      for(j=0;j<2;j++) {
        gl_lfit((float *)xx,(float *)dx,e_fdnx,n_d-1,(float *)a_x,ia,2,(float **)covar,&chisq,poly_serie);
        gl_lfit((float *)xx,(float *)dy,e_fdny,n_d-1,(float *)a_y,ia,2,(float **)covar,&chisq,poly_serie);
        //    free(covar);
        /*
         * We make another clipping taking away 1 sigma
         * from the relation.
         */
        n_d=0;
        for (i=0;i<nnn;i++) {
          if ((fabs(fdnx[i]-a_x[1]-a_x[2]*fi_lambda[i])<sig_dnx)&&(fabs(fdny[i]-a_y[1]-a_y[2]*fi_lambda[i])<sig_dny)) {
            dx[n_d]=fdnx[i];
            dy[n_d]=fdny[i];
            xx[n_d]=fi_lambda[i];
            if (i>0) {
              if (dx[n_d]!=dx[n_d-1]) nulo_x=1;
              if (dy[n_d]!=dy[n_d-1]) nulo_y=1;
            }
            n_d++;
          }
        }
      }

      /*
       * Limpio!
       */
      if ((nulo_x==1)&&(nulo_y==1)) {
        gl_lfit((float *)xx,(float *)dx,e_fdnx,n_d-1,(float *)a_x,ia,ma,(float **)covar,&chisq,poly_serie);
        gl_lfit((float *)xx,(float *)dy,e_fdny,n_d-1,(float *)a_y,ia,ma,(float **)covar,&chisq,poly_serie);
        x = (float *)malloc(npix*sizeof(float));
        y1 = (float *)malloc(npix*sizeof(float));
        y2 = (float *)malloc(npix*sizeof(float));
        r = (float *)malloc(npix*sizeof(float));
        if (plot_all==1) {
          cpgopen("/xterm");
          cpgask(0);
          cpgsch(1.6);
          cpgsubp(1,2);
          cpgenv(start_w,npix*delta_w+start_w,(-1)*sig_dny,sig_dny,0,0);
          cpglab("\\gl (Angstroms)", "\\gDy (arcsec)", "D.A.R");
          cpgpt(nnn,fi_lambda,fdny,4);
          cpgerrb(2,nnn,fi_lambda,fdny,e_fdny,0.2);
          cpgerrb(4,nnn,fi_lambda,fdny,e_fdny,0.4);
          for (i=0;i<npix;i++) {
            x[i]=start_w+delta_w*i;
            y1[i]=0.0;
            y2[i]=0.0;
            for (j=0;j<ma;j++) {
              y1[i]+=a_y[j+1]*pow(x[i],j);
            }

          }
          cpgline(npix,x,y1);
          if (dar_corrected!=0) {
            for (j=0;j<n_groups;j++) {
              cpgsls(j+2);
              cpgline(dars[j].n,x,dars[j].dy);
            }
          }
          cpgsls(1);
          cpgenv(start_w,npix*delta_w+start_w,(-1)*sig_dnx,sig_dnx,0,0);
          cpglab("\\gl (Angstroms)", "\\gDx (arcsec)", "");
          cpgpt(nnn,fi_lambda,fdnx,4);
          cpgerrb(2,nnn,fi_lambda,fdnx,e_fdnx,0.2);
          cpgerrb(4,nnn,fi_lambda,fdnx,e_fdnx,0.4);
          for (i=0;i<npix;i++) {
            x[i]=start_w+delta_w*i;
            y2[i]=0.0;
            for (j=0;j<ma;j++) {
              y2[i]+=a_x[j+1]*pow(x[i],j);
            }
          }
          cpgline(npix,x,y2);
          if (dar_corrected!=0) {
            for (j=0;j<n_groups;j++) {
              cpgsls(j+2);
              cpgline(dars[j].n,x,dars[j].dx);
            }
          }
          cpgsls(1);
          cpgsubp(1,1);

          cpgclos();
        }

        /*
         * Limpio
         */
        for (j=0;j<n_groups;j++) {
          dars[j].n=npix;
          for (i=0;i<npix;i++) {
            fflush(stdout);
            dars[j].dx[i]=y2[i];
            dars[j].dy[i]=y1[i];
          }
        }
        free(x);
        free(r);
        free(y1);
        free(y2);
      }
      free(covar);
      free(a_x);
      free(a_y);
      free(ia);
    }

    free(xx);
    free(dx);
    free(e_dx);
    free(dy);
    free(e_dy);
    free(i_lambda);
    /*
     * free(dnx);
     * free(dny);
     * free(fi_lambda);
     * free(fdnx);
     * free(fdny);
     * free(fdnr);
     * free(e_fdnx);
     * free(e_fdny);
     * free(map_zero);
     * free(map_tmp);
     * free(slice);
    */
    sprintf(interp->result,"Done");
  } else {
    sprintf(interp->result,"Bad selection of comparison index");
  }

  } else {
    sprintf(interp->result,"DARS not defined");
  }

  return TCL_OK;
}

/**********************************
 * PRINCIPAL MOMENTUM DAR D.A.R. Correction
 *********************************/
static int pm_dar_command(Tcl_Interp *interp, int argc, char *argv[]) {
  int j,i,k,jj;
  float *slice;
  int comp_index,delta_index;
  float dpix_tt=0.2;
  float *fdnx,*fdny,*e_fdnx,*e_fdny,*fi_lambda,*fdnr;
  int nnn;
  float fdnx0,fdny0;
  float sig_dnx,sig_dny;
  int plot_all=0;
  double *a_x,*a_y;
  int *ia;
  double **covar;
  float chisq;
  int ma=3;
  float *x,*y1,*y2,*r;
  float min_dn,max_dn;
  double *dx,*e_dx,*dy,*e_dy,*xx;
  int n_d;
  int nulo_x=0;
  int nulo_y=0;
  float suma;
  float x_suma;
  float y_suma;
  float valor;
  float x_c,y_c;

  if (argc < 2 ) {
    Tcl_AppendResult(interp, "USE: pm_dar COMP_INDEX DELTA_INDEX", NULL);
    return TCL_ERROR;
  }

  comp_index=atoi(argv[0]);
  delta_index=atoi(argv[1]);

  if (dars_defined==1) {
    fdnx = (float *)malloc(((float)(npix/delta_index))*sizeof(float));
    fdny = (float *)malloc(((float)(npix/delta_index))*sizeof(float));
    fdnr = (float *)malloc(((float)(npix/delta_index))*sizeof(float));
    fi_lambda = (float *)malloc(((float)(npix/delta_index))*sizeof(float));

    e_fdnx = (float *)malloc(((float)(npix/delta_index))*sizeof(float));
    e_fdny = (float *)malloc(((float)(npix/delta_index))*sizeof(float));
    nnn=0;

    if (argc>2) {
      dpix_tt=atof(argv[2]);
    }
    if (argc>3) {
      plot_all=atoi(argv[3]);
    }


    if ((comp_index>-1)&&(comp_index<npix-delta_index)) {
      slice = (float *)malloc((nb_spec)*sizeof(float));
      for (j=0;j<npix-delta_index;j=j+delta_index) {
        fi_lambda[nnn]=(j+delta_index/2)*delta_w+start_w;
        x_suma=0.0;
        y_suma=0.0;
        for (jj=0;jj<delta_index;jj++) {
          suma=0.0;
          x_c=0.0;
          y_c=0.0;
          for (i=0;i<nb_spec;i++) {
            valor=raw_data[j+jj+i*npix];
            suma=suma+valor;
            for (k=0;k<n_spaxels[i];k++) {
              x_c=x_c+spaxels[i][k].xpos/n_spaxels[i]*valor;
              y_c=y_c+spaxels[i][k].ypos/n_spaxels[i]*valor;
            }
          }
          x_c=x_c/suma;
          y_c=y_c/suma;
          x_suma=x_suma+x_c;
          y_suma=y_suma+y_c;
        }

        fdnx[nnn]=x_suma/delta_index;
        fdny[nnn]=y_suma/delta_index;
        nnn++;
      }
      fdnx0=fdnx[(int)(((float)comp_index/delta_index+0.5))];
      fdny0=fdny[(int)(((float)comp_index/delta_index+0.5))];
      min_dn=300.0;
      max_dn=-300.0;
      for (i=0;i<nnn;i++) {
        fdnx[i]=fdnx[i]-fdnx0;
        fdny[i]=fdny[i]-fdny0;
        if (min_dn>fdnx[i]) min_dn=fdnx[i];
        if (max_dn<fdnx[i]) max_dn=fdnx[i];
        if (min_dn>fdny[i]) min_dn=fdny[i];
        if (max_dn<fdny[i]) max_dn=fdny[i];
        e_fdnx[i]=dpix_tt/2;
        e_fdny[i]=dpix_tt/2;
      }
      min_dn-=dpix_tt;
      max_dn+=dpix_tt;

      sig_dnx=sigma(fdnx,nnn,0.0);
      sig_dny=sigma(fdny,nnn,0.0);

      /*
       * We clean the data from 1sigma
       */
      xx=vector(1,nnn+1);
      dx=vector(1,nnn+1);
      dy=vector(1,nnn+1);
      e_dx=vector(1,nnn+1);
      e_dy=vector(1,nnn+1);
      n_d=0;
      dx[0]=0.0;
      dy[0]=0.0;
      for (i=0;i<nnn;i++) {
          dx[n_d]=fdnx[i];
          dy[n_d]=fdny[i];
          xx[n_d]=fi_lambda[i];
          if (i>0) {
            if (dx[n_d]!=dx[n_d-1]) nulo_x=1;
            if (dy[n_d]!=dy[n_d-1]) nulo_y=1;
          }
          n_d++;
      }

      /*
       * We update the information of the DAR
       */
      if ((nulo_x==1)&&(nulo_y==1)) {

        covar=matrix(1,ma,1,ma);
        ia=ivector(1,ma);
        a_x=vector(1,ma);
        a_y=vector(1,ma);
        for (k=1;k<=ma;k++) {
          a_x[k]=1.0/k;
          a_y[k]=1.0/k;
          ia[k]=1;
        }

        /*
         * We take first the linear version...
         */
        nulo_x=0;
        nulo_y=0;
        for(j=0;j<2;j++) {
          gl_lfit((float *)xx,(float *)dx,e_fdnx,n_d-1,(float *)a_x,ia,2,(float **)covar,&chisq,poly_serie);
          gl_lfit((float *)xx,(float *)dy,e_fdny,n_d-1,(float *)a_y,ia,2,(float **)covar,&chisq,poly_serie);
          //    free(covar);
          /*
           * We make another clipping taking away 1 sigma
           * from the relation.
           */
          n_d=0;
          for (i=0;i<nnn;i++) {
            if ((fabs(fdnx[i]-a_x[1]-a_x[2]*fi_lambda[i])<sig_dnx)&&(fabs(fdny[i]-a_y[1]-a_y[2]*fi_lambda[i])<sig_dny)) {
              dx[n_d]=fdnx[i];
              dy[n_d]=fdny[i];
              xx[n_d]=fi_lambda[i];
              if (i>0) {
                if (dx[n_d]!=dx[n_d-1]) nulo_x=1;
                if (dy[n_d]!=dy[n_d-1]) nulo_y=1;
              }
              n_d++;
            }
          }
        }

        if ((nulo_x==1)&&(nulo_y==1)) {
          gl_lfit((float *)xx,(float *)dx,e_fdnx,n_d-1,(float *)a_x,ia,ma,(float **)covar,&chisq,poly_serie);
          gl_lfit((float *)xx,(float *)dy,e_fdny,n_d-1,(float *)a_y,ia,ma,(float **)covar,&chisq,poly_serie);
          x = (float *)malloc(npix*sizeof(float));
          y1 = (float *)malloc(npix*sizeof(float));
          y2 = (float *)malloc(npix*sizeof(float));
          r = (float *)malloc(npix*sizeof(float));
          if (plot_all==1) {
            cpgopen("/xterm");
            cpgask(0);
            cpgsch(1.6);
            cpgsubp(1,2);
            cpgenv(start_w,npix*delta_w+start_w,(-1)*sig_dny,sig_dny,0,0);
            cpglab("\\gl (Angstroms)", "\\gDy (arcsec)", "D.A.R");
            cpgpt(nnn,fi_lambda,fdny,4);
            cpgerrb(2,nnn,fi_lambda,fdny,e_fdny,0.2);
            cpgerrb(4,nnn,fi_lambda,fdny,e_fdny,0.4);
            for (i=0;i<npix;i++) {
              x[i]=start_w+delta_w*i;
              y1[i]=0.0;
              y2[i]=0.0;
              for (j=0;j<ma;j++) {
                y1[i]+=a_y[j+1]*pow(x[i],j);
              }

            }
            cpgline(npix,x,y1);
            if (dar_corrected!=0) {
              for (j=0;j<n_groups;j++) {
                cpgsls(j+2);
                cpgline(dars[j].n,x,dars[j].dy);
              }
            }
            cpgsls(1);
            cpgenv(start_w,npix*delta_w+start_w,(-1)*sig_dnx,sig_dnx,0,0);
            cpglab("\\gl (Angstroms)", "\\gDx (arcsec)", "");
            cpgpt(nnn,fi_lambda,fdnx,4);
            cpgerrb(2,nnn,fi_lambda,fdnx,e_fdnx,0.2);
            cpgerrb(4,nnn,fi_lambda,fdnx,e_fdnx,0.4);
            for (i=0;i<npix;i++) {
              x[i]=start_w+delta_w*i;
              y2[i]=0.0;
              for (j=0;j<ma;j++) {
                y2[i]+=a_x[j+1]*pow(x[i],j);
              }
            }
            cpgline(npix,x,y2);
            if (dar_corrected!=0) {
              for (j=0;j<n_groups;j++) {
                cpgsls(j+2);
                cpgline(dars[j].n,x,dars[j].dx);
              }
            }
            cpgsls(1);
            cpgsubp(1,1);

            cpgclos();
          }





          for (j=0;j<n_groups;j++) {
            dars[j].n=npix;
            for (i=0;i<npix;i++) {
              dars[j].dx[i]=y2[i];
              dars[j].dy[i]=y1[i];
              //            printf ("%d %f %f",i,dars[j].dx[i],dars[j].dy[i]);
            }
          }
          free(x);
          free(r);
          free(y1);
          free(y2);
        }
        free(covar);
        free(a_x);
        free(a_y);
        free(ia);
      }
      /*
       * free(xx);
       * free(dx);
       * free(e_dx);
       * free(dy);
       * free(e_dy);
       * free(r);
       * free(dnx);
       * free(dny);
       * free(fi_lambda);
       * free(fdnx);
       * free(fdny);
       * free(fdnr);
       * free(e_fdnx);
       * free(e_fdny);
       * free(map_zero);
       * free(map_tmp);
       * free(slice);
       */
      sprintf(interp->result,"Done");
    } else {
      sprintf(interp->result,"Bad selection of comparison index");
    }
  } else {
    sprintf(interp->result,"DARS not defined");
  }

  return TCL_OK;
}

/**********************************
 * PEAK DAR D.A.R. Correction
 *********************************/
static int peak_dar_command(Tcl_Interp *interp, int argc, char *argv[]) {
  int j,i,k,jj;
  float *slice;
  int comp_index,delta_index;
  float dpix_tt=0.2;
  float *fdnx,*fdny,*e_fdnx,*e_fdny,*fi_lambda,*fdnr;
  int nnn;
  float fdnx0,fdny0;
  float sig_dnx,sig_dny;
  int plot_all=0;
  double *a_x,*a_y;
  int *ia;
  double **covar;
  float chisq;
  int ma=3;
  float *x,*y1,*y2,*r;
  float min_dn,max_dn;
  double *dx,*e_dx,*dy,*e_dy,*xx;
  int n_d;
  int nulo_x=0;
  int nulo_y=0;
  float suma;
  float x_suma;
  float y_suma;
  float valor;
  float x_c,y_c;
  float max_valor;

  if (argc < 2 ) {
    Tcl_AppendResult(interp, "USE: peak_dar COMP_INDEX DELTA_INDEx", NULL);
    return TCL_ERROR;
  }
  comp_index=atoi(argv[0]);
  delta_index=atoi(argv[1]);
  if (dars_defined==1) {
    fdnx = (float *)malloc(((float)(npix/delta_index))*sizeof(float));
    fdny = (float *)malloc(((float)(npix/delta_index))*sizeof(float));
    fdnr = (float *)malloc(((float)(npix/delta_index))*sizeof(float));
    fi_lambda = (float *)malloc(((float)(npix/delta_index))*sizeof(float));

    e_fdnx = (float *)malloc(((float)(npix/delta_index))*sizeof(float));
    e_fdny = (float *)malloc(((float)(npix/delta_index))*sizeof(float));
    nnn=0;

    if (argc>2) {
      dpix_tt=atof(argv[2]);
    }
    if (argc>3) {
      plot_all=atoi(argv[3]);
    }

    if ((comp_index>-1)&&(comp_index<npix-delta_index)) {
      slice = (float *)malloc((nb_spec)*sizeof(float));
      for (j=0;j<npix-delta_index;j=j+delta_index) {
        fi_lambda[nnn]=(j+delta_index/2)*delta_w+start_w;
        suma=0.0;
        x_suma=0.0;
        y_suma=0.0;
        for (jj=0;jj<delta_index;jj++) {
          max_valor=-2000;
          for (i=0;i<nb_spec;i++) {
            valor=raw_data[j+jj+i*npix];
            if (max_valor<valor) {
              max_valor=valor;
              x_c=0.0;
              y_c=0.0;
              for (k=0;k<n_spaxels[i];k++) {
                x_c=x_c+spaxels[i][k].xpos/n_spaxels[i];
                y_c=y_c+spaxels[i][k].ypos/n_spaxels[i];
              }

            }
          }
          x_suma=x_suma+x_c;
          y_suma=y_suma+y_c;
        }
        fdnx[nnn]=x_suma/delta_index;
        fdny[nnn]=y_suma/delta_index;
        nnn++;
      }
      fdnx0=fdnx[(int)(((float)comp_index/delta_index+0.5))];
      fdny0=fdny[(int)(((float)comp_index/delta_index+0.5))];
      min_dn=300.0;
      max_dn=-300.0;
      for (i=0;i<nnn;i++) {
        fdnx[i]=fdnx[i]-fdnx0;
        fdny[i]=fdny[i]-fdny0;
        if (min_dn>fdnx[i]) min_dn=fdnx[i];
        if (max_dn<fdnx[i]) max_dn=fdnx[i];
        if (min_dn>fdny[i]) min_dn=fdny[i];
        if (max_dn<fdny[i]) max_dn=fdny[i];
        e_fdnx[i]=dpix_tt/2;
        e_fdny[i]=dpix_tt/2;
      }
      min_dn-=dpix_tt;
      max_dn+=dpix_tt;

      sig_dnx=sigma(fdnx,nnn,0.0);
      sig_dny=sigma(fdny,nnn,0.0);

      /*
       * We clean the data from 1sigma
       */
      xx=vector(1,nnn+1);
      dx=vector(1,nnn+1);
      dy=vector(1,nnn+1);
      e_dx=vector(1,nnn+1);
      e_dy=vector(1,nnn+1);
      n_d=0;
      dx[0]=0.0;
      dy[0]=0.0;
      for (i=0;i<nnn;i++) {
          dx[n_d]=fdnx[i];
          dy[n_d]=fdny[i];
          xx[n_d]=fi_lambda[i];
          if (i>0) {
            if (dx[n_d]!=dx[n_d-1]) nulo_x=1;
            if (dy[n_d]!=dy[n_d-1]) nulo_y=1;
          }
          n_d++;
      }

      /*
       * We update the information of the DAR
       */
      if ((nulo_x==1)&&(nulo_y==1)) {

        covar=matrix(1,ma,1,ma);
        ia=ivector(1,ma);
        a_x=vector(1,ma);
        a_y=vector(1,ma);
        for (k=1;k<=ma;k++) {
          a_x[k]=1.0/k;
          a_y[k]=1.0/k;
          ia[k]=1;
        }

        /*
         * We take first the linear version...
         */
        nulo_x=0;
        nulo_y=0;
        for(j=0;j<2;j++) {
          gl_lfit((float *)xx,(float *)dx,e_fdnx,n_d-1,(float *)a_x,ia,2,(float **)covar,&chisq,poly_serie);
          gl_lfit((float *)xx,(float *)dy,e_fdny,n_d-1,(float *)a_y,ia,2,(float **)covar,&chisq,poly_serie);
          //    free(covar);
          /*
           * We make another clipping taking away 1 sigma
           * from the relation.
           */
          n_d=0;
          for (i=0;i<nnn;i++) {
            if ((fabs(fdnx[i]-a_x[1]-a_x[2]*fi_lambda[i])<sig_dnx)&&(fabs(fdny[i]-a_y[1]-a_y[2]*fi_lambda[i])<sig_dny)) {
              dx[n_d]=fdnx[i];
              dy[n_d]=fdny[i];
              xx[n_d]=fi_lambda[i];
              if (i>0) {
                if (dx[n_d]!=dx[n_d-1]) nulo_x=1;
                if (dy[n_d]!=dy[n_d-1]) nulo_y=1;
              }
              n_d++;
            }
          }
        }

        if ((nulo_x==1)&&(nulo_y==1)) {
          gl_lfit((float *)xx,(float *)dx,e_fdnx,n_d-1,(float *)a_x,ia,ma,(float **)covar,&chisq,poly_serie);
          gl_lfit((float *)xx,(float *)dy,e_fdny,n_d-1,(float *)a_y,ia,ma,(float **)covar,&chisq,poly_serie);
          x = (float *)malloc(npix*sizeof(float));
          y1 = (float *)malloc(npix*sizeof(float));
          y2 = (float *)malloc(npix*sizeof(float));
          r = (float *)malloc(npix*sizeof(float));
          if (plot_all==1) {
            cpgopen("/xterm");
            cpgask(0);
            cpgsch(1.6);
            cpgsubp(1,2);
            cpgenv(start_w,npix*delta_w+start_w,(-1)*sig_dny,sig_dny,0,0);
            cpglab("\\gl (Angstroms)", "\\gDy (arcsec)", "D.A.R");
            cpgpt(nnn,fi_lambda,fdny,4);
            cpgerrb(2,nnn,fi_lambda,fdny,e_fdny,0.2);
            cpgerrb(4,nnn,fi_lambda,fdny,e_fdny,0.4);
            for (i=0;i<npix;i++) {
              x[i]=start_w+delta_w*i;
              y1[i]=0.0;
              y2[i]=0.0;
              for (j=0;j<ma;j++) {
                y1[i]+=a_y[j+1]*pow(x[i],j);
              }

            }
            cpgline(npix,x,y1);
            if (dar_corrected!=0) {
              for (j=0;j<n_groups;j++) {
                cpgsls(j+2);
                cpgline(dars[j].n,x,dars[j].dy);
              }
            }
            cpgsls(1);
            cpgenv(start_w,npix*delta_w+start_w,(-1)*sig_dnx,sig_dnx,0,0);
            cpglab("\\gl (Angstroms)", "\\gDx (arcsec)", "");
            cpgpt(nnn,fi_lambda,fdnx,4);
            cpgerrb(2,nnn,fi_lambda,fdnx,e_fdnx,0.2);
            cpgerrb(4,nnn,fi_lambda,fdnx,e_fdnx,0.4);
            for (i=0;i<npix;i++) {
              x[i]=start_w+delta_w*i;
              y2[i]=0.0;
              for (j=0;j<ma;j++) {
                y2[i]+=a_x[j+1]*pow(x[i],j);
              }
            }
            cpgline(npix,x,y2);
            if (dar_corrected!=0) {
              for (j=0;j<n_groups;j++) {
                cpgsls(j+2);
                cpgline(dars[j].n,x,dars[j].dx);
              }
            }
            cpgsls(1);
            cpgsubp(1,1);

            cpgclos();
          }





          for (j=0;j<n_groups;j++) {
            dars[j].n=npix;
            for (i=0;i<npix;i++) {
              dars[j].dx[i]=y2[i];
              dars[j].dy[i]=y1[i];
              //printf ("%d %f %f",i,dars[j].dx[i],dars[j].dy[i]);
            }
          }
          free(x);
          free(r);
          free(y1);
          free(y2);
        }
        free(covar);
        free(a_x);
        free(a_y);
        free(ia);
      }
      /*
       * free(xx);
       * free(dx);
       * free(e_dx);
       * free(dy);
       * free(e_dy);
       * free(r);
       * free(dnx);
       * free(dny);
       * free(fi_lambda);
       * free(fdnx);
       * free(fdny);
       * free(fdnr);
       * free(e_fdnx);
       * free(e_fdny);
       * free(map_zero);
       * free(map_tmp);
       * free(slice);
       */
      sprintf(interp->result,"Done");
    } else {
      sprintf(interp->result,"Bad selection of comparison index");
    }
  } else {
    sprintf(interp->result,"DARS not defined");
  }

  return TCL_OK;
}

/**********************************
 * D.A.R. Correction
 *********************************/
static int correct_dar_command(Tcl_Interp *interp, int argc, char *argv[]) {
  int j,i;
  float *slice;

  if (dars_defined==1) {
    if (dar_corrected!=1) {
      slice = (float *)malloc((nb_spec)*sizeof(float));
      for (j=0;j<npix;j++) {

        for (i=0;i<nb_spec;i++) {
          slice[i]=raw_data[j+i*npix];
        }
        correct_mono_dar(nb_spec,n_spaxels,spaxels,slice,dars,j);
        for (i=0;i<nb_spec;i++) {
          raw_data[j+i*npix]=slice[i];
        }
      }
      dar_corrected=1;
      sprintf(interp->result,"Ok");
    } else {
      sprintf(interp->result,"Already Done");
    }
  } else {
    sprintf(interp->result,"DARS not defined");
  }

  return TCL_OK;
}

/**********************************
 * D.A.R. Theoretical Determination
 *********************************/
static int theoretical_dar_command(Tcl_Interp *interp, int argc, char *argv[]) {
  int j,i;
  int test=0;

  if (argc<1) {
    test=0;
  } else {
    test=atoi(argv[0]);
  }
  if (dars_valid==1) {
    for (j=0;j<n_groups;j++) {
      determine_dar(dars[j],groups[j],(float)start_w,(float)delta_w,npix);
      if (test==1) {
        for (i=0;i<npix;i++) {
          printf("%d %f %f ",i,dars[j].dx[i],dars[j].dy[i]);
        }
      }
    }
    dar_corrected=-1;
    dars_defined=1;
  } else {
    print_msg("The group information is not correct\nNo DAR determined\n");
  }
  return TCL_OK;
}

/**********************************
 * SAVE SLICE ROUTINE
 *********************************/
static int save_slice_command(Tcl_Interp *interp, int argc, char *argv[]) {
  char text[80],units[80];
  int n_m=0;
  float *tmp_x,*tmp_y,*tmp_slice; // Array of Selection of Slices
  float *tmp_specId;
  int n_tot=0;
  int i,j,k;
  int outputtype=0;

  if (argc < 4 ) {
    Tcl_AppendResult(interp, "USE: save_slice n_slice filename units outputtype[0=fits,1=ascii]", NULL);
    return TCL_ERROR;
  }
  if ((strlen(argv[1])==0)||(strcmp(argv[1],".fits")==0)||(strcmp(argv[1],".fit")==0)||(strcmp(argv[1],".e3d")==0)) {
      Tcl_AppendResult(interp, "No output file defined", NULL);
      return TCL_ERROR;
  }
  n_m=atoi(argv[0]);
  strcpy(text,argv[1]);
  strcpy(units,argv[2]);
  outputtype=atoi(argv[3]);

  for (i=0;i<nb_spec;i++) {
    for (j=0;j<n_spaxels[i];j++) {
      n_tot++;
    }
  }
  tmp_slice = (float *)malloc((n_tot)*sizeof(float));
  tmp_x = (float *)malloc((n_tot)*sizeof(float));
  tmp_y = (float *)malloc((n_tot)*sizeof(float));
  tmp_specId = (float *)malloc((n_tot)*sizeof(float));

  k=0;
  for (i=0;i<nb_spec;i++) {
    for (j=0;j<n_spaxels[i];j++) {
      tmp_specId[k]=(float)spaxels[i][j].specId;
      tmp_x[k]=spaxels[i][j].xpos;
      tmp_y[k]=spaxels[i][j].ypos;
      if (f_slice_def[n_m]==1) {
        tmp_slice[k]=f_slice[n_m][i];
      } else {
        tmp_slice[k]=0;
      }
      k++;
    }
  }
  if (outputtype==0) {
    save_slice(tmp_specId,tmp_x,tmp_y,tmp_slice,n_tot,text,units);
  } else {
    save_slice_ascii(tmp_specId,tmp_x,tmp_y,tmp_slice,n_tot,text,units);
  }
  sprintf(interp->result,"1");

  free(tmp_slice);
  free(tmp_x);
  free(tmp_y);
  free(tmp_specId);

  return TCL_OK;
}

/**********************************
 * SAVE Position Table ROUTINE
 *********************************/
static int save_PT_command(Tcl_Interp *interp, int argc, char *argv[]) {
  char text[80];
  float *tmp_x,*tmp_y;
  int *tmp_group;
  int n_tot=0;
  int i,j,k;

  if (argc < 1 ) {
    Tcl_AppendResult(interp, "USE: save_PT filename", NULL);
    return TCL_ERROR;
  }
  if ((strlen(argv[0])==0)||(strcmp(argv[0],".fits")==0)||(strcmp(argv[0],".fit")==0)||(strcmp(argv[0],".e3d")==0)) {
      Tcl_AppendResult(interp, "No output file defined", NULL);
      return TCL_ERROR;
  }
  strcpy(text,argv[0]);

  for (i=0;i<nb_spec;i++) {
    for (j=0;j<n_spaxels[i];j++) {
      n_tot++;
    }
  }
  tmp_x = (float *)malloc((n_tot)*sizeof(float));
  tmp_y = (float *)malloc((n_tot)*sizeof(float));
  tmp_group = (int *)malloc((n_tot)*sizeof(int));

  k=0;
  for (i=0;i<nb_spec;i++) {
    for (j=0;j<n_spaxels[i];j++) {
      tmp_x[k]=spaxels[i][j].xpos;
      tmp_y[k]=spaxels[i][j].ypos;
      tmp_group[k]=spaxels[i][j].group;
      k++;
    }
  }
  save_PT_ascii(tmp_x,tmp_y,tmp_group,n_tot,groups,n_groups,text);
  sprintf(interp->result,"1");
  free(tmp_x);
  free(tmp_y);
  free(tmp_group);
  return TCL_OK;
}

/**********************************
 * EXPORT ROW STACKED SPECTRA ROUTINE
 *********************************/
static int export_rss_command(Tcl_Interp *interp, int argc, char *argv[]) {
  int j,i,k;
  char text[80],text2[80];
  float *tmp_x,*tmp_y;
  int *tmp_group;
  int n_tot=0;

  if (argc < 2) {
    Tcl_AppendResult(interp, "USE: export_rss FITSFILE POSITION_TABLE", NULL);
    return TCL_ERROR;
  }
  strcpy(text,argv[0]);
  strcpy(text2,argv[1]);
  save_raw(raw_data,npix,nb_spec,text,start_w,delta_w);

  for (i=0;i<nb_spec;i++) {
    for (j=0;j<n_spaxels[i];j++) {
      n_tot++;
    }
  }
  tmp_x = (float *)malloc((n_tot)*sizeof(float));
  tmp_y = (float *)malloc((n_tot)*sizeof(float));
  tmp_group = (int *)malloc((n_tot)*sizeof(int));

  k=0;
  for (i=0;i<nb_spec;i++) {
    for (j=0;j<n_spaxels[i];j++) {
      tmp_x[k]=spaxels[i][j].xpos;
      tmp_y[k]=spaxels[i][j].ypos;
      tmp_group[k]=spaxels[i][j].group;
      k++;
    }
  }
  save_PT_ascii(tmp_x,tmp_y,tmp_group,n_tot,groups,n_groups,text2);
  sprintf(interp->result,"1");
  free(tmp_x);
  free(tmp_y);
  free(tmp_group);

  return TCL_OK;
}

/**********************************
 * SAVE MAP ROUTINE
 *********************************/
static int save_map_command(Tcl_Interp *interp, int argc, char *argv[]) {
  char text[80];
  int n_m=0;
  int i_out=1;
  if (argc < 2 ) {
    Tcl_AppendResult(interp, "USE: save_map n_map filename", NULL);
    return TCL_ERROR;
  }
  if ((strlen(argv[1])==0)||(strcmp(argv[1],".fits")==0)||(strcmp(argv[1],".fit")==0)||(strcmp(argv[1],".e3d")==0)) {
      Tcl_AppendResult(interp, "No output file defined", NULL);
      return TCL_ERROR;
  }
  n_m=atoi(argv[0]);
  strcpy(text,argv[1]);

  if (map_created[n_m]!=1) {
#ifdef DEBUG_dpix
    printf("save_map_command: dpix=%f/%f", dpix1,dpix2);
#endif
    i_out=func_create_map(last_i0,last_i1,dpix1,dpix2,0,n_m);
#ifdef DEBUG_dpix
    printf(" dpix=%f/%f\n", dpix1,dpix2);
#endif
  }
  if (i_out==-1) {
    sprintf(interp->result,"0");
  } else {
    save_map(map_data[n_m],nx_map[n_m],ny_map[n_m],text);
    sprintf(interp->result,"1");
  }
  return TCL_OK;
}

/**********************************
 * EXPORT CUBE ROUTINE
 *********************************/
static int export_cube_command(Tcl_Interp *interp, int argc, char *argv[])
{
  char text[80];

  if (argc < 1) {
    Tcl_AppendResult(interp, "USE: export_cube filename", NULL);
    return TCL_ERROR;
  }
  if ((strlen(argv[0]) == 0) ||
      (strcmp(argv[0],".fits") == 0) ||
      (strcmp(argv[0],".fit") == 0) ||
      (strcmp(argv[0],".e3d")==0)) {
    Tcl_AppendResult(interp, "No output file defined", NULL);
    return TCL_ERROR;
  }
  strcpy(text,argv[0]);
  printf("Exporting Cube to %s\n", text);
#ifdef DEBUG_e3d_save
  printf("%d %d %d SPAXELS RAW_DATA %f %s %lf %lf %lf %lf GRID_FUNC GRID_OPT %f %f\n",
         nb_spec,npix,*n_spaxels,
         dpix,text,
         x_min,x_max,y_min,y_max,
         start_w,delta_w);
#endif
#ifdef DEBUG_dpix
  printf("export_cube_command: dpix=%f/%f", dpix1,dpix2);
#endif
  save_cube(nb_spec,npix,n_spaxels,spaxels,raw_data,dpix1,dpix2,text,
            x_min,x_max,y_min,y_max,grid_func,grid_opt,start_w,delta_w);
#ifdef DEBUG_dpix
  printf(" dpix=%f/%f\n", dpix1,dpix2);
#endif

  return TCL_OK;
}

/**********************************
 * SAVE A CUT OF THE RAW DATA ROUTINE, ie, the selected portion
 *********************************/
static int save_cut_command(Tcl_Interp *interp, int argc, char *argv[])
{
  int i,k;
  char text[80];
  float *cut_data;

  if (argc < 1) {
    Tcl_AppendResult(interp, "USE: save_cut fitsfile", NULL);
    return TCL_ERROR;
  }
  if ((strlen(argv[0])==0)||(strcmp(argv[0],".fits")==0)||(strcmp(argv[0],".fit")==0)||(strcmp(argv[0],".e3d")==0)) {
      Tcl_AppendResult(interp, "No output file defined", NULL);
      return TCL_ERROR;
  }
  strcpy(text,argv[0]);
  cut_data = (float *)malloc(((npix_w2-npix_w1)*nb_spec)*sizeof(float));
  for (k=0;k<nb_spec;k++) {
      for (i=npix_w1;i<npix_w2;i++) {
          cut_data[i-npix_w1+k*(npix_w2-npix_w1)]=raw_data[i+k*npix];
      }
  }
  save_raw(cut_data,npix_w2-npix_w1,nb_spec,text,start_w+npix_w1*delta_w,delta_w);
  free(cut_data);
  return TCL_OK;
}

/**********************************
 * SAVE RAW DATA ROUTINE
 *********************************/
static int save_raw_command(Tcl_Interp *interp, int argc, char *argv[]) {
  char text[80];

  if (argc < 1) {
    Tcl_AppendResult(interp, "USE: save_raw SPECTRA_FILE", NULL);
    return TCL_ERROR;
  }
  if ((strlen(argv[0])==0)||(strcmp(argv[0],".fits")==0)||(strcmp(argv[0],".fit")==0)||(strcmp(argv[0],".e3d")==0)) {
      Tcl_AppendResult(interp, "No output file defined", NULL);
      return TCL_ERROR;
  }
  strcpy(text,argv[0]);
  save_raw(raw_data,npix,nb_spec,text,start_w,delta_w);
  return TCL_OK;
}

/**********************************
 * SAVE SPEC ROUTINE
 *********************************/
static int save_spec_command(Tcl_Interp *interp, int argc, char *argv[]) {
  int nargc;
  int j,i,k;
  float *spectra;
  float *raw_spectra;
  float *slice;
  char **tokens;
  char text[80];

  if (argc < 2) {
    Tcl_AppendResult(interp, "USE: save_spec MAPFILE i_spax", NULL);
    return TCL_ERROR;
  }
  if ((strlen(argv[0])==0)||(strcmp(argv[0],".fits")==0)||(strcmp(argv[0],".fit")==0)||(strcmp(argv[0],".e3d")==0)) {
    Tcl_AppendResult(interp, "No output file defined", NULL);
    return TCL_ERROR;
  }

  strcpy(text,argv[0]);
  split_text(argv[1]," ",&nargc,&tokens);
  if (nargc>0) {
    n_i_spax=nargc;
    n_sel_spec=nargc;

    for (j=0;j<nargc;j++) {
      i_spax[j]=atoi(tokens[j]);
    }
    /*
     * We determine the values to plot
     */
    spectra = (float *)malloc((npix)*sizeof(float));
    raw_spectra = (float *)malloc((npix*nargc)*sizeof(float));
    slice = (float *)malloc((nargc)*sizeof(float));
    for (i=0;i<npix;i++) {
      spectra[i]=0.0;
    }
    for (i=0;i<npix;i++) {
      for (k=0;k<nargc;k++) {
        spectra[i]+=raw_data[i+i_spax[k]*npix];
        raw_spectra[i+k*npix]=raw_data[i+i_spax[k]*npix];
        slice[k]=raw_data[i+i_spax[k]*npix];
      }
    }
    for (i=0;i<npix;i++) {
      spectra[i]/=nargc;
    }
    save_raw(raw_spectra,npix,nargc,text,start_w,delta_w);
    sprintf(interp->result,"1");
  } else {
    sprintf(interp->result,"0");
  }
  return TCL_OK;
}

/**********************************
 * SAVE SPEC TABLE ROUTINE
 *********************************/
static int save_spec_table_command(Tcl_Interp *interp, int argc, char *argv[])
{
  int nargc;
  int j,i,k;
  float *spectra;
  float *raw_spectra;
  float *slice;
  float *w;
  char **tokens;
  char text[80],units[80];
  int flag;

  if (argc < 4) {
    Tcl_AppendResult(interp, "USE: save_spec_table MAPFILE units flag i_spax", NULL);
    return TCL_ERROR;
  }
  if ((strlen(argv[0])==0)||(strcmp(argv[0],".fits")==0)||(strcmp(argv[0],".fit")==0)||(strcmp(argv[0],".e3d")==0)) {
      Tcl_AppendResult(interp, "No output file defined", NULL);
      return TCL_ERROR;
  }

  strcpy(text,argv[0]);
  strcpy(units,argv[1]);
  flag=atoi(argv[2]);
  split_text(argv[3]," ",&nargc,&tokens);

  if (nargc>0) {
    n_i_spax=nargc;
    n_sel_spec=nargc;

    for (j=0;j<nargc;j++) {
      i_spax[j]=atoi(tokens[j]);
    }
    /*
     * We determine the values to plot
     */
    spectra = (float *)malloc((npix)*sizeof(float));
    w = (float *)malloc((npix)*sizeof(float));
    raw_spectra = (float *)malloc((npix*nargc)*sizeof(float));
    slice = (float *)malloc((nargc)*sizeof(float));
    for (i=0;i<npix;i++) {
      spectra[i]=0.0;
    }
    for (i=0;i<npix;i++) {
      w[i]=(float)start_w+((float)delta_w)*i;
      for (k=0;k<nargc;k++) {
        spectra[i]+=raw_data[i+i_spax[k]*npix];
        raw_spectra[i+k*npix]=raw_data[i+i_spax[k]*npix];
        slice[k]=raw_data[i+i_spax[k]*npix];
      }
    }
    for (i=0;i<npix;i++) {
      spectra[i]/=nargc;
    }
    if (flag==0) {
      save_spec_table(w,spectra,npix,text,units);
    } else {
      save_spec_ascii(w,spectra,npix,text,units);
    }
    //    save_map(spectra,npix,nargc,text);

    sprintf(interp->result,"1");
  } else {
    sprintf(interp->result,"0");
  }
  return TCL_OK;
}

/**********************************
 * PLOT MAP ROUTINE
 *********************************/
static int plot_map_command(Tcl_Interp *interp, int argc, char *argv[])
{
  char *p_names[]={"grey","rainbow","heat","iraf","aips"};
  float min;
  float max;
  int id,j;
  int p;
  float bright,contra,sign;
  int n_m=0;
  float out_min,out_max;

  if (argc < 7 || *argv[1] == '-') {
    Tcl_AppendResult(interp, "USE: plot_map min max device palette bright contra sign n_map", NULL);
    return TCL_ERROR;
  }
  min=atof(argv[0]);
  max=atof(argv[1]);
  id=atoi(argv[2]);
  bright=atof(argv[4]);
  contra=atof(argv[5]);
  sign=atof(argv[6]);
  for (j=0;j<5;j++) {
    if (strstr(p_names[j],argv[3])) p=j+1;
  }

  /*
   * Now we determine the number of spectra
   * that are passed as arguments
   */
  n_m=atoi(argv[7]);
  cpgslct(id);
  palette(p,sign*0.8,0.6);
  tr_palett(nx_map[n_m]*ny_map[n_m],map_data[n_m],&tmp_data,sign,contra,bright,min,max,&out_min,&out_max);
  cpgask(0);
  cpgpage();
  image_plot(tmp_data,out_min,out_max,nx_map[n_m],ny_map[n_m],id,tr);
  free(tmp_data);
  return TCL_OK;
}

/*********************************
 * Subroutine to create a certain MAP
 *********************************/
static int create_map_command(Tcl_Interp *interp, int argc, char *argv[])
{
  int i0,i1;
  int square=0;

  if (argc < 2 || *argv[1] == '-') {
    Tcl_AppendResult(interp, "USE: create_map i1 i2 [dpix] [square=0/1]", NULL);
    return TCL_ERROR;
  }
  i0=floor(atof(argv[0]));
  i1=floor(atof(argv[1]));
  if (argc>2) {
      dpix1=atof(argv[2]);
      dpix2=atof(argv[2]);
      if (argc==4) {
          square=atoi(argv[3]);
      }
  }
  fflush(stdout);
  if (n_map>MAX_NUM_MAPS) {
    n_map=0;
  }
#ifdef DEBUG_dpix
  printf("create_map_command: dpix=%f/%f", dpix1,dpix2);
#endif
  func_create_map(i0,i1,dpix1,dpix2,square,n_map);
#ifdef DEBUG_dpix
  printf(" dpix=%f/%f\n", dpix1,dpix2);
#endif
  sprintf(interp->result,"%d",n_map);
  n_map++;
  return TCL_OK;
}


/*****************************************
 * Procedure to substract a certain spectra
 *****************************************/
static int div_spec_command(Tcl_Interp *interp, int argc, char *argv[])
{
  int n1,n2;
  int j;

  if (argc < 2) {
    Tcl_AppendResult(interp, "USE: div_spec n_spec1 n_spec2", NULL);
    return TCL_ERROR;
  }
  n1=atoi(argv[0]);
  n2=atoi(argv[1]);
  for (j=0;j<npix;j++) {
    if (f_spec[n1][j]!=0.0) {
      f_spec[n_f_spec][j]=f_spec[n2][j]/f_spec[n1][j];
    } else {
      f_spec[n_f_spec][j]=0.0;
    }
  }
  sprintf(interp->result,"%d",n_f_spec);
  n_f_spec++;
  if (n_f_spec>=MAX_NUM_SPEC) {
    n_f_spec=0;
  }
  return TCL_OK;
}

/*****************************************
 * Procedure to substract a certain spectra
 *****************************************/
static int sub_spec_command(Tcl_Interp *interp, int argc, char *argv[])
{
  int n1,n2;
  int j;

  if (argc < 2) {
    Tcl_AppendResult(interp, "USE: sub_spec n_spec1 n_spec2", NULL);
    return TCL_ERROR;
  }
  n1=atoi(argv[0]);
  n2=atoi(argv[1]);
  f_spec[n_f_spec] = (float *)malloc(npix*sizeof(float));
  for (j=0;j<npix;j++) {
    f_spec[n_f_spec][j]=f_spec[n2][j]-f_spec[n1][j];
  }
  sprintf(interp->result,"%d",n_f_spec);
  n_f_spec++;
  if (n_f_spec>=MAX_NUM_SPEC) {
    n_f_spec=0;
  }
  return TCL_OK;
}

/*****************************************
 * Procedure to substract a the SKY
 *****************************************/
static int skysub_command(Tcl_Interp *interp, int argc, char *argv[])
{
  int n1;
  int i,j;

  if (argc < 1) {
    Tcl_AppendResult(interp, "USE: skysub n_spec", NULL);
    return TCL_ERROR;
  }
  n1=atoi(argv[0]);
  for (i=0;i<nb_spec;i++) {
    for (j=0;j<npix;j++) {
      raw_data[j+i*npix]=raw_data[j+i*npix]-f_spec[n1][j];
    }
  }
  return TCL_OK;
}

/*****************************************
 * Procedure for Arithmetical operations with the selected SLICEs
 * Operations (+) (-) (/) (*)
 *****************************************/
static int slicearith_command(Tcl_Interp *interp, int argc, char *argv[])
{
  int n1,n2,n3;
  int j;
  char command[1];

  if (argc < 3) {
    Tcl_AppendResult(interp, "USE: slicearith n_spec1 command n_spec2 n3", NULL);
    return TCL_ERROR;
  }
  n1=atoi(argv[2]);
  strcpy(command,argv[1]);
  n2=atoi(argv[0]);
  n3=atoi(argv[3]);
  if (n3>MAX_NUM_MAPS) n3=0;

  f_slice[n3] = (float *)malloc((nb_spec)*sizeof(float));
  f_slice_def[n3]=1;
  // Substract
  if ((f_slice_def[n2]!=1)||(f_slice_def[n1]!=1)) {
    for (j=0;j<nb_spec;j++) {
      f_slice[n3][j]=0.0;
    }
  } else {
    if (strcmp(command, "-") == 0) {
      for (j=0;j<nb_spec;j++) {
        f_slice[n3][j]=f_slice[n2][j]-f_slice[n1][j];
      }
    }
    // Add
    if (strcmp(command, "+") == 0) {
      for (j=0;j<nb_spec;j++) {
        f_slice[n3][j]=f_slice[n2][j]+f_slice[n1][j];
      }
    }
    // Multiply
    if (strcmp(command, "*") == 0) {
      for (j=0;j<nb_spec;j++) {
        f_slice[n3][j]=f_slice[n2][j]*f_slice[n1][j];
      }
    }
    // Divide
    if (strcmp(command, "/") == 0) {
      for (j=0;j<nb_spec;j++) {
        if (f_slice[n1][j]!=0.0) {
          f_slice[n3][j]=f_slice[n2][j]/f_slice[n1][j];
        } else {
          f_slice[n3][j]=0.0001;
        }
      }
    }
  }

  n_i_spax=-1;
  sprintf(interp->result,"%d",n3);
  return TCL_OK;
}

/*****************************************
 * Procedure for Arithmetical operations with the selected spectra
 * Operations (+) (-) (/) (*)
 *****************************************/
static int specarith_command(Tcl_Interp *interp, int argc, char *argv[])
{
  int n1,n2;
  int j;
  char command[1];
  float operator;
  int op_type=0;

  if (argc < 3) {
    Tcl_AppendResult(interp, "USE: specarith n_spec1 command n_spec2 [n_spec_out]", NULL);
    return TCL_ERROR;
  }

  n2=atoi(argv[0]);
  strcpy(command,argv[1]);
  n1=atoi(argv[2]);
  operator=atoi(argv[2]);
  if (argc==4) {
    n_f_spec=atoi(argv[3]);
    if (n_f_spec>=MAX_NUM_SPEC) {
      n_f_spec=0;
    }
    if (n_f_spec<0) {
      n_f_spec=9;
    }
  }

  if (operator!=n2) op_type=1;
  f_spec[n_f_spec] = (float *)malloc(npix*sizeof(float));
  // Substract
  if (strcmp(command, "-") == 0) {
    for (j=0;j<npix;j++) {
      f_spec[n_f_spec][j]=f_spec[n2][j]-f_spec[n1][j];
    }
  }
  // Add
  if (strcmp(command, "+") == 0) {
    for (j=0;j<npix;j++) {
      f_spec[n_f_spec][j]=f_spec[n2][j]+f_spec[n1][j];
    }
  }
  // Multiply
  if (strcmp(command, "*") == 0) {
    for (j=0;j<npix;j++) {
      f_spec[n_f_spec][j]=f_spec[n2][j]*f_spec[n1][j];
    }
  }
  // Divide
  if (strcmp(command, "/") == 0) {
    for (j=0;j<npix;j++) {
      if (f_spec[n1][j]!=0.0) {
        f_spec[n_f_spec][j]=f_spec[n2][j]/f_spec[n1][j];
      } else {
        f_spec[n_f_spec][j]=0.0001;
      }
    }
  }
  n_i_spax=-1;
  sprintf(interp->result,"%d",n_f_spec);
  n_f_spec++;
  if (n_f_spec>=MAX_NUM_SPEC) {
    n_f_spec=0;
  }
  return TCL_OK;
}

/*****************************************
 * Procedure to create a certain new Spectrum
 *****************************************/
static int create_spec_command(Tcl_Interp *interp, int argc, char *argv[])
{
  int i;

  if (start_n_f_spec!=0) {
    f_spec[n_f_spec] = (float *)malloc(npix*sizeof(float));
    for (i=0;i<npix;i++) {
      f_spec[n_f_spec][i]=f_spec_now[i];
    }
    sprintf(interp->result,"%d",n_f_spec);
    n_f_spec++;
    if (n_f_spec>=MAX_NUM_SPEC) {
      n_f_spec=0;
    }
  } else {
    sprintf(interp->result,"-666");
  }
  return TCL_OK;
}

/*****************************************
 * Procedure to load a new Spectrum
 *****************************************/
static int load_spec_command(Tcl_Interp *interp, int argc, char *argv[])
{
  int n_spec;
  if (argc < 1) {
    Tcl_AppendResult(interp, "USE: load_spec n_spec", NULL);
    return TCL_ERROR;
  }
  n_spec=atoi(argv[0]);
  f_spec_now=f_spec[n_spec];
  n_i_spax=-1;
  return TCL_OK;
}

/*****************************************
 * Procedure to create a certain new obj.
 *****************************************/
static int create_obj_command(Tcl_Interp *interp, int argc, char *argv[])
{
  int j;
  int n_obj;

  if (que_j_spax>(MAX_NUM_OBJ-1)) {
    que_j_spax=0;
  }
  n_obj=que_j_spax;
  if (n_i_spax==0) {
    n_obj=-666;
  } else {
    n_j_spax[n_obj]=n_i_spax;
    for (j=0;j<n_i_spax;j++) {
      j_spax[n_obj][j]=i_spax[j];
    }
    que_j_spax++;
  }
  sprintf(interp->result,"%d",n_obj);

  return TCL_OK;
}

/*****************************************
 * Procedure to read a certain object as the actual obj.
 *****************************************/
static int load_obj_command(Tcl_Interp *interp, int argc, char *argv[])
{
  int j;
  int n_obj;
  if (argc < 1) {
    Tcl_AppendResult(interp, "USE: load_obj n_obj", NULL);
    return TCL_ERROR;
  }
  n_obj=atoi(argv[0]);
  if (n_obj>MAX_NUM_OBJ) {
    Tcl_AppendResult(interp, "n_obj>MAX_NUM_OBJ : ERROR", NULL);
    return TCL_ERROR;
  }
  if (n_j_spax[n_obj]==0) {
    Tcl_AppendResult(interp, "No Spaxels in this object : ERROR", NULL);
    return TCL_ERROR;
  }
  n_i_spax=n_j_spax[n_obj];
  for (j=0;j<n_i_spax;j++) {
    i_spax[j]=j_spax[n_obj][j];
  }
  sprintf(interp->result,"%d",n_i_spax);
  return TCL_OK;
}

/*****************************************
 * Procedure to reverse a certain object as the actual obj.
 *****************************************/
static int reverse_obj_command(Tcl_Interp *interp, int argc, char *argv[])
{
  int j,k;
  int n_obj;
  int *i_spax_tmp;
  int n_spax_tmp;
  int exist;

  if (argc < 1) {
    Tcl_AppendResult(interp, "USE: reverse_obj n_obj", NULL);
    return TCL_ERROR;
  }
  n_obj=atoi(argv[0]);
  if (n_obj>MAX_NUM_OBJ) {
    Tcl_AppendResult(interp, "n_obj>MAX_NUM_OBJ : ERROR", NULL);
    return TCL_ERROR;
  }
  if (n_j_spax[n_obj]==0) {
    Tcl_AppendResult(interp, "No Spaxels in this object : ERROR", NULL);
    return TCL_ERROR;
  }

  i_spax_tmp = (int *)malloc((nb_spec+1)*sizeof(int));

  n_spax_tmp=0;
  for (k=0;k<nb_spec;k++) {
      exist=0;
      for (j=0;j<n_j_spax[n_obj];j++) {
          if (k==j_spax[n_obj][j]) {
              exist=1;
          }
      }
      if (exist==0) {
          i_spax_tmp[n_spax_tmp]=k;
          n_spax_tmp++;
      }
  }
  /* We copy the results!  */
  n_j_spax[n_obj]=n_spax_tmp;
  for (j=0;j<n_j_spax[n_obj];j++) {
      j_spax[n_obj][j]=i_spax_tmp[j];
  }

  sprintf(interp->result,"%d %d",n_obj,n_spax_tmp);
  free(i_spax_tmp);
  return TCL_OK;
}

/*************************************
 * Mark a certain region with an arrow
 ***********************************/
static int mark_range_command(Tcl_Interp *interp, int argc, char *argv[])
{
  float x1,y1,x2,y2;
  int dev_id,colorindex,width,style;
  char text[5];

  if (argc < 9 ) {
    Tcl_SetResult(interp, "USE: mark_range dev_id colorindex lweight lstyle text x1 y1 x2 y2", TCL_STATIC);
    return TCL_ERROR;
  }

  dev_id=atoi(argv[0]);
  colorindex=atoi(argv[1]);
  width=atoi(argv[2]);
  style=atoi(argv[3]);
  strcpy(text,argv[4]);
  x1=atof(argv[5]);
  y1=atof(argv[6]);
  x2=atof(argv[7]);
  y2=atof(argv[8]);
  cpgslct(dev_id);
  cpgsci(colorindex);
  cpgsls(style);
  cpgsch(width);
  mark_range(x1,y1,x2,y2,text);
  cpgsch(1.0);
  cpgsls(1);
  cpgsci(1);
  cpgslw(1);
  return TCL_OK;
}

/**********************************
 * Routine to determine the nearest SPAX
 * to a certain X,Y coordinates
 **********************************/
static int nearest_spax_command(Tcl_Interp *interp, int argc, char *argv[])
{
  float x,y;
  float x_c,y_c;
  float dist=10000000.0,dist_temp;
  int i,j,spax_id;

  if (argc < 2 ) {
    Tcl_AppendResult(interp, "USE: nearest_spax X Y", NULL);
    return TCL_ERROR;
  }

  x=atof(argv[0]);
  y=atof(argv[1]);
  spax_id=0;
  for (i=0;i<nb_spec;i++) {
    for (j=0;j<n_spaxels[i];j++) {
      x_c=spaxels[i][j].xpos;
      y_c=spaxels[i][j].ypos;
      dist_temp=sqrt(pow(x_c-x,2)+pow(y_c-y,2));
      if (dist_temp<dist) {
        dist=dist_temp;
        spax_id=i;
      }
    }
  }
  sprintf(interp->result,"%d",spax_id);
  return TCL_OK;
}

/**********************************
 * Routine to determine the flux
 * from a certain SPAX ID
 **********************************/
static int flux_spax_command(Tcl_Interp *interp, int argc, char *argv[])
{
  int i;
  float flux;

  if (argc < 1 ) {
    Tcl_AppendResult(interp, "USE: flux_spax SPAXEL_INDEX", NULL);
    return TCL_ERROR;
  }
  i=atoi(argv[0]);
  if (loaded==1) {
    flux=f_slice_now[i];
  } else {
    flux=0.0;
  }
  sprintf(interp->result,"%f",flux);
  return TCL_OK;
}

/**************************************************
 * Plot the Flux from a selection of spaxels SPAXEL
 *************************************************/
static int plot_flux_spax_command(Tcl_Interp *interp, int argc, char *argv[])
{
  int i,j;
  float *r,*flux,*s_flux,*flux_mod,*flux_res;
  int id,clean;
  float l_min_f,l_max_f,l_min_r,l_max_r;
  float Ig,Lg,Sg,Cg;
  int i_fwhm1,i_fwhm2,i_max;
  int fit=0;
  float a[4],s_a[4],d_a[4];
  int ia[4];
  float chisq;
  float chisq_cont=1.0;
  int niter=20;
  float f_tmp;

  if (argc < 2 ) {
    Tcl_AppendResult(interp, "USE: plot_flux_spax device clean", NULL);
    return TCL_ERROR;
  }

  id=atoi(argv[0]);
  clean=atoi(argv[1]);
  if (argc==3) {
    fit=atoi(argv[2]);
  }
  flux = (float *)malloc((n_i_spax)*sizeof(float));
  s_flux = (float *)malloc((n_i_spax)*sizeof(float));
  flux_res = (float *)malloc((n_i_spax)*sizeof(float));
  flux_mod = (float *)malloc((n_i_spax)*sizeof(float));
  r = (float *)malloc((n_i_spax)*sizeof(float));

  for (i=0;i<n_i_spax;i++) {
    flux[i]=f_slice_now[i_spax[i]];
    s_flux[i]=sqrt(fabs(flux[i]));
    for (j=0;j<n_spaxels[i_spax[i]];j++) {
      r[i]=sqrt(pow(spaxels[i_spax[i]][j].xpos-spaxels[i_spax[0]][j].xpos,2)+pow(spaxels[i_spax[i]][j].ypos-spaxels[i_spax[0]][j].ypos,2));
    }
  }
  l_min_f=flux[0];
  l_max_f=flux[0];
  l_min_r=r[0];
  l_max_r=r[0];
  i_max=0;
  for (i=0;i<n_i_spax;i++) {
    if (flux[i]<l_min_f) l_min_f=flux[i];
    if (flux[i]>l_max_f) {
      l_max_f=flux[i];
      i_max=i;
    }
    if (r[i]<l_min_r) l_min_r=r[i];
    if (r[i]>l_max_r) l_max_r=r[i];
  }

  Cg=l_min_f;
  Ig=(l_max_f-l_min_f);
  Lg=r[i_max];
  i=i_max;
  while(i<n_i_spax) {
    if (flux[i]<(0.5*Ig+l_min_f)) {
      i_fwhm2=i;
      i=n_i_spax;
    }
    i++;
  }
  i=i_max;
  while(i>0) {
    if (flux[i]<(0.5*Ig+l_min_f)) {
      i_fwhm1=i;
      i=0;
    }
    i--;
  }

  if (l_min_f>0) l_min_f=0.8*l_min_f;
  if (l_min_f<0) l_min_f=1.2*l_min_f;
  l_max_f=1.2*l_max_f;
  l_min_r=l_min_r-max_size;
  l_max_r=l_max_r+max_size;

  cpgslct(id);
  cpgask(0);
  if (clean==0) {
    cpgsch(1.2);
    cpgenv(l_min_r,l_max_r,l_min_f,l_max_f,0,0);
    cpgsch(1.5);
    cpglab("Distance (Arcsec)", "Flux", "");
  }
  cpgsls(1);
  cpgsci(1);
  cpgline(n_i_spax,r,flux);
  cpgsci(3);
  cpgpt(n_i_spax,r,flux,2);
  cpgsci(1);
  Sg=(i_fwhm2-i_fwhm1)/2.00;

  a[0]=Ig;
  a[1]=Lg;
  a[2]=Sg;
  a[3]=Cg;
  ia[0]=1;  ia[1]=1;  ia[2]=1;  ia[3]=1;
  d_a[0]=0.2*Ig;
  d_a[1]=0.05*Lg;
  d_a[2]=0.5*Sg;
  d_a[3]=0.5*Cg;

  if (fit==1) {
    for(i=0;i<niter;i++) {
      gl_fit(r,flux,s_flux,0,n_i_spax,a,s_a,d_a,ia,4,&chisq,chisq_cont,&gl_fit_gauss);
      d_a[0]=0.2*a[0];
      d_a[1]=0.05*a[1];
      d_a[2]=0.5*a[2];
      d_a[3]=0.5*a[3];
    }
    for (i=0;i<n_i_spax;i++) {
      gl_fit_gauss(r[i],a,&f_tmp);
      flux_mod[i]=f_tmp;
      flux_res[i]=flux[i]-flux_mod[i];
    }

    cpgsls(3);
    cpgsci(2);
    cpgline(n_i_spax,r,flux_mod);
    cpgsci(5);
    cpgpt(n_i_spax,r,flux_res,2);
    cpgsci(1);
    cpgsls(1);
  }

  free(flux);
  free(s_flux);
  free(flux_mod);
  free(flux_res);
  free(r);
  //  sprintf(text,"%f %f %f %f %f",a[0],a[1],2.345*a[2],a[3],chisq);
  cpgswin(0.0, 1.0, 0.0, 1.0);
  //cpgptxt(0.1,1.0,0.0,1.5,text);

  sprintf(interp->result,"%f %f %f %f %f",a[0],a[1],2.345*a[2],a[3],chisq);
  return TCL_OK;
}

/***************************************************************
 * Routine that determines the mean, standard deviation, median,
 * and possibly centroid positions from the current selection of
 * spaxels
 ***************************************************************/
static int stats_spax_command(Tcl_Interp *interp, int argc, char *argv[]) {
  int i,n;
  float *flux;
  float mean,sigma,median;
  float x_c,y_c,x_cen=0,y_cen=0;

  if (loaded==1) {
    n=n_i_spax;
    if (n>1) {
      flux = (float *)malloc((n_i_spax)*sizeof(float));
      for (i=0;i<n_i_spax;i++) {
        flux[i]=f_slice_now[i_spax[i]];
        x_c=spaxels[i_spax[i]][0].xpos;
        y_c=spaxels[i_spax[i]][0].ypos;
        x_cen=x_cen+x_c*flux[i];
        y_cen=y_cen+y_c*flux[i];
      }
      mean=fmean(flux,n);
      median=fmedian(flux,n);
      sigma=fsigma(flux,n,mean);
      if (mean!=0) {
        x_cen=x_cen/(mean*n);
        y_cen=y_cen/(mean*n);
      } else {
        x_cen=0;
        y_cen=0;
      }
    } else {
      mean=f_slice_now[0];
      median=f_slice_now[0];
      sigma=0.0;
    }
  } else {
    n=0;
    mean=0.0;
    median=0.0;
    sigma=0.0;
  }
  sprintf(interp->result,"%d %f %f %f %f %f",n,mean,sigma,median,x_cen,y_cen);
  return TCL_OK;
}

/**********************************
 * Routine to determine a list of SPAXELS
 * that distance less than a certain
 * radius from the given position
 *
 * The routine extract the list element by element
 **********************************/
static int encircled_spax_command(Tcl_Interp *interp, int argc, char *argv[]) {
  float x,y;
  float x_c,y_c,r;
  float dist_temp;
  int i,j,n_spax;
  int *i_spax;
  int index;

  if (argc < 4 ) {
    Tcl_AppendResult(interp, "USE: encircled_spax X Y R index", NULL);
    return TCL_ERROR;
  }

  x=atof(argv[0]);
  y=atof(argv[1]);
  r=atof(argv[2]);
  index=atoi(argv[3]);
  n_spax=0;
  i_spax = (int *)malloc((nb_spec)*sizeof(int));
  for (i=0;i<nb_spec;i++) {
    for (j=0;j<n_spaxels[i];j++) {
      x_c=spaxels[i][j].xpos;
      y_c=spaxels[i][j].ypos;
      dist_temp=sqrt(pow(x_c-x,2)+pow(y_c-y,2));
      if (dist_temp<r) {
        i_spax[n_spax]=i;
        n_spax++;
      }
    }
  }
  if (index>=n_spax) index=n_spax-1;
  sprintf(interp->result,"%d %d",n_spax,i_spax[index]);
  free(i_spax);

  return TCL_OK;
}

/**********************************
 * routine to save the selected spectra
 **********************************/
static int save_spectra_sel_command(Tcl_Interp *interp, int argc, char *argv[]) {
  char **tokens;
  int n_tokens,j;

  if (argc < 1 ) {
    n_i_spax=0;
  } else {
    split_text(argv[0]," ",&n_tokens,&tokens);
    if (n_tokens>0) {
      n_i_spax=n_tokens;
      for (j=0;j<n_tokens;j++) {
        i_spax[j]=atoi(tokens[j]);
      }
    }
  }
  return TCL_OK;
}

/**********************************
 * routine to load the number selected spectra
 **********************************/
static int load_n_spectra_sel_command(Tcl_Interp *interp, int argc, char *argv[]) {
  sprintf(interp->result,"%d",n_i_spax);
  return TCL_OK;
}

/**********************************
 * routine to load the selected spectra
 * from C-Memory
 **********************************/
static int load_spectra_sel_command(Tcl_Interp *interp, int argc, char *argv[]) {
  int n_sel;

  if (argc < 1) {
    Tcl_AppendResult(interp, "USE: load_sepectra_sel n_sel", NULL);
    return TCL_ERROR;
  }
  n_sel=atoi(argv[0]);
  if (n_i_spax>0) {
    sprintf(interp->result,"%d",i_spax[n_sel]);
  } else {
    sprintf(interp->result,"-666");
  }
  return TCL_OK;
}

/*********************************
 * Subroutine that plots the Spectra
 * in the Spectra inspector
 *********************************/
static int plot_spectra_command(Tcl_Interp *interp, int argc, char *argv[]) {
  int nargc;
  char *p_names[]={"grey","rainbow","heat","iraf","aips"};
  float min;
  float max;
  int id,j,i,k,clean;
  float *spectra;
  float *raw_spectra;
  float *slice;
  int p;
  float bright,contra,sign;
  char **tokens;
  float local_min,local_max;
  int flag=0;
  float *x,*y;
  float temp_mean,temp_sigma;
  int save=0;
  int n_buffer=-1;
  int n_sls=1;
  int n_sci=16;
  float out_min,out_max;

  if (argc < 7) {
    Tcl_AppendResult(interp, "USE: plot_spectra min max device clean palette bright contra sign i_spax", NULL);
    return TCL_ERROR;
  }
  min=atof(argv[0]);
  max=atof(argv[1]);
  id=atoi(argv[2]);
  clean=atoi(argv[3]);
  bright=atof(argv[5]);
  contra=atof(argv[6]);
  sign=atof(argv[7]);
  flag=atof(argv[8]);
  for (j=0;j<5;j++) {
    if (strstr(p_names[j],argv[4])) p=j+1;
  }
  split_text(argv[8]," ",&nargc,&tokens);
  save=type_spec_save;
  n_buffer=type_spec_n_buffer;

  /*
   * We determine the values to plot
   */
  spectra = (float *)malloc((npix)*sizeof(float));

  if (nargc>0) {
    /*
     * We create a new pseudo-spectra
     */
    if (n_sel_spec!=nargc) pintados[id]=0;
    n_i_spax=nargc;
    n_sel_spec=nargc;
    for (j=0;j<nargc;j++) {
      i_spax[j]=atoi(tokens[j]);
      if (i_spax[j]<0) i_spax[j]=0;
      if (i_spax[j]>=nb_spec) i_spax[j]=nb_spec-1;
    }
    if (i_spax[0]!=old_spax_id[id]) pintados[id]=0;
    old_spax_id[id]=i_spax[0];

    raw_spectra = (float *)malloc((npix*nargc)*sizeof(float));
    slice = (float *)malloc((nargc)*sizeof(float));

    for (i=0;i<npix;i++) {
      spectra[i]=0.0;
    }
    for (i=0;i<npix;i++) {
      for (k=0;k<nargc;k++) {
        raw_spectra[i+k*npix]=raw_data[i+i_spax[k]*npix];
        slice[k]=raw_data[i+i_spax[k]*npix];
      }
      if (flag_spec==0) {

        spectra[i]=mean(slice,nargc);//temp_mean;
      }
      if (flag_spec==1) {
        mean_loop(slice,nargc,2,2,&temp_mean,&temp_sigma);
        spectra[i]=temp_mean;
      }
    }

    /********************************************************
     * START PLOTTING OF PLOTTING
     *******************************************************/
    cpgslct(id);
    cpgask(0);
    palette(p,sign*0.8,0.6);

    if (old_p[id]!=p)  pintados[id]=0;
    if (old_min[id]!=min) pintados[id]=0;
    if (old_max[id]!=max) pintados[id]=0;
    if (clean!=0) pintados[id]=0;
    if (old_npix_w1!=npix_w1) pintados[id]=0;
    if (old_npix_w2!=npix_w2) pintados[id]=0;

    if (pintados[id]<1) {
      cpgpage();
      old_p[id]=p;
    }
    switch (type_spec_rep) {
    case(0):
      cpgsvp(0.1,0.95,0.55,0.99);
      cpgswin(1.0+npix_w1,npix_w2+1.0,1.0,nargc+0.1);
      local_min=min;
      local_max=max;
      tr_palett(npix*nargc,raw_spectra,&tmp_data,sign,contra,bright,min,max,&out_min,&out_max);
      cpgimag(tmp_data,npix,nargc,1,npix,1,nargc,out_min,out_max,tr);
      free(tmp_data);
      pintados[id]++;
      if (pintados[id]>1) pintados[id]=1;
      x = (float *)malloc((npix_w2-npix_w1)*sizeof(float));
      y = (float *)malloc((npix_w2-npix_w1)*sizeof(float));
      for (i=npix_w1;i<npix_w2;i++) {
          x[i-npix_w1]=(float)start_w+(float)delta_w*i;
          y[i-npix_w1]=spectra[i];
      }

      cpgsvp(0.1,0.95,0.1,0.55);
      cpgswin(start_w+delta_w*(npix_w1),start_w+delta_w*(npix_w2),min,max);
      cpgsls(1);
      cpgline((npix_w2-npix_w1),x,y);
      free(x);
      free(y);

      cpgsvp(0.1,0.95,0.1,0.54);
      cpgsci(1);
      cpgsch(1.2);
      cpgbox("IBTSVN", 0, 0, "BITSNVC", 0, 0);
      cpgsch(1.2);
      break;

    case(1):
      cpgsvp(0.1,0.95,0.1,0.99);
      cpgswin(1.0+npix_w1,npix_w2+1.0,1.0,nargc+0.1);
      local_min=min;
      local_max=max;
      tr_palett(npix*nargc,raw_spectra,&tmp_data,sign,contra,bright,min,max,&out_min,&out_max);
      cpgimag(tmp_data,npix,nargc,1,npix,1,nargc,out_min,out_max,tr);
      free(tmp_data);
      pintados[id]++;
      if (pintados[id]>1) pintados[id]=1;
      x = (float *)malloc((npix_w2-npix_w1)*sizeof(float));
      y = (float *)malloc((npix_w2-npix_w1)*sizeof(float));
      for (i=npix_w1;i<npix_w2;i++) {
          x[i-npix_w1]=(float)start_w+(float)delta_w*i;
          y[i-npix_w1]=spectra[i];
      }
      cpgsvp(0.1,0.95,0.1,0.99);
      cpgswin(start_w+delta_w*(npix_w1),start_w+delta_w*(npix_w2),min,max);
      cpgsls(1);
      free(x);
      free(y);

      cpgsci(1);
      cpgsch(1.2);
      cpgbox("IBTSVNC", 0, 0, "BITSVC", 0, 0);
      cpgsch(1.2);
      break;

    case(2):
      cpgsvp(0.1,0.95,0.55,0.99);
      cpgswin(1.0+npix_w1,npix_w2+1.0,1.0,nargc+0.1);
      local_min=min;
      local_max=max;
      tr_palett(npix*nargc,raw_spectra,&tmp_data,sign,contra,bright,min,max,&out_min,&out_max);
      cpgimag(tmp_data,npix,nargc,1,npix,1,nargc,out_min,out_max,tr);
      free(tmp_data);
      pintados[id]++;
      if (pintados[id]>1) pintados[id]=1;
      x = (float *)malloc((npix_w2-npix_w1)*sizeof(float));
      y = (float *)malloc((npix_w2-npix_w1)*sizeof(float));
      cpgsvp(0.1,0.95,0.1,0.55);
      cpgswin(start_w+delta_w*(npix_w1),start_w+delta_w*(npix_w2),min,max);
      for (j=0;j<nargc;j++) {
        for (i=npix_w1;i<npix_w2;i++) {
          x[i-npix_w1]=(float)start_w+(float)delta_w*i;
          y[i-npix_w1]=raw_spectra[i+j*npix];
        }
        cpgsls(n_sls);
        cpgsci(n_sci);
        cpgline((npix_w2-npix_w1),x,y);
        n_sci=n_sci+20;
        if (n_sci>255) n_sci=16;
      }
      free(x);
      free(y);

      cpgsvp(0.1,0.95,0.1,0.54);
      cpgsci(1);
      cpgsch(1.2);
      cpgbox("IBTSVN", 0, 0, "BITSNVC", 0, 0);
      cpgsch(1.2);
      break;

    case(3):
      pintados[id]++;
      if (pintados[id]>1) pintados[id]=1;
      x = (float *)malloc((npix_w2-npix_w1)*sizeof(float));
      y = (float *)malloc((npix_w2-npix_w1)*sizeof(float));
      cpgsvp(0.1,0.95,0.1,0.95);
      cpgswin(start_w+delta_w*(npix_w1),start_w+delta_w*(npix_w2),min,max);
      for (j=0;j<nargc;j++) {
        for (i=npix_w1;i<npix_w2;i++) {
          x[i-npix_w1]=(float)start_w+(float)delta_w*(i);
          y[i-npix_w1]=raw_spectra[i+j*npix];
        }
        cpgsls(n_sls);
        cpgsci(n_sci);
        cpgline((npix_w2-npix_w1),x,y);
        n_sci=n_sci+20;
        if (n_sci>255) n_sci=16;
      }
      free(x);
      free(y);

      cpgsvp(0.1,0.95,0.1,0.95);
      cpgsci(1);
      cpgsch(1.2);
      cpgbox("ITSBCN", 0, 0, "ITSBCNV", 0, 0);
      cpgsch(1.2);
      break;

    case(4):
      pintados[id]++;
      if (pintados[id]>1) pintados[id]=1;
      x = (float *)malloc((npix_w2-npix_w1)*sizeof(float));
      y = (float *)malloc((npix_w2-npix_w1)*sizeof(float));
      cpgsvp(0.1,0.95,0.1,0.95);
      cpgswin(start_w+delta_w*(npix_w1),start_w+delta_w*(npix_w2),min,max);
      for (i=npix_w1;i<npix_w2;i++) {
        x[i-npix_w1]=(float)start_w+(float)delta_w*(i);
        y[i-npix_w1]=spectra[i];
      }
      cpgsls(n_sls);
      cpgsci(n_sci);
      cpgline((npix_w2-npix_w1),x,y);
      free(x);
      free(y);

      cpgsvp(0.1,0.95,0.1,0.95);
      cpgsci(1);
      cpgsch(1.2);
      cpgbox("ITSBCN", 0, 0, "ITSBCNV", 0, 0);
      cpgsch(1.2);
      break;
    } /* switch */
    /********************************************************
     * END OF PLOTTING
     *******************************************************/

    /*
     * We save the plotted spectrum in the current n_f_spec
     */
    free(f_spec_now);
    f_spec_now = (float *)malloc((npix)*sizeof(float));

    for (i=0;i<npix;i++) {
      f_spec_now[i]=spectra[i];
    }
    start_n_f_spec=1;
    old_min[id]=min;
    old_max[id]=max;
  }
  old_npix_w1=npix_w1;
  old_npix_w2=npix_w2;
  free(raw_spectra);
  free(spectra);
  free(slice);

  return TCL_OK;
}

/*********************************
 * Subroutine that plots the Spectra
 * from the memory
 *********************************/
static int plot_spectra_mem_command(Tcl_Interp *interp, int argc, char *argv[]) {
  char *p_names[]={"grey","rainbow","heat","iraf","aips"};
  float min;
  float max;
  int id,j,id_s;
  int p;
  float bright,contra,sign;
  int clean=0;

  if (argc < 8) {
    Tcl_AppendResult(interp, "USE: plot_spectra_mem min max device device2 palette bright contra sign [clean]", NULL);
    return TCL_ERROR;
  }

  min=atof(argv[0]);
  max=atof(argv[1]);
  id=atoi(argv[2]);
  id_s=atoi(argv[3]);

  bright=atof(argv[5]);
  contra=atof(argv[6]);
  sign=atof(argv[7]);
  for (j=0;j<5;j++) {
    if (strstr(p_names[j],argv[4])) p=j+1;
  }
  if (argc==9) {
    clean=atoi(argv[8]);
  }
  cpgslct(id);
  palette(p,sign*0.8,0.6);
  cpgask(0);
  spec_plot_range(f_spec_now,min,max,npix_w1,npix_w2,start_w,delta_w,clean);

  return TCL_OK;
}

/*********************************
 * Subroutine that fits an spectra
 * to a single line gaussian function plus continuum.
 * It uses as input the last mean spectrum available.
 *********************************/
static int fit_single_line_command(Tcl_Interp *interp, int argc, char *argv[]) {
  char *p_names[] = {"grey","rainbow","heat","iraf","aips"};
  float min, max;
  int id, j, i, p;
  float bright, contra, sign;
  int clean = 0;
  float a[4], d_a[4];
  float chisq,chisq_lim;
  float *noise_tmp;
  int ia[4];

//printf("___fit_single_line_command___\n");
  if (argc < 21) {
    Tcl_AppendResult(interp, "USE: fit_single_line min max device palette bright contra sign Ig D_Ig fit Wo D_Wo fit sigma D_sigma fit Continuum D_Continuum fit CHISQ_LIM paint", NULL);
    return TCL_ERROR;
  }
  min=atof(argv[0]);
  max=atof(argv[1]);
  id=atoi(argv[2]);

  bright=atof(argv[4]);
  contra=atof(argv[5]);
  sign=atof(argv[6]);
  for (j=0; j<5; j++) {
    if (strstr(p_names[j],argv[3])) p=j+1;
  }
  for (j=0; j<4; j++) {
    a[j]=atof(argv[3*j+7]);
    d_a[j]=atof(argv[3*j+8]);
    ia[j]=atoi(argv[3*j+9]);
  }
  if (a[2]>1000) {
    a[2]=10;
  }
  if (a[2]==0.0) {
    a[2]=0.1;
  }
  if (a[3]<=0.0) {
    a[3]=0.1;
  }
  chisq_lim=atof(argv[19]);
  clean=atoi(argv[20]);

  cpgslct(id);
  palette(p,sign*0.8,0.6);
  cpgask(0);

  noise_tmp = (float *)malloc((npix)*sizeof(float));
  for (i=0;i<npix;i++) {
    noise_tmp[i]=sqrt(sqrt(f_spec_now[i]*f_spec_now[i]));
  }
//cpgswin(0.0, 1.0, 0.0, 1.0);
//printf("in:  a=%f %f %f %f, chisq=%f\n",a[0],a[1],a[2],a[3],chisq);
  chisq=fit_single_line(f_spec_now,noise_tmp,min,max,npix_w1,npix_w2,start_w,delta_w,clean,a,d_a,ia,chisq_lim);
//printf("out: a=%f %f %f %f, chisq=%f\n",a[0],a[1],a[2],a[3],chisq);
  sprintf(interp->result,"%f %f %f %f %f",a[0],a[1],a[2],a[3],chisq);
  free(noise_tmp);
  return TCL_OK;
}

/****************************************************
 * Subroutine that guess the values that correspond
 * to a single line gaussian function plus continuum,
 * to be used by "fit_single_line".
 * It uses as input the last mean spectrum available.
 ****************************************************/
static int guess_single_line_command(Tcl_Interp *interp, int argc, char *argv[]) {
  float max=0,min=0;
  double Ig=0,Lg=0,Sg=0,Cg=0;
  int i,i_max=0,i_fwhm1=0,i_fwhm2=0;

  min = max = f_spec_now[npix_w1];
  i_max = npix_w1;
  for (i = npix_w1; i < npix_w2; i++) {
    if (f_spec_now[i] > max) {
      max = f_spec_now[i];
      i_max = i;
    }
    if (f_spec_now[i] < min)
      min = f_spec_now[i];
  }
//printf("guess_single_line_comment: %f %f %d\n", min, max, i_max);
  Cg = min;
  Ig = max - min;
  Lg = start_w + delta_w*i_max;
  i = i_max;
  while (i < npix_w2) {
    if (f_spec_now[i] < (0.5*Ig + min)) {
      i_fwhm2 = i;
      i = npix_w2;
    }
    i++;
  }
  i = i_max;
  while (i > npix_w1) {
    if (f_spec_now[i] < (0.5*Ig + min)) {
      i_fwhm1 = i;
      i = npix_w1;
    }
    i--;
  }
  Sg = fabs(i_fwhm2 - i_fwhm1)/2;
  sprintf(interp->result, "%f %f %f %f", Ig, Lg, Sg, Cg);
  return TCL_OK;
}

/*********************************
 * Subroutine that feeds a slice
 * with a list
 *********************************/
static int feed_slice_command(Tcl_Interp *interp, int argc, char *argv[]) {
  int j;
  char **tokens;
  int n_tokens;
  int n_feed;

  if (argc < 2) {
    Tcl_AppendResult(interp, "USE: feed_slice n_feed list", NULL);
    return TCL_ERROR;
  }
  n_feed=atoi(argv[0]);
  split_text(argv[1]," ",&n_tokens,&tokens);
  f_slice_def[n_feed]=1;
  for (j=0;j<nb_spec;j++) {
    f_slice[n_feed][j]=0.0;
  }
  if (n_tokens>nb_spec) n_tokens=nb_spec;
  for (j=0;j<n_tokens;j++) {
    f_slice[n_feed][j]=atof(tokens[j]);
  }
  sprintf(interp->result,"%d %d",n_feed,slice_index);
  return TCL_OK;
}

static int plot_scale_command(Tcl_Interp *interp, int argc, char *argv[]) {
  char *p_names[]={"grey","rainbow","heat","iraf","aips"};
  float min;
  float max;
  int id,j,i;
  int color_index,n_colors;
  int nc=240;
  int p;
  float bright,contra,sign;
  float scale[255],*tmp_scale;
  char title[20];
  float top_max,top_min;
  int border=0;
  float out_min,out_max;

  if (argc < 11) {
    Tcl_AppendResult(interp, "USE: plot_scale min max device bright contra sign colormap title top_min top_max border", NULL);
    return TCL_ERROR;
  }
  min=atof(argv[0]);
  max=atof(argv[1]);
  id=atoi(argv[2]);
  bright=atof(argv[3]);
  contra=atof(argv[4]);
  sign=atoi(argv[5]);
  for (j=0;j<5;j++) {
    if (strstr(p_names[j],argv[6])) p=j+1;
  }
  strcpy(title,argv[7]);
  top_min=atof(argv[8]);
  top_max=atof(argv[9]);
  border=atof(argv[10]);
  if (bright==0) bright=0.01;
  if (contra==0) bright=0.01;

  cpgslct(id);
  cpgask(0);
  cpgsch(0.9);

  cpgsvp(0.90,0.95,0.1,0.9);
  cpgswin(0.0, 1.0, top_min, (1.0)*top_max);
  cpgbox("IBC", 1.0, 5, "ITSBCNV", floor((top_max-top_min)/6), 5);
  cpglab("", "", title);
  for (i=0;i<255;i++) {
    scale[i]=top_min+((top_max-top_min)/255)*i;
  }
  tr_palett(255,scale,&tmp_scale,sign,contra,bright,min,max,&out_min,&out_max);
  for (i=1;i<254;i++) {
    n_colors=nc;
    color_index=16+floor((tmp_scale[i]-out_min)/(out_max-out_min)*(n_colors-1));
    if (color_index>255) color_index=255;
    if (color_index<17) color_index=17;
    cpgsci(color_index);
    cpgsfs(1);
    cpgrect(0.01,0.99,scale[i],scale[i+1]);
  }
  free(tmp_scale);
  cpgsci(1);
  if (border==1) {
    cpgsvp(0.203,0.797,0.1,0.9);
    cpgswin(x_max+0.2,x_min-0.2,y_min-0.2,y_max+0.2);
    cpgbox("ITSBCN", floor((x_max-x_min)/5), 5, "ITSBCN",floor((y_max-y_min)/5),5);
    cpglab("", "E", "N");
  }

  return TCL_OK;
}

/*********************************
 * Subroutine that plots the Spaxels
 * in the Spaxels inspector
 *********************************/
static int plot_spaxels_command(Tcl_Interp *interp, int argc, char *argv[])
{
  char *p_names[]={"grey","rainbow","heat","iraf","aips"};
  float min;
  float max;
  float old_max,old_min;
  int id,j,i,k;
  int i0,i1,i_temp;
  float *slice;
  float *tmp_val,tmp_val_out,tmp_sval_out;
  GROUP grupo;
  int color_index,n_colors;
  int nc=240;
  float x_c,y_c;
  int p;
  float bright,contra,sign;
  int save=0;
  int fill=1;
  int text_flag=0;
  float temp_float[200];
  char text[5];
  int new_img=0;
  float tr_tmp[]={0.0,1.0,0.0,0.0,0.0,1.0};
  float *tmp_slice;
  float real_min,real_max;
  float error;
  int back_color=1;
  float scale[255],*tmp_scale;
  char title[20];
  // Physical size of the plotting paper
  float fis_x_min,fis_x_max,fis_y_min,fis_y_max;
  float out_min,out_max;

  if (argc < 5) {
    Tcl_AppendResult(interp, "USE: plot_spaxels min max dev_id i1 i2 palette bright contra sign save fill type [new_img] [back_color] [n_contours]", NULL);
    return TCL_ERROR;
  }

  min=atof(argv[0]);
  max=atof(argv[1]);
  id=atoi(argv[2]);
  i0=floor(atof(argv[3]));
  i1=floor(atof(argv[4]));
  save=0;
  fill=atoi(argv[10]);
  text_flag=atoi(argv[11]);
  if (argc>12) {
    new_img=atoi(argv[12]);
  }
  if (argc>13) {
    back_color=atoi(argv[13]);
  }
  if (argc==15) {
      n_contours=atoi(argv[14]);
      if (n_contours>199) n_contours=199;
  }
  old_max=max;
  old_min=min;
  if (text_flag<90) {
    strcpy(title,"Flux");
    if (i1<i0) {
      i_temp=i1;
      i1=i0;
      i0=i_temp;
    }
    if (i0<0) i0=0;
    if (i0>npix) i0=npix-1;
    if (i1>npix) i1=npix;
    if (i1<1) i1=1;
  }

  bright=atof(argv[6]);
  contra=atof(argv[7]);
  sign=atof(argv[8]);
  for (j=0;j<5;j++) {
    if (strstr(p_names[j],argv[5])) p=j+1;
  }

  if (bright==0) bright=0.01;
  if (contra==0) bright=0.01;

  /*
   * TEXT_FLAG>100 Handling of VELOCITY MAPS
   * i0 is the velocity MAP to display
   * i1 is the values to plot: 0-> flux, 1-> Vel., 2 ->Delta_Vel
   */
  if (text_flag>90) {
    text_flag=text_flag-100;
    if (vel_map_fill[i0]==0) {
        sprintf(interp->result,"Velocity Map %d not loaded",i0);
        return TCL_OK;
    }
    if (i1>2) i1=2;
    if (i1<0) i1=0;
    slice_index=i0;
    slice = (float *)malloc((nb_spec)*sizeof(float));
    for (i=0;i<nb_spec;i++) {
      if (i1==0) {
        slice[i]=vel_map[i0].f[i];
        error=fabs(vel_map[i0].e_f[i]/(vel_map[i0].f[i]+0.0001));
        strcpy(title,"Flux");
      }
      if (i1==1) {
        slice[i]=vel_map[i0].w[i];
        error=fabs(vel_map[i0].e_w[i]/(vel_map[i0].w[i]+0.0001));
        strcpy(title,"km/s");
      }
      if (i1==2) {
        slice[i]=vel_map[i0].dw[i];
        error=fabs(vel_map[i0].e_dw[i]/(vel_map[i0].dw[i]+0.0001));
        strcpy(title,"km/s");
      }
      slice[i]=slice[i]*vel_map[i0].q[i];
    }
  } else {
    /* TEXT_FLAG>15 Handling with saved data */
    if (text_flag<15) {
      /* We determine the range of spectra to ask for */
      slice = (float *)malloc(nb_spec*sizeof(float));
      tmp_val = (float *)malloc((i1-i0)*sizeof(float));
      for (i=0;i<nb_spec;i++) {
        if ((i1-i0)<4) {
          slice[i]=0;
          for (j=i0;j<i1;j++) {
            slice[i]+=raw_data[j+i*npix];
          }
          slice[i]/=(i1-i0);
        } else {
          for (j=i0;j<i1;j++) {
            tmp_val[j-i0]=raw_data[j+i*npix];
          }
          mean_loop(tmp_val,i1-i0,1,2,&tmp_val_out,&tmp_sval_out);
          slice[i]=tmp_val_out;
        }
        f_slice_now[i]=slice[i];
      }
      free(tmp_val);
      /*
       * We save the data for future
       */
    } else {
      text_flag=text_flag-20;
      save=0;
      slice_index=i0;
      slice = (float *)malloc((nb_spec)*sizeof(float));
      for (i=0;i<nb_spec;i++) {
        if (f_slice_def[slice_index]==1) {
          slice[i]=f_slice[slice_index][i];
          f_slice_now[i]=f_slice[slice_index][i];
        } else {
          slice[i]=0.0;
          f_slice_now[i]=0.0;
        }
      }
    }
  }
  if ((text_flag>3)||(text_flag==-1)||(text_flag==-2)) {
    n_map=slice_index;
#ifdef DEBUG_dpix
    printf("plot_spaxels_command: dpix=%f/%f", dpix1,dpix2);
#endif
    create_map(nb_spec,n_spaxels,spaxels,npts,slice,dpix1,dpix2,&map_data[n_map],x_min,x_max,y_min,y_max,tr_map[n_map],&nx_map[n_map],&ny_map[n_map],grid_func,grid_opt);
#ifdef DEBUG_dpix
    printf(" dpix=%f/%f\n", dpix1,dpix2);
#endif
    map_created[n_map]=1;
    i0_slice=i0;
  }

  /*
   * We plot in the desired Screen
   */
  if (spax_painted==0) {
    spax_painted=1;
  }
  cpgslct(id);
  cpgask(0);
  palette(p,sign*0.8,0.6);
  color_palett(p,0.8,0.6,floor(sign));
  if (new_img==0) {
    if (pintados[id]<2) {
      cpgpage();
      cpgsvp(0.0,1.0,0,1.0);
      cpgqvp(1,&fis_x_min,&fis_x_max,&fis_y_min,&fis_y_max);
      if (fabs(fis_x_max-fis_x_min)<fabs(fis_y_max-fis_y_min)) {
        fis_y_min=fis_y_max-fabs(fis_x_max-fis_x_min);
      } else {
        fis_x_max=fis_x_min+fabs(fis_y_max-fis_y_min);
      }
      if (fis_x_max>1) {
        cpgvsiz(fis_x_min,fis_x_max,fis_y_min,fis_y_max);
      }
      cpgswin(x_max+max_size,x_min-max_size,y_min-max_size,y_max+max_size);
      old_p[id]=p;
      old_text_flag[id]=text_flag;
    } else {
      if ((old_p[id]!=p)||(old_text_flag[id]!=text_flag)) {
        cpgpage();
        pintados[id]=0;
      }
    }
    palette(p,sign*0.8,0.6);
    pintados[id]++;
    if (pintados[id]>3) pintados[id]=3;
  }

  /*
   * In case of a Postscript!
   */
  if (new_img==2) {
    cpgsvp(0.22,0.80,0.1,0.85);
    cpgswin(x_max+max_size,x_min-max_size,y_min-max_size,y_max+max_size);
    cpgbox("BTSVNC",0.0,0.0, "BTSNC",0.0,0.0);
    cpgsci(1);
    cpglab("\\gD\\ga", "\\gD\\gd", "");
  }

  /***********************
   * with text_flag>3 we plot maps
   ***********************/
  if (text_flag>3) {
    n_map=slice_index;
    if (i0_slice!=i0) {
      map_created[n_map]=1;
      i0_slice=i0;
    }
    tr_tmp[0]=x_min-0.5*dpix1;//+0.2-dpix-0.5*max_size;
    tr_tmp[1]=dpix1; /* in the hope that this is about x */
    tr_tmp[2]=0;
    tr_tmp[3]=y_min-0.5*dpix2;//+0.2-dpix-0.5*max_size;
    tr_tmp[4]=0;
    tr_tmp[5]=dpix2; /* in the hope that this is about y */

    color_palett(p,0.8,0.6,floor(sign));
    if ((nx_map[n_map]==0)||(ny_map[n_map]==0)) {
#ifdef DEBUG_dpix
      printf("plot_spaxels_command: dpix=%f/%f", dpix1,dpix2);
#endif
      create_map(nb_spec,n_spaxels,spaxels,npts,slice,dpix1,dpix2,
                 &map_data[n_map],x_min,x_max,y_min,y_max,
                 tr_map[n_map],&nx_map[n_map],&ny_map[n_map],
                 grid_func,grid_opt);
#ifdef DEBUG_dpix
      printf(" dpix=%f/%f\n", dpix1,dpix2);
#endif
    }
    tr_palett(nx_map[n_map]*ny_map[n_map],map_data[n_map],&tmp_data,
              sign,contra,bright,min,max,&out_min,&out_max);
    cpgimag(tmp_data,nx_map[n_map],ny_map[n_map],1,nx_map[n_map],1,
            ny_map[n_map],out_min,out_max,tr_tmp);
      free(tmp_data);

    if (new_img==2) {
      cpgsch(0.6);
      cpgbox("bci",0.0,0,"bci",0.0,0);
      cpgwedg("RI",0.0,8.0,min,max,"");
      cpgsch(1.0);
    }
  } else {
    tr_palett(nb_spec,slice,&tmp_slice,sign,contra,bright,min,max,&out_min,&out_max);
    if (text_flag!=-2) {
      color_palett(p,0.8,0.6,floor(sign));
      for (i=0;i<nb_spec;i++) {
        n_colors=nc;
        color_index=16+floor((tmp_slice[i]-out_min)/(out_max-out_min)*(n_colors-1));
        if (color_index>255) color_index=255;
        if (color_index<17) color_index=17;
        cpgsci(color_index);
        for (j=0;j<n_spaxels[i];j++) {
          x_c=spaxels[i][j].xpos;
          y_c=spaxels[i][j].ypos;
          grupo=groups[spaxels[i][j].group-1];
          plot_spaxel(color_index,x_c,y_c,grupo,fill,0);
        }
      }
    }

    if ((text_flag==1)||(text_flag==2)) {
      for (i=0;i<nb_spec;i++) {
        for (j=0;j<n_spaxels[i];j++) {
          x_c=spaxels[i][j].xpos;
          y_c=spaxels[i][j].ypos;
          grupo=groups[spaxels[i][j].group-1];
          cpgslw(2.0);
          plot_spaxel(2,x_c,y_c,grupo,2,0);
          cpgsch(0.6);
          if (text_flag==2) {
            cpgsci(1);
            sprintf(text,"%d",i+1); // start counting elements at 1 not 0
            cpgptxt(x_c,y_c,0.0,0.5,text);
          }
          cpgsch(1.1);
        }
      }
    }
    cpgsci(1);

    if ((text_flag==-1)||(text_flag==-2)) {
      n_map=slice_index;
      if (i0_slice!=i0) {
        map_created[n_map]=1;
        i0_slice=i0;
      }
      tr_tmp[0]=x_min-0.5*dpix1;
      tr_tmp[1]=dpix1;
      tr_tmp[2]=0;
      tr_tmp[3]=y_min-0.5*dpix2;
      tr_tmp[4]=0;
      tr_tmp[5]=dpix2;
      for (i=0;i<n_contours;i++) temp_float[i]=min+(max-min)/n_contours*i;
      cpgslw(2);
      cpgsci(back_color);
      cpgsls(5);
      cpgsch(1.6);
      cpgslw(3);
      cpgcont(map_data[n_map],nx_map[n_map],ny_map[n_map],1,nx_map[n_map],1,ny_map[n_map],temp_float,n_contours,tr_tmp);
      cpgsch(1.6);
      cpgslw(1);
      cpgsls(1);
    } /* if */

    /*
     * We plot the selected spaxels
     */
    for (k=0;k<n_i_spax;k++) {
      i=i_spax[k];
      if ((i>=0)&&(i<nb_spec)) {
        for (j=0;j<n_spaxels[i];j++) {
          x_c=spaxels[i][j].xpos;
          y_c=spaxels[i][j].ypos;
          grupo=groups[spaxels[i][j].group-1];
          cpgslw(1.0);
          plot_spaxel(0,x_c,y_c,grupo,3,0);
          plot_spaxel(1,x_c,y_c,grupo,2,0);
          cpgsch(0.6);
        }
      }
    }
    cpgsci(1);

    /*
     * We plot the border levels!
     */
    if (text_flag!=-2) {
      if (new_img==2) {
        real_min=30000;
        real_max=-30000;

        for (i=0;i<nb_spec;i++) {
          if (tmp_slice[i]>real_max) real_max=tmp_slice[i];
          if (tmp_slice[i]<real_min) real_min=tmp_slice[i];
        }

        n_colors=nc;
        cpgsch(0.9);
        cpgsvp(0.80,0.84,0.1,0.85);
        cpgswin(0.1, 0.5, min, (1.0)*max);
        for (i=0;i<255;i++) {
            scale[i]=min+((max-min)/255)*i;
        }
        tr_palett(255,scale,&tmp_scale,sign,contra,bright,min,max,&out_min,&out_max);
        for (i=0;i<254;i++) {
          n_colors=nc;
          color_index=16+floor((tmp_scale[i]-out_min)/(out_max-out_min)*(n_colors-1));
          if (color_index>255) color_index=255;
          if (color_index<17) color_index=17;
          cpgsci(color_index);
          cpgsfs(1);
          cpgrect(0.01,0.99,scale[i],scale[i+1]);
        }
        free(tmp_scale);
        cpgsci(1);
        cpgsch(1.2);
        cpgbox("BC", 0.0, 0.0, "BCMTS", 0.0, 0.0);
        cpgsch(0.9);

        cpgsvp(0.15,0.80,0.1,0.85);
        cpgsch(1.1);
      }
    }
    free(tmp_slice);
  }

  if (text_flag==5) {
    /* We plot all the spaxels */
    for (i=0;i<nb_spec;i++) {
      for (j=0;j<n_spaxels[i];j++) {
        x_c=spaxels[i][j].xpos;
        y_c=spaxels[i][j].ypos;
        grupo=groups[spaxels[i][j].group-1];
        cpgslw(1.0);
        plot_spaxel(1,x_c,y_c,grupo,2,0);
        cpgsch(0.6);
      }
    }
  }
  cpgsci(1);
  free(slice);
  if (save==1) {
    i_slices[slice_index][0]=i0;
    i_slices[slice_index][1]=i1;
    slice_index++;
  }
  if (slice_index>MAX_NUM_MAPS) slice_index=0;

  sprintf(interp->result,"%f %f %d %f %f %f %f %f Zero Level=%f, Delta_level=%f",min,max,slice_index,x_min,x_max,y_min,y_max,max_size,temp_float[0],temp_float[1]-temp_float[0]);

  last_i0=i0;
  last_i1=i1;

  return TCL_OK;
}

/*********************************
 * Subroutine to create an slice
 *********************************/
static int create_slice_command(Tcl_Interp *interp, int argc, char *argv[]) {
  int j,i,i1,i2,i_temp;
  float *slice;
  int text_flag=0;

  if (argc < 4) {
    Tcl_AppendResult(interp, "USE: create_slice i1 i2 slice_index text_flag", NULL);
    return TCL_ERROR;
  }

  i1=floor(atof(argv[0]));
  i2=floor(atof(argv[1]));
  slice_index=floor(atof(argv[2]));
  text_flag=floor(atof(argv[3]));

  if (slice_index>MAX_NUM_MAPS) slice_index=0;
  if (text_flag<90) {
    if (i2<i1) {
      i_temp=i2;
      i2=i1;
      i1=i_temp;
    }
    if (i1 < 0 || i2 < 0) {
      print_msg("create_slice_command: one index is less than 0!");
      return TCL_ERROR;
    }
    if (i2==i1) i2=i1+1;
    /*
     * We determine the range of spectras to ask for
     */
    slice = (float *)malloc(nb_spec*sizeof(float));
    for (i=0;i<nb_spec;i++) {
      slice[i]=0.0;
      f_slice_now[i]=0.0;
      f_slice[slice_index][i]=0.0;
      for (j=i1;j<i2;j++) {
        slice[i]+=raw_data[j+i*npix];
        f_slice_now[i]+=raw_data[j+i*npix];
        f_slice[slice_index][i]+=raw_data[j+i*npix];
      }
      slice[i]/=(i2-i1);
      f_slice_now[i]/=(i2-i1);
      f_slice[slice_index][i]/=(i2-i1);
      f_slice_def[slice_index]=1;
    }
  } else {
    if (vel_map_fill[i1]==0) {
      sprintf(interp->result,"Velocity Map %d not loaded",i1);
      return TCL_OK;
    }
    if (i2>2) i2=2;
    if (i2<0) i2=0;
    slice = (float *)malloc((nb_spec)*sizeof(float));
    for (i=0;i<nb_spec;i++) {
      if (i2==0) {
        if (fabs(vel_map[i1].e_f[i]/(vel_map[i1].f[i]+0.0001))<5.0) {
          slice[i]=vel_map[i1].f[i];
        } else {
          slice[i]=0;
        }
      }
      if (i2==1) {
        if (fabs(vel_map[i1].e_w[i]/(vel_map[i1].w[i]+0.0001))<20.0) {
          if (fabs(vel_map[i1].e_f[i]/(vel_map[i1].f[i]+0.0001))<5.0) {
            slice[i]=vel_map[i1].w[i];
          }
        } else {
          slice[i]=0;
        }
      }
      if (i2==2) {
        if (fabs(vel_map[i1].e_dw[i]/(vel_map[i1].dw[i]+0.0001))<20.0) {
          if (fabs(vel_map[i1].e_f[i]/(vel_map[i1].f[i]+0.0001))<5.0) {
            slice[i]=vel_map[i1].dw[i];
          }
        } else {
          slice[i]=0;
        }
      }
      f_slice_now[i]=slice[i];
      // XXX: compiler warns: statement with no effect?!
      //f_slice[slice_index][i]==slice[i];
      f_slice_def[slice_index]=1;
    }
  }

  free(slice);

  /*
   * We save the data for future
   */
  i_slices[slice_index][0]=i1;
  i_slices[slice_index][1]=i2;
  sprintf(interp->result,"%d",slice_index);

  return TCL_OK;
}

/*****************************************************
 * This subroutine plot a line
 * The basic paramters are:
 * dev_id colorindex lweight lstyle X_points Y_points
 *****************************************************/
static int plot_line_command(Tcl_Interp *interp, int argc, char *argv[]) {
  int nargc,nargc_tmp;
  float *x,*y;
  int j;
  int dev_id,colorindex,width,style;
  char **tokens;

  if (argc<6) {
    Tcl_SetResult(interp, "USE: plot_line dev_id colorindex lweight lstyle X_points Y_points", TCL_STATIC);
    return TCL_ERROR;
  }
  dev_id=atoi(argv[0]);
  colorindex=atoi(argv[1]);
  width=atoi(argv[2]);
  style=atoi(argv[3]);
  split_text(argv[4]," ",&nargc,&tokens);
  x = (float *)malloc((nargc)*sizeof(float));
  for (j=0;j<nargc;j++) {
    x[j]=atof(tokens[j]);
  }
  nargc_tmp=nargc;
  split_text(argv[5]," ",&nargc,&tokens);
  y = (float *)malloc((nargc)*sizeof(float));
  for (j=0;j<nargc;j++) {
    y[j]=atof(tokens[j]);
  }
  if (nargc>nargc_tmp) nargc=nargc_tmp;
  cpgslct(dev_id);
  cpgsci(colorindex);
  cpgslw(width);
  cpgsls(style);
  cpgline(nargc,x,y);
  cpgsls(1);
  cpgsci(1);
  cpgslw(1);
  free(x);
  free(y);
  return TCL_OK;
}

/*****************************************************
 * Function that draws a messsage in a certain window
 *****************************************************/
static int message_command(Tcl_Interp *interp, int argc, char *argv[]) {
  if (argc < 2) {
    Tcl_AppendResult(interp, "USE: message dev_id message", NULL);
    return TCL_ERROR;
  }
  display_message(atoi(argv[0]),argv[1]);
  return TCL_OK;
}

/********************************
 * Function that closes a device
 ********************************/
static int close_device_command(Tcl_Interp *interp, int argc, char *argv[]) {
  if (argc < 1) {
    Tcl_AppendResult(interp, "USE: close_device dev_id", NULL);
    return TCL_ERROR;
  }
  close_device(atoi(argv[0]));
  return TCL_OK;
}

/********************************
 * Function that cleans a device
 ********************************/
static int clear_device_command(Tcl_Interp *interp, int argc, char *argv[]) {
  if (argc < 1) {
    Tcl_AppendResult(interp, "USE: clear_device dev_id", NULL);
    return TCL_ERROR;
  }
  cpgslct(atoi(argv[0]));
  cpgask(0);
  cpgpage();
  return TCL_OK;
}

/***************************************
 * Function that stats the start_server
 ***************************************/
static int start_server_command(Tcl_Interp *interp, int argc, char *argv[]) {
  int alive=0;
  pid_t child;

  alive=ask_alive();
  if (alive==1) {
    print_msg("Starting the server...");
    server_started=1;
    if ((child=fork())==-1) {
    } else if (child == 0 ) {
      server();
    }
  } else {
    print_msg("The server was already running...");
  }
  return TCL_OK;
}

/******************
 * Stop the SERVER
 ******************/
static int stop_server_command(Tcl_Interp *interp, int argc, char *argv[]) {
  int flag[15];
  int tmp_address;
  int mem_key=3,n_bytes=15;

  ask_delete_shm();
  flag[0]=666;
  load_mem_int(flag,mem_key,n_bytes,&tmp_address);
  return TCL_OK;
}

/*******************
 * Clean the SERVER
 *******************/
static int clean_server_command(Tcl_Interp *interp, int argc, char *argv[]) {
  ask_delete_shm();
  return TCL_OK;
}

/*******************
 * Save a e3d file
 ******************/
static int save_file_command(Tcl_Interp *interp, int argc, char *argv[]) {
  int i,j;
  char text[80];
  E3D_file e3d_tmp;
  SPECTRUM signal;
  SPAXEL spax;

  if (argc < 1 ) {
    Tcl_AppendResult(interp, "USE: save_file filename", NULL);
    return TCL_ERROR;
  }
  if ((strlen(argv[0])==0)||(strcmp(argv[0],".fits")==0)||(strcmp(argv[0],".fit")==0)||(strcmp(argv[0],".e3d")==0)) {
    Tcl_AppendResult(interp, "No output file defined", NULL);
    return TCL_ERROR;
  }
  strcpy(text,argv[0]);
  print_msg("Saving %s...",text);
  /*
   * SAVE and Euro3D function.
   */
  create_E3D_frame(&e3d_tmp,text,npix,(double)start_w,(double)delta_w,FLOAT,"save",NULL);
  for (i=0;i<nb_spec;i++) {
    init_new_E3D_spec(&e3d_tmp,&signal,npix,start_w);
    for (j=0;j<signal.npts;j++) {
      WR_spec(&signal,j,raw_data[j+i*npix]);
    }
    put_E3D_spec(&e3d_tmp,&signal,NULL,i+1);
    for (j=0;j<n_spaxels[i];j++) {
      spax.specId=spaxels[i][j].specId;
      spax.group=spaxels[i][j].group;
      spax.xpos=spaxels[i][j].xpos;
      spax.ypos=spaxels[i][j].ypos;
      put_E3D_spaxels(&e3d_tmp,i+1,1,&spax);
    }
  }
  e3d_tmp.ngroups = n_groups;
  e3d_tmp.groups = groups;
  close_E3D_frame(&e3d_tmp);

  return TCL_OK;
}

/**************************************
 * Save a portion of an e3d file (OBJ)
 **************************************/
static int save_obj_file_command(Tcl_Interp *interp, int argc, char *argv[]) {
  int n_obj;
  int i,j;
  char text[80];
  E3D_file e3d_tmp;
  SPECTRUM signal;
  SPAXEL spax;

  if (argc < 2) {
    Tcl_AppendResult(interp, "USE: save_obj_file n_obj FILE", NULL);
    return TCL_ERROR;
  }
  if ((strlen(argv[1])==0)||(strcmp(argv[1],".fits")==0)||(strcmp(argv[1],".fit")==0)||(strcmp(argv[1],".e3d")==0)) {
    Tcl_AppendResult(interp, "No output file defined", NULL);
    return TCL_ERROR;
  }
  n_obj=atoi(argv[0]);
  if (n_obj<0) {
    Tcl_AppendResult(interp, "n_obj<0 : ERROR", NULL);
    return TCL_ERROR;
  }
  if (n_obj>MAX_NUM_OBJ) {
    Tcl_AppendResult(interp, "n_obj>MAX_NUM_OBJ : ERROR", NULL);
    return TCL_ERROR;
  }
  if (n_j_spax[n_obj]==0) {
    Tcl_AppendResult(interp, "No Spaxels in this object : ERROR", NULL);
    return TCL_ERROR;
  }
  n_i_spax=n_j_spax[n_obj];
  for (j=0;j<n_i_spax;j++) {
    i_spax[j]=j_spax[n_obj][j];
  }

  strcpy(text,argv[1]);
  print_msg("Saving object %d to file %s...",n_obj,text);
  /*
   * SAVE and Euro3D function.
   */
  create_E3D_frame(&e3d_tmp,text,npix,(double)start_w,(double)delta_w,FLOAT,"save",NULL);
  for (i=0;i<n_i_spax;i++) {
    init_new_E3D_spec(&e3d_tmp,&signal,npix,start_w);
    for (j=0;j<signal.npts;j++) {
      WR_spec(&signal,j,raw_data[j+i_spax[i]*npix]);
    }
    put_E3D_spec(&e3d_tmp,&signal,NULL,i+1);
    for (j=0;j<n_spaxels[i_spax[i]];j++) {
      spax.specId=spaxels[i_spax[i]][j].specId;
      spax.group=spaxels[i_spax[i]][j].group;
      spax.xpos=spaxels[i_spax[i]][j].xpos;
      spax.ypos=spaxels[i_spax[i]][j].ypos;
      put_E3D_spaxels(&e3d_tmp,i+1,1,&spax);
    }
  }
  e3d_tmp.ngroups = n_groups;
  e3d_tmp.groups = groups;
  close_E3D_frame(&e3d_tmp);
  return TCL_OK;
}

/************************************
 * Load a new e3d file in the server
 ************************************/
static int load_file_command(Tcl_Interp *interp, int argc, char *argv[]) {
  int i,j;
  float common_parameters[3];

  nb_spec=0;
  if (argc < 1) {
    Tcl_AppendResult(interp, "USE: load_file filename", NULL);
    return TCL_ERROR;
  }

  is_e3d=is_e3d_file(argv[0]);
  sprintf(interp->result,"%d",nb_spec*is_e3d);
  if (is_e3d!=1) {
    Tcl_AppendResult(interp, "Not an Euro3D file", NULL);
    return TCL_ERROR;
  }

  for (i=0;i<MAX_NUM_MAPS;i++) {
    map_created[i]=0;
  }

  if (server_started==1) {
    print_msg("Reading using SHM");
    if (strcmp(argv[0],"SHM")!=0) {
      ask_load_e3d(argv[0]); // In other case the file is already in the SHM
    }
    ask_e3d_info(&e3d_image,&n_spaxels,&spaxels,&index_start,&npts,&groups);
	 if (groups) {
      dpix1 = groups->size1;
	   dpix2 = groups->size2;
	 }
    start_w=e3d_image.common_parameters[0];
    end_w=e3d_image.common_parameters[1];
    npix=e3d_image.common_parameters[2];
    delta_w=(end_w-start_w)/npix;
    nb_spec=e3d_image.nbspec;
    n_groups=e3d_image.ngroups;
    if (i_spax) free(i_spax); i_spax = (int *)malloc((nb_spec+1)*sizeof(int));
    if (raw_data) free(raw_data); raw_data = (float *)malloc((nb_spec+1)*npix*sizeof(float));
    for (j=0;j<MAX_NUM_OBJ;j++) {
      j_spax[j] = (int *)malloc((nb_spec+1)*sizeof(int));
      n_j_spax[j]=0;
    }
    ask_raw_data(&raw_data,nb_spec,npix,delta_n);
  } else {
    print_msg("Reading %s", argv[0]);
    load_e3d_only_groups(argv[0],&n_groups,&groups,common_parameters,&nb_spec);
	 if (groups) {
      dpix1 = groups->size1;
	   dpix2 = groups->size2;
	 }
    start_w=common_parameters[0];
    end_w=common_parameters[1];
    npix=common_parameters[2];
    if (raw_data) free(raw_data); raw_data = (float *)calloc(nb_spec*npix,sizeof(float));
    if (raw_noise) free(raw_noise); raw_noise = (float *)calloc(nb_spec*npix, sizeof(float));
    if (n_spaxels) free(n_spaxels); n_spaxels = (int *)calloc(nb_spec, sizeof(int));
    if (npts) free(npts); npts = (int *)calloc(nb_spec, sizeof(int));
    if (index_start) free(index_start); index_start = (int *)calloc(nb_spec, sizeof(int));
    if (spaxels) free(spaxels); spaxels = (SPAXEL **)calloc(nb_spec, sizeof(SPAXEL *));

    load_e3d_only_data(argv[0],nb_spec,npix,n_spaxels,&spaxels, index_start, npts, raw_data, raw_noise);

    delta_w=(end_w-start_w)/npix;
    i_spax = (int *)malloc((nb_spec+1)*sizeof(int));
    for (j=0;j<MAX_NUM_OBJ;j++) {
      j_spax[j] = (int *) calloc((nb_spec+1),sizeof(int));
      n_j_spax[j]=0;
    }
  }

  datamin=3000000.0;
  datamax=-3000000.0;
  for (j=0;j<npix*nb_spec;j++) {
    if (datamin>raw_data[j])
      datamin=raw_data[j];
    if (datamax<raw_data[j])
      datamax=raw_data[j];
  }
  x_min=1000000;
  y_min=1000000;
  x_max=-1000000;
  y_max=-1000000;

  for (i=0;i<nb_spec;i++) {
    i_spax[i]=i;
    for (j=0;j<n_spaxels[i];j++) {
      if (x_min>spaxels[i][j].xpos) x_min=spaxels[i][j].xpos;
      if (x_max<spaxels[i][j].xpos) x_max=spaxels[i][j].xpos;
      if (y_min>spaxels[i][j].ypos) y_min=spaxels[i][j].ypos;
      if (y_max<spaxels[i][j].ypos) y_max=spaxels[i][j].ypos;
    }
  }

  /*
   * Cleaning the previous objects selection
   */
  for (j=0;j<MAX_NUM_OBJ;j++) {
    n_j_spax[j]=0;
  }

  /* DAR */
  if (loaded==1) {
    free(dars);
  }
  dar_corrected=0;
  dars = (struct DAR *) calloc(n_groups,sizeof(struct DAR));

  loaded=1;
  spax_painted=0;

  /*
   * Cleaning variables... a lot to include!!!! (16.05.03)
   */
  for (i=0;i<n_pintados;i++) {
    pintados[i]=0;
  }

  f_slice_now = (float *) calloc((nb_spec),sizeof(float));
  for (i=0;i<MAX_NUM_MAPS;i++) {
    f_slice[i] = (float *) calloc((nb_spec),sizeof(float));
  }

  npix_w1=0;
  npix_w2=npix;

  /*
   * Max size;
   */
  max_size=-1.0;
  for (i=0;i<n_groups;i++) {
    if ((groups[i].shape=='R')||(groups[i].shape=='S')) {
      if (max_size<0.5*groups[i].size1) max_size=0.5*groups[i].size1;
    } else {
      if (max_size<groups[i].size1) max_size=groups[i].size1;
    }
  }
  if (nb_spec<last_spec) last_spec=nb_spec;

  sprintf(interp->result,"%d",nb_spec*is_e3d);

  save_def_max();
  return TCL_OK;
}

/*************************************************************
 * Import a Row Stacked Spectra (RSS) fitsfile+position table
 *************************************************************/
static int import_rss_command(Tcl_Interp *interp, int argc, char *argv[]) {
  int i,j;
  int correcto=0;

  if (argc < 2) {
    Tcl_AppendResult(interp, "USE: import_rss rss_fitsfile positiontable", NULL);
    return TCL_ERROR;
  }

  print_msg("Importing file %s using position table %s", argv[0], argv[1]);
  for (i=0;i<MAX_NUM_MAPS;i++) {
    map_created[i]=0;
  }

  if (raw_data) free(raw_data);
  if (raw_noise) free(raw_noise);
  if (n_spaxels) free(n_spaxels);
  if (npts) free(npts);
  if (index_start) free(index_start);
  if (spaxels) free(spaxels);
#ifdef DEBUG_dpix
  printf("import_rss_command: dpix=%f/%f", dpix1,dpix2);
#endif
  correcto=load_rss(argv[0],argv[1],&nb_spec,&npix,&n_spaxels,&spaxels,
                    &index_start,&npts,&raw_data,&raw_noise,&n_groups,&groups,
                    &start_w,&delta_w,&dpix1,&dpix2);
#ifdef DEBUG_dpix
  printf(" dpix=%f/%f\n", dpix1,dpix2);
#endif
  if (correcto>0) {
    end_w=delta_w*npix+start_w;
    i_spax = (int *)malloc((nb_spec+1)*sizeof(int));
    for (j=0;j<MAX_NUM_OBJ;j++) {
      j_spax[j] = (int *) calloc((nb_spec+1),sizeof(int));
      n_j_spax[j]=0;
    }
    datamin=3000000.0;
    datamax=-3000000.0;
    for (j=0;j<npix*nb_spec;j++) {
    if (datamin>raw_data[j])
      datamin=raw_data[j];
    if (datamax<raw_data[j])
      datamax=raw_data[j];
    }
    x_min=1000000;
    y_min=1000000;
    x_max=-1000000;
    y_max=-1000000;
    for (i=0;i<nb_spec;i++) {
      i_spax[i]=i;
      for (j=0;j<n_spaxels[i];j++) {
      if (x_min>spaxels[i][j].xpos) x_min=spaxels[i][j].xpos;
      if (x_max<spaxels[i][j].xpos) x_max=spaxels[i][j].xpos;
      if (y_min>spaxels[i][j].ypos) y_min=spaxels[i][j].ypos;
      if (y_max<spaxels[i][j].ypos) y_max=spaxels[i][j].ypos;
      }
    }
    for (j=0;j<MAX_NUM_OBJ;j++) {
    n_j_spax[j]=0;
    }
    if (loaded==1) {
      free(dars);
    }
    dar_corrected=0;
    dars = (struct DAR *) calloc(n_groups,sizeof(struct DAR));
    loaded=1;
    spax_painted=0;
    for (i=0;i<n_pintados;i++) {
      pintados[i]=0;
    }
    f_slice_now = (float *) calloc((nb_spec),sizeof(float));
    for (i=0;i<MAX_NUM_MAPS;i++) {
      f_slice[i] = (float *) calloc((nb_spec),sizeof(float));
    }
    npix_w1=0;
    npix_w2=npix;
    max_size=-1.0;
    for (i=0;i<n_groups;i++) {
      if ((groups[i].shape=='R')||(groups[i].shape=='S')) {
        if (max_size<0.5*groups[i].size1) max_size=0.5*groups[i].size1;
      } else {
        if (max_size<groups[i].size1) max_size=groups[i].size1;
      }
    }
    if (nb_spec<last_spec) last_spec=nb_spec;
    save_def_max();
  }

  sprintf(interp->result,"%d",correcto);
  if (correcto==1) {
    return TCL_OK;
  } else {
    return TCL_ERROR;
  }
}

/*********************************
 * Import a Slit Spectra fitsfile
 *********************************/
static int import_spectra_command(Tcl_Interp *interp, int argc, char *argv[]) {
  int i,j;
  int correcto;

  if (argc < 1) {
    Tcl_AppendResult(interp, "USE: import_spectra spectra_fitsfile", NULL);
    return TCL_ERROR;
  }

  print_msg("Importing spectrum %s", argv[0]);

  for (i=0;i<MAX_NUM_MAPS;i++) {
    map_created[i]=0;
  }

  if (raw_data) free(raw_data);
  if (raw_noise) free(raw_noise);
  if (n_spaxels) free(n_spaxels);
  if (npts) free(npts);
  if (index_start) free(index_start);
  if (spaxels) free(spaxels);
  correcto=load_spectra(argv[0],&nb_spec,&npix,&n_spaxels,&spaxels,&index_start,&npts,&raw_data,&raw_noise,&n_groups,&groups,&start_w,&delta_w);
  end_w=delta_w*npix+start_w;
  i_spax = (int *)malloc((nb_spec+1)*sizeof(int));
  for (j=0;j<MAX_NUM_OBJ;j++) {
    j_spax[j] = (int *) calloc((nb_spec+1),sizeof(int));
    n_j_spax[j]=0;
  }
  datamin=3000000.0;
  datamax=-3000000.0;
  for (j=0;j<npix*nb_spec;j++) {
    if (datamin>raw_data[j])
      datamin=raw_data[j];
    if (datamax<raw_data[j])
      datamax=raw_data[j];
  }
  x_min=1000000;
  y_min=1000000;
  x_max=-1000000;
  y_max=-1000000;
  for (i=0;i<nb_spec;i++) {
    i_spax[i]=i;
    for (j=0;j<n_spaxels[i];j++) {
      if (x_min>spaxels[i][j].xpos) x_min=spaxels[i][j].xpos;
      if (x_max<spaxels[i][j].xpos) x_max=spaxels[i][j].xpos;
      if (y_min>spaxels[i][j].ypos) y_min=spaxels[i][j].ypos;
      if (y_max<spaxels[i][j].ypos) y_max=spaxels[i][j].ypos;
    }
  }
  for (j=0;j<MAX_NUM_OBJ;j++) {
    n_j_spax[j]=0;
  }
  if (loaded==1) {
    free(dars);
  }
  dar_corrected=0;
  dars = (struct DAR *) calloc(n_groups,sizeof(struct DAR));
  loaded=1;
  spax_painted=0;
  for (i=0;i<n_pintados;i++) {
    pintados[i]=0;
  }
  f_slice_now = (float *) calloc((nb_spec),sizeof(float));
  for (i=0;i<MAX_NUM_MAPS;i++) {
    f_slice[i] = (float *) calloc((nb_spec),sizeof(float));
  }
  npix_w1=0;
  npix_w2=npix;
  max_size=-1.0;
  for (i=0;i<n_groups;i++) {
    if ((groups[i].shape=='R')||(groups[i].shape=='S')) {
      if (max_size<0.5*groups[i].size1) max_size=0.5*groups[i].size1;
    } else {
      if (max_size<groups[i].size1) max_size=groups[i].size1;
    }
  }
  if (nb_spec<last_spec) last_spec=nb_spec;
  save_def_max();
  if (correcto==0) {
    return TCL_OK;
  } else {
    return TCL_ERROR;
  }
}

/****************
 * Import a Cube
 ****************/
static int import_cube_command(Tcl_Interp *interp, int argc, char *argv[]) {
  int i,j;
  int correcto;

  if (argc < 1) {
    Tcl_AppendResult(interp, "USE: import_cube cube_fitsfile", NULL);
    return TCL_ERROR;
  }

  print_msg("Importing Cube %s", argv[0]);
  for (i=0;i<MAX_NUM_MAPS;i++) map_created[i]=0;
  if (raw_data) free(raw_data);
  if (raw_noise) free(raw_noise);
  if (n_spaxels) free(n_spaxels);
  if (npts) free(npts);
  if (index_start) free(index_start);
  if (spaxels) free(spaxels);
#ifdef DEBUG_dpix
  printf("import_cube_command: dpix=%f/%f", dpix1,dpix2);
#endif
  correcto=load_cube(argv[0],&nb_spec,&npix,&n_spaxels,&spaxels,
                     &index_start,&npts,&raw_data,&raw_noise,&n_groups,
                     &groups,&start_w,&delta_w,&dpix1,&dpix2);
  if (correcto != 0) {
    return TCL_ERROR;
  }
#ifdef DEBUG_dpix
  printf(" dpix=%f/%f\n", dpix1,dpix2);
#endif
  end_w=delta_w*npix+start_w;
  i_spax = (int *)malloc((nb_spec+1)*sizeof(int));
  for (j=0;j<MAX_NUM_OBJ;j++) {
    j_spax[j] = (int *) calloc((nb_spec+1),sizeof(int));
    n_j_spax[j]=0;
  }
  datamin=3000000.0;
  datamax=-3000000.0;
  for (j=0;j<npix*nb_spec;j++) {
    if (datamin>raw_data[j])
      datamin=raw_data[j];
    if (datamax<raw_data[j])
      datamax=raw_data[j];
  }
  x_min=1000000;
  y_min=1000000;
  x_max=-1000000;
  y_max=-1000000;
#ifdef DEBUG_e3d_load
  printf("1st middle of import_cube_command %f %f %f %f %f\n",
         dpix, x_max, x_min, y_max, y_min);
#endif
  for (i=0;i<nb_spec;i++) {
    i_spax[i]=i;
    for (j=0;j<n_spaxels[i];j++) {
      if (x_min>spaxels[i][j].xpos) x_min=spaxels[i][j].xpos;
      if (x_max<spaxels[i][j].xpos) x_max=spaxels[i][j].xpos;
      if (y_min>spaxels[i][j].ypos) y_min=spaxels[i][j].ypos;
      if (y_max<spaxels[i][j].ypos) y_max=spaxels[i][j].ypos;
    }
  }
#ifdef DEBUG_e3d_load
  printf("2nd middle of import_cube_command %f %f %f %f %f\n",
         dpix, x_max, x_min, y_max, y_min);
#endif

  for (j=0;j<MAX_NUM_OBJ;j++) {
    n_j_spax[j]=0;
  }
  if (loaded==1) {
    free(dars);
  }
  dar_corrected=0;
  dars = (struct DAR *) calloc(n_groups,sizeof(struct DAR));
  loaded=1;
  spax_painted=0;
  for (i=0;i<n_pintados;i++) {
    pintados[i]=0;
  }
  f_slice_now = (float *) calloc((nb_spec),sizeof(float));
  for (i=0;i<MAX_NUM_MAPS;i++) {
    f_slice[i] = (float *) calloc((nb_spec),sizeof(float));
  }
  npix_w1=0;
  npix_w2=npix;
  max_size=-1.0;
  for (i=0;i<n_groups;i++) {
    if ((groups[i].shape=='R')||(groups[i].shape=='S')) {
      if (max_size<0.5*groups[i].size1) max_size=0.5*groups[i].size1;
    } else {
      if (max_size<groups[i].size1) max_size=groups[i].size1;
    }
  }
  if (nb_spec<last_spec) last_spec=nb_spec;
  save_def_max(); /* XXX What is this supposed to do?! */

#ifdef DEBUG_e3d
  printf("end of import_cube_command %f %f %f %f %f\n",
         dpix, x_max, x_min, y_max, y_min);
#endif

  if (correcto==0) {
    return TCL_OK;
  } else {
    return TCL_ERROR;
  }
}

/*******************************************************
 * Import a MAP (2D Fits file, no wavelength range)
 *******************************************************/
static int import_map_command(Tcl_Interp *interp, int argc, char *argv[]) {
  int i,j;
  int correcto;

  if (argc < 1) {
    Tcl_AppendResult(interp, "USE: import_map cube_fitsfile", NULL);
    return TCL_ERROR;
  }

  print_msg("Importing map %s", argv[0]);

  for (i=0;i<MAX_NUM_MAPS;i++) {
    map_created[i]=0;
  }

  if (raw_data) free(raw_data);
  if (raw_noise) free(raw_noise);
  if (n_spaxels) free(n_spaxels);
  if (npts) free(npts);
  if (index_start) free(index_start);
  if (spaxels) free(spaxels);

  correcto=load_map(argv[0],&nb_spec,&npix,&n_spaxels,&spaxels,&index_start,&npts,&raw_data,&raw_noise,&n_groups,&groups,&start_w,&delta_w);
  end_w=delta_w*npix+start_w;
  i_spax = (int *)malloc((nb_spec+1)*sizeof(int));
  for (j=0;j<MAX_NUM_OBJ;j++) {
    j_spax[j] = (int *) calloc((nb_spec+1),sizeof(int));
    n_j_spax[j]=0;
  }
  datamin=3000000.0;
  datamax=-3000000.0;
  for (j=0;j<npix*nb_spec;j++) {
    if (datamin>raw_data[j])
      datamin=raw_data[j];
    if (datamax<raw_data[j])
      datamax=raw_data[j];
  }
  x_min=1000000;
  y_min=1000000;
  x_max=-1000000;
  y_max=-1000000;
  for (i=0;i<nb_spec;i++) {
    i_spax[i]=i;
    for (j=0;j<n_spaxels[i];j++) {
      if (x_min>spaxels[i][j].xpos) x_min=spaxels[i][j].xpos;
      if (x_max<spaxels[i][j].xpos) x_max=spaxels[i][j].xpos;
      if (y_min>spaxels[i][j].ypos) y_min=spaxels[i][j].ypos;
      if (y_max<spaxels[i][j].ypos) y_max=spaxels[i][j].ypos;
    }
  }

  for (j=0;j<MAX_NUM_OBJ;j++) {
    n_j_spax[j]=0;
  }
  if (loaded==1) {
    free(dars);
  }
  dar_corrected=0;
  dars = (struct DAR *) calloc(n_groups,sizeof(struct DAR));
  loaded=1;
  spax_painted=0;
  for (i=0;i<n_pintados;i++) {
    pintados[i]=0;
  }
  f_slice_now = (float *) calloc((nb_spec),sizeof(float));
  for (i=0;i<MAX_NUM_MAPS;i++) {
    f_slice[i] = (float *) calloc((nb_spec),sizeof(float));
  }
  npix_w1=0;
  npix_w2=npix;
  max_size=-1.0;
  for (i=0;i<n_groups;i++) {
    if ((groups[i].shape=='R')||(groups[i].shape=='S')) {
      if (max_size<0.5*groups[i].size1) max_size=0.5*groups[i].size1;
    } else {
      if (max_size<groups[i].size1) max_size=groups[i].size1;
    }
    /* if (max_size<groups[i].size1) max_size=groups[i].size1; */
    /* if (max_size<groups[i].size2) max_size=groups[i].size2; */
  }
  if (nb_spec<last_spec) last_spec=nb_spec;
  save_def_max();

  if (correcto==0) {
    return TCL_OK;
  } else {
    return TCL_ERROR;
  }
}

/*******************************************************
 * Draw Stacked Spectra Image
 * The parameters to pass are:
 *******************************************************/
static int draw_raw_command(Tcl_Interp *interp, int argc, char *argv[]) {
  char *p_names[] = {"grey","rainbow","heat","iraf","aips"};
  float min, max;
  float contra, bright;
  float sign;
  int i,j, jmax, id,p;
  int clean=0;
  char wavestring[8];
  float lw, xch,ych;
  float *cut_data;
  float tr_raw[]={0.0,1.0,0.0,0.0,0.0,1.0};
  float out_min,out_max;

  if (argc < 3 || *argv[1] == '-') {
    Tcl_AppendResult(interp, "USE: draw_raw min max device clean palette bright contrast sign first_spec last_spec", NULL);
    return TCL_ERROR;
  }

  min=atof(argv[0]);
  max=atof(argv[1]);
  id=atoi(argv[2]);
  clean=atoi(argv[3]);
  if (argc>4) {
    bright=atof(argv[5]);
    contra=atof(argv[6]);
    sign=atof(argv[7]);
    for (j=0;j<5;j++) {
      if (strstr(p_names[j],argv[4])) p=j+1;
    }
  } else {
    bright=0.5;
    contra=0.7;
    p=1;
  }

  if (argc>8) {
    first_spec=atoi(argv[8]);
    last_spec=atoi(argv[9]);
    if (nb_spec<last_spec) last_spec=nb_spec;
    if (nb_spec<first_spec) first_spec=nb_spec;
    if (last_spec<0) last_spec=0;
    if (first_spec<0) first_spec=0;
  }

  cut_data = (float *)malloc((npix*(last_spec-first_spec))*sizeof(float));
  for (j=first_spec;j<last_spec;j++) {
    for (i=0;i<npix;i++) {
      cut_data[i+(j-first_spec)*npix]=raw_data[i+j*npix];
    }
  }

  if (min==0.0) min=datamin;
  if (max==0.0) max=datamax;
  cpgslct(id);
  cpgask(0);

  if (old_p[id] != p) pintados[id]=0;
  if (clean==1) {
    cpgpage();
    pintados[id]=0;
  }

  /*
   * tr_raw[0]=(float)(start_w+npix_w1*delta_w);
   * tr_raw[1]=(float)delta_w;
   * tr_raw[2]=0;
   * tr_raw[3]=1.0;
   * tr_raw[4]=0;
   * tr_raw[5]=1.0;
   */
  if (pintados[id]<2) {
    cpgsch(1.2);
    cpgsvp(-0.5, 1.3, -0.3, 1.3); // XXX PmW is this required at all?
    cpgenv(npix_w1, npix_w2+1, -10, last_spec-first_spec, 1, -2);
    /* max. number of displayed wavelengths based on pixel sizes of characters */
    cpgqcs(3, &xch, &ych);                /* get char sizes in pixels */
    jmax = (npix_w2-npix_w1) / (xch+ych); /* the maximum that would comfortably fit... */
    if (jmax > 12) jmax = 12;             /* but not more than 12 */
    if (jmax <  1) jmax =  1;             /* and not less than  1 */
    /* now use char height in world coordinates to determine y-position of label */
    cpgqcs(4, &xch, &ych);
    for (j=0;j<=jmax;j++) {
      lw = start_w + (npix_w1)*delta_w + ((npix_w2-npix_w1)*delta_w)*j/jmax;
      if (lw > 1000) {
        /* optical wavelength range, units most likely Angstroems, use int */
        sprintf(wavestring, "%d", (int)lw);
      } else {
        /* small values: probably NIR in Micron, use float */
        sprintf(wavestring, "%5.3f", lw);
      }
      /* draw the ticks by hand ... */
      cpgmove((npix_w2-npix_w1)*((float)j/(float)jmax)+npix_w1, 0.0);
      cpgdraw((npix_w2-npix_w1)*((float)j/(float)jmax)+npix_w1, -0.5*ych);
      /* ... then the tick labels */
      cpgptxt((npix_w2-npix_w1)*((float)j/(float)jmax)+npix_w1, -1.5*ych, 30.0, 0.8, wavestring);
    }
    old_p[id]=p;
  }
  cpgsch(1.4);
  tr_palett(npix*(last_spec-first_spec),cut_data,&tmp_data,sign,contra,bright,min,max,&out_min,&out_max);
  cpgimag(tmp_data,npix,(last_spec-first_spec),1+npix_w1,npix_w2,1,(last_spec-first_spec),out_min,out_max,tr_raw);
  free(tmp_data);
  free(cut_data);

  pintados[id]++;
  if (pintados[id]>3) pintados[id]=3;

  sprintf(interp->result,"%d %d %f %f %f %f",npix,nb_spec,min,max,datamin,datamax);
  return TCL_OK;
}

/**************************************************************
 * To Change the Palette
 **************************************************************/
static int recolor_image_command(Tcl_Interp *interp, int argc, char *argv[]) {
  int p;
  char *p_names[]={"grey","rainbow","heat","iraf","aips"};
  int j;
  int dev_id;
  float bright,contra,sign;
  char commands[80];

  if (argc < 2) {
    Tcl_SetResult(interp, "USE: recolor_image palette dev_id", TCL_STATIC);
    return TCL_ERROR;
  };

  dev_id=atoi(argv[1]);
  bright=0.65;
  contra=0.65;
  sign=1;

  for (j=0;j<5;j++) {
    if (strstr(p_names[j],argv[0])) p=j+1;
  }
  sprintf(commands,"%d %f %f %f %d",p,bright,contra,sign,dev_id);
  cpgslct(dev_id);
  palette(p,sign*0.8,0.6);
  return TCL_OK;
}

/**************************************************************
 * Ask for the Intensity in a certain point
 **************************************************************/
static int intensity_command(Tcl_Interp *interp, int argc, char *argv[]) {
  int x, y;
  float intensity;

  if (argc < 2) {
    Tcl_SetResult(interp, "USE: intensity x y", TCL_STATIC);
    return TCL_ERROR;
  }

  x=floor(atof(argv[0]));
  y=floor(atof(argv[1]));

  if ((x>=0)&&(x<npix)&&(y>=0)&&(y<nb_spec)) {
    intensity=raw_data[x+y*npix];
  } else {
    intensity=0.0;
  }
  sprintf(interp->result,"%f",intensity);
  return TCL_OK;
}

/*************************************************
 * Asking for Basic E3D Parameters
 *************************************************/
static int ask_e3d_info_command(Tcl_Interp *interp, int argc, char *argv[]) {
  float mean,sigma,median;

  if (loaded==1) {
    mean=fmean_clean(raw_data,nb_spec*npix);
    median=mean;
    sigma=sqrt(fabs(mean));
    /*
     * if (mean==NaN) mean=0.0;
     * if (mean!=(1*mean)) mean=0.0;
     * median=fmean(raw_data,nb_spec*npix);
     * if (median==NaN) median=0.0;
     * if (median!=(1*median)) median=0.0;
     * sigma=fsigma(raw_data,nb_spec*npix,mean);
     * if (sigma==NaN) sigma=1.0;
     * if (sigma!=(1*sigma)) sigma=sqrt(fabs(median));
     * if (!sigma) sigma=1.0;
     * if (sigma>median) sigma=1.0;
     */
    sprintf(interp->result,"%d %d %f %f %f %d %f %f %d %d %f %f %f",npix,nb_spec,start_w,end_w,delta_w,n_groups,median-3*sigma,median+10*sigma,first_spec,last_spec,datamin,datamax,max_size);
  }
  return TCL_OK;
}

/*************************************************
 * Small function to determine the maxmin
 * for the Bright and Constrast determination
 ************************************************/
static int bc_command(Tcl_Interp *interp, int argc, char *argv[]) {
  float x,y;

  if (argc < 1) {
    Tcl_SetResult(interp, "USE: bc x", TCL_STATIC);
    return TCL_ERROR;
  };

  x=atof(argv[0]);
  y=Maximum(0,Minimum(1,x));

  sprintf(interp->result,"%f",y);
  return TCL_OK;
}

/**********************************************************
 * PGPLOT ROUTINES 04.01.04
 **********************************************************/
static int pgenv(ClientData data, Tcl_Interp *interp, int argc, char *argv[])
{
  float x_min,x_max,y_min,y_max;
  int flag1,flag2;

  if (argc < 7) {
    Tcl_SetResult(interp, "USE: pgenv x_min x_max y_min y_max flag1 flag2", TCL_STATIC);
    return TCL_ERROR;
  }

  x_min=atof(argv[1]);
  x_max=atof(argv[2]);
  y_min=atof(argv[3]);
  y_max=atof(argv[4]);
  flag1=atoi(argv[5]);
  flag2=atoi(argv[6]);
  cpgenv(x_min,x_max,y_min,y_max,flag1,flag2);
  return TCL_OK;
}

/**************************
 * Auxiliary functions
 **************************/
int save_def_max() {
  /*
   * We define the scales
   */
  if (aspect==1) {
    if (fabs(x_max-x_min)!=fabs(y_max-y_min)) {
      if (fabs(x_max-x_min)>fabs(y_max-y_min)) {
        y_max=(y_max+y_min)/2+fabs(x_max-x_min)/2;//+max_size/2;
        y_min=(y_max+y_min)/2-fabs(x_max-x_min)/2;//-max_size/2;
      } else {
        x_max=(x_max+x_min)/2+fabs(y_max-y_min)/2;//+max_size;
        x_min=(x_max+x_min)/2-fabs(y_max-y_min)/2;//-max_size;
      }
    }
  }
  x_max_def=x_max;
  y_max_def=y_max;
  x_min_def=x_min;
  y_min_def=y_min;
  max_size_def=max_size;

  dars[j].n=npix;
  dars[j].dx= (float *)malloc(npix*sizeof(float));
  dars[j].dy= (float *)malloc(npix*sizeof(float));

  dars_valid=1;
  for (j=0;j<n_groups;j++) {
    if ((groups[j].pressure!=groups[j].pressure)||(groups[j].pressure<0)||(groups[j].pressure>10000)||(groups[j].temperature!=groups[j].temperature)||(groups[j].temperature<0)||(groups[j].temperature>1000)||(groups[j].rel_humidity!=groups[j].rel_humidity)||(groups[j].rel_humidity<0)||(groups[j].rel_humidity>100)) {
      dars_valid=0;
    }
  }

  return 1;
}

int restore_def_max() {
  x_max=x_max_def;
  y_max=y_max_def;
  x_min=x_min_def;
  y_min=y_min_def;
  max_size=max_size_def;
  return 1;
}

int func_create_map(int i0, int i1, float dpix1, float dpix2,
                    int square, int n_m)
{
  int i;
  float *slice;
  float xy_max,xy_min;

  if ((n_m>MAX_NUM_MAPS)||(n_m<0)) {
    return -1;
  }

  slice = (float *)malloc(nb_spec*sizeof(float));
  for (i=0;i<nb_spec;i++) {
    slice[i]=0.0;
    for (j=i0;j<i1;j++) {
      slice[i]+=raw_data[j+i*npix];
    }
    slice[i]/=(i1-i0);
  }

  if (square==1) {
    if (x_min<y_min) {
        xy_min=x_min;
    } else {
        xy_min=y_min;
    }
    if (x_max>y_max) {
        xy_max=x_max;
    } else {
        xy_max=y_max;
    }
#ifdef DEBUG_dpix
    printf("func_create_map: dpix=%f/%f", dpix1,dpix2);
#endif
    create_map(nb_spec,n_spaxels,spaxels,npts,slice,dpix1,dpix2,
               &map_data[n_m],xy_min,xy_max,xy_min,xy_max,tr_map[n_m],
               &nx_map[n_m],&ny_map[n_m],grid_func,grid_opt);
#ifdef DEBUG_dpix
    printf(" dpix=%f/%f\n", dpix1,dpix2);
#endif
  } else {
#ifdef DEBUG_dpix
    printf("func_create_map: dpix=%f/%f", dpix1,dpix2);
#endif
    create_map(nb_spec,n_spaxels,spaxels,npts,slice,dpix1,dpix2,
               &map_data[n_m],x_min,x_max,y_min,y_max,tr_map[n_m],
               &nx_map[n_m],&ny_map[n_m],grid_func,grid_opt);
#ifdef DEBUG_dpix
    printf(" dpix=%f/%f\n", dpix1,dpix2);
#endif
  }
  map_created[n_m]=1;

  free(slice);
  return n_m;
}
