/******************************************************************************
 *
 * File:           nnbathy.c
 *
 * Created:        04/08/2000
 *
 * Author:         Pavel Sakov
 *                 CSIRO Marine Research
 *
 * Purpose:        Interpolate scalar 2D data in specified points using
 *                 Natural Neighbours interpolation.
 *
 * Description:    See usage().
 *
 *  Revisions:      None
 *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <limits.h>
#include <float.h>
#include <math.h>
#include <errno.h>
#include "nan.h"
#include "minell.h"
#include "nn.h"

#define NMAX 2048
#define STRBUFSIZE 64

int generate_points = 0;
int thin = 0;
int linear = 0;
int invariant = 0;
int square = 0;

static void version()
{
  //    printf("nnbathy/nn version %s\n", nn_version);
    exit(0);
}

static void usage()
{
    printf("Usage: nnbathy -i <XYZ file> {-o <XY file>|-n <nx>x<ny> [-c|-s] [-z <zoom>]}\n");
    printf("       [-v|-T <vertex id>|-V] [-D [<nx>x<ny>]] [-W <min weight>]\n");
    printf("Options:\n");
    printf("  -c              -- scale internally so that the enclosing minimal ellipse\n");
    printf("                     turns into a circle (this produces results invariant to\n");
    printf("                     affine transformations)\n");
    printf("  -i <XYZ file>   -- three-column file with points to interpolate from\n");
    printf("                     (use \"-i stdin\" or \"-i -\" for standard input)\n");
    printf("  -n <nx>x<ny>    -- generate <nx>x<ny> output rectangular grid\n");
    printf("  -o <XY file>    -- two-column file with points to interpolate in\n");
    printf("                     (use \"-o stdin\" or \"-o -\" for standard input)\n");
    printf("  -s              -- scale internally so that Xmax - Xmin = Ymax - Ymin\n");
    printf("  -v              -- verbose / version\n");
    printf("  -z <zoom>       -- zoom in (if <zoom> < 1) or out (<zoom> > 1)\n");
    printf("  -D [<nx>x<ny>]  -- average input X,Y,Z values in every cell of the\n");
    printf("                     rectangular grid <nx>x<ny> before processing (size\n");
    printf("                     optional with -n)\n");
    printf("  -P alg=<value>  -- use this algorithm. Possible values:\n");
    printf("                     l -- linear interpolation\n");
    printf("                     nn -- Natural Neighbours Sibson interpolation (default)\n");
    printf("                     ns -- Natural Neighbours Non-Sibsonian interpolation\n");
    printf("  -T <vertex id>  -- verbose; in weights output print weights associated with\n");
    printf("                     this vertex only\n");
    printf("  -V              -- very verbose / version\n");
    printf("  -W <min weight> -- minimal allowed weight for a vertex (normally -1 or so;\n");
    printf("                     lower values correspond to lower reliability; matters for\n");
    printf("                     points outside convex hall only)\n");
    printf("Description:\n");
    printf("  `nnbathy' interpolates scalar 2D data in specified points using Natural\n");
    printf("  Neighbours interpolation scheme. The interpolated values are written to\n");
    printf("  standard output.\n");

    exit(0);
}

static void quit(char* format, ...)
{
    va_list args;

    fflush(stdout);             /* just in case, to have exit message last */

    fprintf(stderr, "error: ");
    va_start(args, format);
    vfprintf(stderr, format, args);
    va_end(args);

    exit(1);
}

static void parse_commandline(int argc, char* argv[], char** fdata, char** fpoints, int* nx, int* ny, int* nxd, int* nyd, double* wmin, double* zoom)
{
    int i;

    if (argc < 2)
        usage();

    i = 1;
    while (i < argc) {
        if (argv[i][0] != '-')
            usage();

        switch (argv[i][1]) {
        case 'c':
            i++;
            square = 0;
            invariant = 1;
            break;
        case 'i':
            i++;
            if (i >= argc)
                quit("no file name found after -i\n");
            *fdata = argv[i];
            i++;
            break;
        case 'l':
            i++;
            linear = 1;
            break;
        case 'n':
            i++;
            *fpoints = NULL;
            generate_points = 1;
            if (i >= argc)
                quit("no grid dimensions found after -n\n");
            if (sscanf(argv[i], "%dx%d", nx, ny) != 2)
                quit("could not read grid dimensions after \"-n\"\n");
            if (*nx <= 0 || *nx > NMAX || *ny <= 0 || *ny > NMAX)
                quit("invalid size for output grid\n");
            i++;
            break;
        case 'o':
            i++;
            if (i >= argc)
                quit("no file name found after -o\n");
            *fpoints = argv[i];
            i++;
            break;
        case 's':
            i++;
            square = 1;
            invariant = 0;
            break;
        case 'v':
            i++;
            nn_verbose = 1;
            break;
        case 'z':
            i++;
            if (i >= argc)
                quit("no zoom value found after -z\n");
            *zoom = atof(argv[i]);
            i++;
            break;
        case 'D':
            i++;
            thin = 1;
            if (argc > i && argv[i][0] != '-') {
                if (sscanf(argv[i], "%dx%d", nxd, nyd) != 2)
                    quit("could not read grid dimensions after \"-D\"\n");
                if (*nxd <= 0 || *nyd <= 0)
                    quit("invalid value for nx = %d or ny = %d after -D option\n", *nxd, *nyd);
                if (*nxd > NMAX || *nyd > NMAX)
                    quit("too big value after -D option (expected < %d)\n", NMAX);
                i++;
            }
            break;
        case 'P':{
                char delim[] = "=";
                char prmstr[STRBUFSIZE] = "";
                char* token;

                i++;
                if (i >= argc)
                    quit("no input found after -P\n");

                if (strlen(argv[i]) >= STRBUFSIZE)
                    quit("could not interpret \"%s\" after -P option\n", argv[i]);

                strcpy(prmstr, argv[i]);
                token = strtok(prmstr, delim);
                if (token == NULL)
                    quit("could not interpret \"%s\" after -P option\n", argv[i]);

                if (strcmp(token, "alg") == 0) {
                    token = strtok(NULL, delim);
                    if (token == NULL)
                        quit("could not interpret \"%s\" after -P option\n", argv[i]);

                    if (strcmp(token, "nn") == 0) {
                        nn_rule = SIBSON;
                        linear = 0;
                    } else if (strcmp(token, "ns") == 0) {
                        nn_rule = NON_SIBSONIAN;
                        linear = 0;
                    } else if (strcmp(token, "l") == 0)
                        linear = 1;
                    else
                        usage();
                }

                i++;
                break;
            }
        case 'W':
            i++;
            if (i >= argc)
                quit("no minimal allowed weight found after -W\n");
            *wmin = atof(argv[i]);
            i++;
            break;
        case 'T':
            i++;
            if (i >= argc)
                quit("no vertex id found after -T\n");
            nn_test_vertice = atof(argv[i]);
            nn_verbose = 1;
            i++;
            break;
        case 'V':
            i++;
            nn_verbose = 2;
            break;
        default:
            usage();
            break;
        }
    }

    if (nn_verbose && argc == 2)
        version();

    if (thin) {
        if (*nxd == -1)
            *nxd = *nx;
        if (*nyd == -1)
            *nyd = *ny;
        if (*nxd <= 0 || *nyd <= 0)
            quit("invalid decimation grid size\n");
    }
}

static void points_write(int n, point* points)
{
    int i;

    for (i = 0; i < n; ++i) {
        point* p = &points[i];

        printf("%.15g %.15g %.15g\n", p->x, p->y, p->z);
    }
}

int main(int argc, char* argv[])
{
    char* fdata = NULL;
    char* fpoints = NULL;
    int nin = 0;
    point* pin = NULL;
    minell* me = NULL;
    int nout = 0;
    point* pout = NULL;
    int nx = -1;
    int ny = -1;
    int nxd = -1;
    int nyd = -1;
    double wmin = -DBL_MAX;
    double zoom = 1.0;
    double k = NaN;

    parse_commandline(argc, argv, &fdata, &fpoints, &nx, &ny, &nxd, &nyd, &wmin, &zoom);

    if (fdata == NULL)
        quit("no input data\n");

    if (!generate_points && fpoints == NULL)
        quit("no output grid specified\n");

    points_read(fdata, 3, &nin, &pin);

    if (nin < 3)
        return 0;

    if (thin)
        points_thin(&nin, &pin, nxd, nyd);

    if (generate_points)
        points_generate1(nin, pin, nx, ny, zoom, &nout, &pout);
    else
        points_read(fpoints, 2, &nout, &pout);

    if (invariant) {
        me = minell_build(nin, pin);
        minell_scalepoints(me, nin, pin);
        minell_scalepoints(me, nout, pout);
    } else if (square) {
        k = points_scaletosquare(nin, pin);
        points_scale(nout, pout, k);
    }

    if (linear)
        lpi_interpolate_points(nin, pin, nout, pout);
    else
        nnpi_interpolate_points(nin, pin, wmin, nout, pout);

    if (invariant)
        minell_rescalepoints(me, nout, pout);
    else if (square)
        points_scale(nout, pout, 1.0 / k);

    points_write(nout, pout);

    if (me != NULL)
        minell_destroy(me);
    free(pin);
    free(pout);

    return 0;
}
